/*
 * JHPdf Free PDF Library : HPdfObject.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.pdfobject;

import java.util.EnumSet;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sf.jhpdf.HPdfException;
import net.sf.jhpdf.encrypt.HPdfEncrypt;
import net.sf.jhpdf.io.HPdfWriteStream;

/**
 * Class represents PDF Object.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public abstract class HPdfObject {

    private static final Logger logger = LoggerFactory.getLogger(HPdfObject.class);
    
    public enum HPdfObjectType {
        //NONE(0L),
        DIRECT(0x80000000L),
        INDIRECT(0x40000000L),
        HIDDEN(0x10000000L),
        ;
        
        private long code;
        HPdfObjectType(long code) {
            this.code = code;
        }
        public long getCode() {
            return this.code;
        }
    };
    
    public HPdfObject() {
        setObjId(0);
        setGenNo(0);
    }
    
    private EnumSet<HPdfObjectType> objType = EnumSet.noneOf(HPdfObjectType.class);
    
    public final EnumSet<HPdfObjectType> getObjType() {
        return this.objType;
    }
    
    public final void setObjType(HPdfObjectType t) {
        this.objType.add(t);
    }
    
    public final void resetObjType(HPdfObjectType t) {
        this.objType.remove(t);
    }
    
    public final boolean isTypeDirect() {
        return getObjType().contains(HPdfObjectType.DIRECT);
    }
    
    public final boolean isTypeIndirect() {
        return getObjType().contains(HPdfObjectType.INDIRECT);
    }
    
    public final boolean isTypeAny() {
        return isTypeDirect() || isTypeIndirect();
    }
    
    public final boolean isTypeHidden() {
        return getObjType().contains(HPdfObjectType.HIDDEN);
    }
    
    private int objId;
    
    public final int getObjId() {
        return this.objId;
    }
    
    public final void setObjId(int id) {
        this.objId = id;
    }
    
    public final long getFullObjId() {
        long id = this.objId;
        for (final HPdfObjectType t : getObjType()) {
            id += t.getCode();
        }
        return id;
    }
    
    private int genNo;
    
    public final int getGenNo() {
        return this.genNo;
    }
    
    public final void setGenNo(int no) {
        this.genNo = no;
    }
    
    public final void write(HPdfWriteStream stream, HPdfEncrypt e) {
        logger.trace("HPdfObject#write");
        
        if (this.isTypeHidden()) {
            if (logger.isTraceEnabled()) {
                logger.trace(String.format("HPdfDict#write obj=%X skipped",
                        this.hashCode()));
            }
            return;
        }
        
        writeValue(stream, e);
    }
    
    /**
     * writes object own value to stream.<br>
     * subclasses must override this method for their own output behavior.
     * @param stream stream to output value.
     * @param e encrypt worker if encryption needed. no encryption needed, probably null. 
     */
    protected abstract void writeValue(HPdfWriteStream stream, HPdfEncrypt e);
}
