/*
 * JHPdf Free PDF Library : HPdfAnnotation.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.pdfobject;

import net.sf.jhpdf.HPdfErrorCode;
import net.sf.jhpdf.HPdfException;
import net.sf.jhpdf.graphics.HPdfRect;
import net.sf.jhpdf.graphics.HPdfRGBColor;
import net.sf.jhpdf.graphics.HPdfCMYKColor;
import net.sf.jhpdf.pdfobject.annotation.HPdf3DAnnotation;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Class represents PDF Annotation.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public abstract class HPdfAnnotation extends HPdfInfo {

    private static final Logger logger = LoggerFactory.getLogger(HPdfAnnotation.class);
    
    public enum HPdfBSSubtype {
        SOLID("S"),
        DASHED("D"),
        BEVELED("B"),
        INSET("I"),
        UNDERLINED("U"),
        ;
        String code;
        HPdfBSSubtype(String code) {
            this.code = code;
        }
        String getCode() {
            return this.code;
        }
    }
    
    private static int HPDF_BS_DEF_WIDTH = 1;
    
    protected HPdfAnnotation(HPdfXref xref, HPdfRect rect) {
        super();
        
        logger.trace("HPdfAnnotation#ctor");
        
        xref.add(this);
        
        HPdfArray array = new HPdfArray();
        
        this.add("Rect", array);
        
        HPdfRect rect_ = rect.clone(); // MEMO: original code 'rect' is copy of caller's.
        rect_.reviseVertically();
        
        array.addReal(rect_.getLeft());
        array.addReal(rect_.getBottom());
        array.addReal(rect_.getRight());
        array.addReal(rect_.getTop());
        
        this.addName("Type", "Annot");
        
        // MEMO: concrete subclasses MUST call addName("Subtype", foo) for regist their Subtype.
    }
    
    protected boolean validate() {
        // MEMO: original HPDF_Annotation_Validate() and CheckSubType()
        // are almost substituted by runtime type check.
        // so here check only subtype is surely defined.
        HPdfName subtype = this.getItem("Subtype", HPdfName.class);
        if (subtype == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_ANNOTATION, 0);
        }
        return true;
    }
    
    public final void setBorderStyle(HPdfBSSubtype subtype, float width, int dashOn, int dashOff, int dashPhase) {
        logger.trace("HPdfAnnotation#setBorderStyle");
        
        HPdfDict bs = new HPdfDict();
        this.add("BS", bs);
        
        if (subtype == HPdfBSSubtype.DASHED) {
            HPdfArray dash = new HPdfArray();
            bs.add("D", dash);
            bs.addName("Type", "Border");
            dash.addReal(dashOn);
            dash.addReal(dashOff);
            
            if (dashPhase != 0) {
                // MEMO same to original code,
                // but probably dashPhase ?
                dash.addReal(dashOff);
            }
        }
        
        bs.addName("S", subtype.getCode());
        
        if (width != HPDF_BS_DEF_WIDTH) {
            bs.addReal("W", width);
        }
    }
    
    public final void setRGBColor(HPdfRGBColor color) {
        logger.trace("HPdfAnnotation#setRGBColor");
        
        HPdfArray cArray = new HPdfArray();
        
        this.add("C", cArray);
        cArray.addReal(color.getR());
        cArray.addReal(color.getG());
        cArray.addReal(color.getB());
    }
    
    public final void setCMYKColor(HPdfCMYKColor color) {
        logger.trace("HPdfAnnotation#setCMYKColor");
        
        HPdfArray cArray = new HPdfArray();
        
        this.add("C", cArray);
        cArray.addReal(color.getC());
        cArray.addReal(color.getM());
        cArray.addReal(color.getY());
        cArray.addReal(color.getK());
    }
    
    public final void setGrayColor(float color) {
        logger.trace("HPdfAnnotation#setGrayColor");
        
        HPdfArray cArray = new HPdfArray();
        
        this.add("C", cArray);
        cArray.addReal(color);
    }
    
    public final void setNoColor() {
        logger.trace("HPdfAnnotation#setNoColor");
        
        HPdfArray cArray = new HPdfArray();
        
        this.add("C", cArray);
    }
    
    public final void set3DView(HPdf3DAnnotation annot3d, HPdfDict view3d) {
        HPdfDict exData = new HPdfDict();
        
        exData.addName("Type", "ExData");
        exData.addName("Subtype", "Markup3D");
        exData.add("3DA", annot3d);
        
        HPdfProxy proxyView3d = new HPdfProxy(view3d);
        
        exData.add("3DV", proxyView3d);
        this.add("ExData", exData);
    }

}
