/*
 * JHPdf Free PDF Library : HPdfAbstractWriteStream.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.io;

import java.nio.charset.StandardCharsets;

import net.sf.jhpdf.HPdfConst;
import net.sf.jhpdf.HPdfUtil;
import net.sf.jhpdf.encrypt.HPdfEncrypt;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Abstract stream class for writing.
 * It has some utility methods.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public abstract class HPdfAbstractWriteStream implements HPdfWriteStream {

    private static final Logger logger = LoggerFactory.getLogger(HPdfAbstractWriteStream.class);

    private final HPdfStreamSignature signature = new HPdfStreamSignature();
    
    @Override
    public boolean validate() {
        return signature.validate();
    }
    
    @Override
    public final void write(byte[] buf, int siz) {
        logger.trace("HPdfAbstractWriteStream#write");
        write(buf, 0, siz);
    }

    @Override
    public final void writeAsBytes(String value) {
        logger.trace("HPdfAbstractWriteStream#writeAsBytes");
        this.writeStr(value);
    }

    @Override
    public final void writeStr(String value) {
        logger.trace("HPdfAbstractWriteStream#writeStr");
        // MEMO: assumes encoding is ISO-8859-1 compatible.
        byte[] tmp = value.getBytes(StandardCharsets.ISO_8859_1);
        write(tmp, tmp.length);
    }

    @Override
    public final void writeInt(long value) {
        logger.trace("HPdfAbstractWriteStream#writeInt");
        writeStr(Integer.toString((int) value));
    }

    @Override
    public final void writeUInt(long value) {
        logger.trace("HPdfAbstractWriteStream#writeUInt");
        writeStr(Long.toString(value & 0xFFFFFFFFL));
    }
    
    @Override
    public final void writeReal(float value) {
        logger.trace("HPdfAbstractWriteStream#writeReal");
        // MEMO: PDF accepts only 5 digits of precision in fractional part.
        // see PDF specification "C Implementation Limits".
        writeStr(String.format("%.5f", value));
    }

    @Override
    public final void writeChar(char value) {
        logger.trace("HPdfAbstractWriteStream#writeChar");
        // MEMO: this method really writes a byte.
        writeByte((byte) value);
    }

    @Override
    public final void writeUChar(char value) {
        logger.trace("HPdfAbstractWriteStream#writeUChar");
        // MEMO: this method really writes a byte as unsigned.
        writeByte((byte) (value & 0xFF));
    }

    private byte[] bufToWriteByte = new byte[1];
    
    private final void writeByte(byte value) {
        bufToWriteByte[0] = value;
        write(bufToWriteByte, 0, 1);
    }
    
    @Override
    public final void writeBinary(byte[] value, long len, HPdfEncrypt e) {
        logger.trace("HPdfAbstractWriteStream#writeBinary");

        byte[] src;
        if (e != null) {
            src = new byte[(int) len];
            e.cryptBuf(value, src, (int) len);
        } else {
            src = value;
        }
        
        byte[] buf = new byte[HPdfConst.HPDF_TEXT_DEFAULT_LEN];
        int idx = 0;
        for (int i = 0; i < len; ++i) {
            byte2HexChars(src[i], buf, idx);
            idx += 2;
            
            if (idx > HPdfConst.HPDF_TEXT_DEFAULT_LEN - 2) {
                write(buf, idx);
                idx = 0;
            }
        }
        
        if (idx > 0) {
            write(buf, idx);
        }
    }

    @Override
    public final void writeEscapeName(String value) {
        logger.trace("HPdfAbstractWriteStream#writeEscapeName");
        
        // MEMO: assumes byte encoding is ISO-8859-1
        byte[] textBytes = value.getBytes(StandardCharsets.ISO_8859_1);
        byte[] buf =
                new byte[HPdfConst.HPDF_LIMIT_MAX_NAME_LEN * 3 + 2];
        int len = Math.min(textBytes.length, HPdfConst.HPDF_LIMIT_MAX_NAME_LEN);
        int pos2 = 0;
        
        buf[pos2++] = '/';
        
        for (int i = 0; i < len; ++i) {
            // MEMO: here treats char type as 'unsigned 16bit int'.
            // OK, 0x30 is not equivalent '0', I see...
            char c = (char)(textBytes[i] & 0xffff);
            if (HPdfUtil.needsEscape(c)) {
                // MEMO: convert to hexadecimal expression.
                buf[pos2++] = '#';
                byte2HexChars((byte) c, buf, pos2);
                pos2 += 2;
            } else {
                buf[pos2++] = (byte)c;
            }
        }
        
        write(buf, pos2);
    }
    
    private final void byte2HexChars(byte val, byte[] result, int offset) {
        result[offset] = (byte) (val >> 4);
        result[offset + 1] = (byte)(val & 0x0f);
        for (int i = 0; i < 1; ++i) {
            if (result[offset + i] <= 9) {
                result[offset + i] += 0x30;
            } else {
                result[offset + i] += 0x41 - 10;
            }
        }
    }

    @Override
    public final void writeEscapeText(String text) {
        logger.trace("HPdfAbstractWriteStream#writeEscapeText");
        int len = 0;
        if (text != null) {
            // MEMO: assume text consists only of 'single byte characters' here.
            len = Math.min(text.length(), HPdfConst.HPDF_LIMIT_MAX_STRING_LEN);
        }
        writeEscapeText2(text, len);
    }

    @Override
    public final void writeEscapeText2(String text, int len) {
        logger.trace("HPdfAbstractWriteStream#writeEscapeText2");
       /* The following block is commented out because it violates "PDF Spec 7.3.4.2 Literal Strings". 
        * It states that the two matching parentheses must still be present to represent an empty 
        * string of zero length. 
        */
       /*
        if (len == 0) {
            return;
        }
       */
        // MEMO: assume byte encoding is ISO-8859-1 compatible
        byte[] textBytes = text.getBytes(StandardCharsets.ISO_8859_1);
        byte[] buf = new byte[HPdfConst.HPDF_TEXT_DEFAULT_LEN];
        int idx = 0;
        
        buf[idx++] = '(';
        
        for (int i = 0; i < len; ++i) {
            // MEMO: here treats char type as 'unsigned 16bit int'.
            // OK, 0x30 is not equivalent '0', I see...
            char c = (char)(textBytes[i] & 0xffff);
            if (HPdfUtil.needsEscape(c)) {
                // MEMO: convert to octal expression.
                buf[idx++] = '\\';
                buf[idx] = (byte)(c >> 6);
                buf[idx] += 0x30;
                idx++;
                buf[idx] = (byte)((c & 0x38) >> 3);
                buf[idx] += 0x30;
                idx++;
                buf[idx] = (byte)(c & 0x07);
                buf[idx] += 0x30;
                idx++;
            } else {
                buf[idx++] = (byte)c;
            }
            
            if (idx > HPdfConst.HPDF_TEXT_DEFAULT_LEN - 4) {
                write(buf, idx);
                idx = 0;
            }
        }
        buf[idx++] = ')';
        
        write(buf, idx);
    }
}
