/*
 * JHPdf Free PDF Library : HPdfEncoder.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.encoder;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sf.jhpdf.HPdfException;
import net.sf.jhpdf.io.HPdfWriteStream;
import net.sf.jhpdf.pdfobject.HPdfArray;
import net.sf.jhpdf.pdfobject.HPdfFont.HPdfWritingMode;

/**
 * Abstract class represents various encoders.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public abstract class HPdfEncoder {
    // TODO only stub, needs methods definition.

    private static final Logger logger = LoggerFactory.getLogger(HPdfEncoder.class);
    
    public enum HPdfByteType {
        SINGLE,
        LEAD,
        TRIAL,
        UNKNOWN,
    }
    
    public enum HPdfEncoderType {
        SINGLE_BYTE,
        DOUBLE_BYTE,
        UNINITIALIZED,
        UNKNOWN,
    }
    /*----------------------------------------------------------------------------*/
    /*------ predefined font encodings -------------------------------------------*/

    public static final String HPDF_ENCODING_FONT_SPECIFIC  = "FontSpecific";
    public static final String HPDF_ENCODING_STANDARD       = "StandardEncoding";
    public static final String HPDF_ENCODING_MAC_ROMAN      = "MacRomanEncoding";
    public static final String HPDF_ENCODING_WIN_ANSI       = "WinAnsiEncoding";
    public static final String HPDF_ENCODING_ISO8859_2      = "ISO8859-2";
    public static final String HPDF_ENCODING_ISO8859_3      = "ISO8859-3";
    public static final String HPDF_ENCODING_ISO8859_4      = "ISO8859-4";
    public static final String HPDF_ENCODING_ISO8859_5      = "ISO8859-5";
    public static final String HPDF_ENCODING_ISO8859_6      = "ISO8859-6";
    public static final String HPDF_ENCODING_ISO8859_7      = "ISO8859-7";
    public static final String HPDF_ENCODING_ISO8859_8      = "ISO8859-8";
    public static final String HPDF_ENCODING_ISO8859_9      = "ISO8859-9";
    public static final String HPDF_ENCODING_ISO8859_10     = "ISO8859-10";
    public static final String HPDF_ENCODING_ISO8859_11     = "ISO8859-11";
    public static final String HPDF_ENCODING_ISO8859_13     = "ISO8859-13";
    public static final String HPDF_ENCODING_ISO8859_14     = "ISO8859-14";
    public static final String HPDF_ENCODING_ISO8859_15     = "ISO8859-15";
    public static final String HPDF_ENCODING_ISO8859_16     = "ISO8859-16";
    public static final String HPDF_ENCODING_CP1250         = "CP1250";
    public static final String HPDF_ENCODING_CP1251         = "CP1251";
    public static final String HPDF_ENCODING_CP1252         = "CP1252";
    public static final String HPDF_ENCODING_CP1253         = "CP1253";
    public static final String HPDF_ENCODING_CP1254         = "CP1254";
    public static final String HPDF_ENCODING_CP1255         = "CP1255";
    public static final String HPDF_ENCODING_CP1256         = "CP1256";
    public static final String HPDF_ENCODING_CP1257         = "CP1257";
    public static final String HPDF_ENCODING_CP1258         = "CP1258";
    public static final String HPDF_ENCODING_KOI8_R         = "KOI8-R";
    
    public static final String char_NOTDEF = ".notdef";
    
    private static Map<String, Character> HPDF_UNICODE_GLYPH_NAME_MAP;
    private static String HPDF_DEFAULT_GLYPH_NAME;
    
    public static char glyphNameToUnicode(String glyphName) {
        checkUnicodeGlyphMap();
        Character c = HPDF_UNICODE_GLYPH_NAME_MAP.get(glyphName);
        return c == null ? (char) 0 : c.charValue();
    }

    public static String unicodeToGlyphName(char code) {
        checkUnicodeGlyphMap();
        for (Map.Entry<String, Character> e : HPDF_UNICODE_GLYPH_NAME_MAP.entrySet()) {
            if (e.getValue().charValue() == code) {
                return e.getKey();
            }
        }
        return HPDF_DEFAULT_GLYPH_NAME;
    }
    
    private static final long HPDF_ENCODER_SIG_BYTES = 0x454E4344L;
    
    private long sigBytes;
    
    protected HPdfEncoder() {
        super();
        this.sigBytes = HPDF_ENCODER_SIG_BYTES;
        this.type = HPdfEncoderType.UNKNOWN;
    }
    
    public boolean validate() {
        logger.trace("HPdfEncoder#validate");
        
        return this.sigBytes == HPDF_ENCODER_SIG_BYTES;
    }
    
    private String name;

    /**
     * returns encoding name.
     * @return encoding name
     */
    public String getName() {
        return this.name;
    }
    
    /**
     * sets encoding name.
     * @param name
     */
    protected void setName(String name) {
        this.name = name;
    }
    

    private HPdfEncoderType type;
    
    public final HPdfEncoderType getType() {
        logger.trace("HPdfEncoder#getType");
        if (!this.validate()) {
            return HPdfEncoderType.UNKNOWN;
        }
        return type;
    }

    protected final void setType(HPdfEncoderType type) {
        this.type = type;
    }

    public final void setParseText(HPdfParseText parseState, byte[] src, int len) {
        logger.trace("HPdfEncoder#setParseText");
        parseState.setTarget(src, len);
    }
    
    /**
     * returns head byte type of parsing state.<br>
     * subclasses should override this method for custom behaviour. <br>
     * original function name is simply 'HPDF_Encoder_ByteType',
     * it's confusing against '_GetByteType'.
     * @param state encode parsing state
     * @return head byte type of parsing state.
     */
    public HPdfByteType getHeadByteType(HPdfParseText state) {
        return HPdfByteType.SINGLE;
    }
    
    public final HPdfByteType getByteType(byte[] text, int index) {
        logger.trace("HPdfEncoder#getByteType");
        
        if (!this.validate()) {
            return HPdfByteType.UNKNOWN;
        }
        if (this.getType() != HPdfEncoderType.DOUBLE_BYTE) {
            return HPdfByteType.SINGLE;
        }
        
        return getByteTypeByParse(text, index);
    }
    
    public HPdfWritingMode getWritingMode() {
        return HPdfWritingMode.HORIZONTAL;
    }
    
    // MEMO: original code is the 'class' specific logic
    //       in HPDF_Encoder_GetByteType().
    protected abstract HPdfByteType getByteTypeByParse(byte[] text, int index);

    public abstract char toUnicode(int code);
    
    /**
     * writes itself to stream as PDF Encoding Object.
     * @param stream
     */
    public abstract void writeValue(HPdfWriteStream stream);

    /**
     * decode byte array to Java(Unicode) String with Encoder specific method.<br>
     * subclasses must implement this.
     * @param buf byte array represents String of each Encoder's encoding.
     * @return
     */
    public abstract String decode(byte[] buf);
    
    /**
     * encode Java(Unicode) String to byte array with Encoder specific method.<br>
     * subclasses must implement this.
     * @param txt String to byte-encode.
     * @return
     */
    public abstract byte[] encode(String txt);
    
    /**
     * initialize encoder.
     * this method exists for delayed initialization of heavy instance.
     */
    public abstract void init();
    
    private static void checkUnicodeGlyphMap() {
        if (HPDF_UNICODE_GLYPH_NAME_MAP == null) {
            Map<String, Character> map = new HashMap<String, Character>();
            
            addUnicodeGlyphPair(map, 0x0000, char_NOTDEF);
            addUnicodeGlyphPair(map, 0x0020, "space");
            addUnicodeGlyphPair(map, 0x0021, "exclam");
            addUnicodeGlyphPair(map, 0x0022, "quotedbl");
            addUnicodeGlyphPair(map, 0x0023, "numbersign");
            addUnicodeGlyphPair(map, 0x0024, "dollar");
            addUnicodeGlyphPair(map, 0x0025, "percent");
            addUnicodeGlyphPair(map, 0x0026, "ampersand");
            addUnicodeGlyphPair(map, 0x0027, "quotesingle");
            addUnicodeGlyphPair(map, 0x0028, "parenleft");
            addUnicodeGlyphPair(map, 0x0029, "parenright");
            addUnicodeGlyphPair(map, 0x002A, "asterisk");
            addUnicodeGlyphPair(map, 0x002B, "plus");
            addUnicodeGlyphPair(map, 0x002C, "comma");
            addUnicodeGlyphPair(map, 0x002D, "hyphen");
            addUnicodeGlyphPair(map, 0x002E, "period");
            addUnicodeGlyphPair(map, 0x002F, "slash");
            addUnicodeGlyphPair(map, 0x0030, "zero");
            addUnicodeGlyphPair(map, 0x0031, "one");
            addUnicodeGlyphPair(map, 0x0032, "two");
            addUnicodeGlyphPair(map, 0x0033, "three");
            addUnicodeGlyphPair(map, 0x0034, "four");
            addUnicodeGlyphPair(map, 0x0035, "five");
            addUnicodeGlyphPair(map, 0x0036, "six");
            addUnicodeGlyphPair(map, 0x0037, "seven");
            addUnicodeGlyphPair(map, 0x0038, "eight");
            addUnicodeGlyphPair(map, 0x0039, "nine");
            addUnicodeGlyphPair(map, 0x003A, "colon");
            addUnicodeGlyphPair(map, 0x003B, "semicolon");
            addUnicodeGlyphPair(map, 0x003C, "less");
            addUnicodeGlyphPair(map, 0x003D, "equal");
            addUnicodeGlyphPair(map, 0x003E, "greater");
            addUnicodeGlyphPair(map, 0x003F, "question");
            addUnicodeGlyphPair(map, 0x0040, "at");
            addUnicodeGlyphPair(map, 0x0041, "A");
            addUnicodeGlyphPair(map, 0x0042, "B");
            addUnicodeGlyphPair(map, 0x0043, "C");
            addUnicodeGlyphPair(map, 0x0044, "D");
            addUnicodeGlyphPair(map, 0x0045, "E");
            addUnicodeGlyphPair(map, 0x0046, "F");
            addUnicodeGlyphPair(map, 0x0047, "G");
            addUnicodeGlyphPair(map, 0x0048, "H");
            addUnicodeGlyphPair(map, 0x0049, "I");
            addUnicodeGlyphPair(map, 0x004A, "J");
            addUnicodeGlyphPair(map, 0x004B, "K");
            addUnicodeGlyphPair(map, 0x004C, "L");
            addUnicodeGlyphPair(map, 0x004D, "M");
            addUnicodeGlyphPair(map, 0x004E, "N");
            addUnicodeGlyphPair(map, 0x004F, "O");
            addUnicodeGlyphPair(map, 0x0050, "P");
            addUnicodeGlyphPair(map, 0x0051, "Q");
            addUnicodeGlyphPair(map, 0x0052, "R");
            addUnicodeGlyphPair(map, 0x0053, "S");
            addUnicodeGlyphPair(map, 0x0054, "T");
            addUnicodeGlyphPair(map, 0x0055, "U");
            addUnicodeGlyphPair(map, 0x0056, "V");
            addUnicodeGlyphPair(map, 0x0057, "W");
            addUnicodeGlyphPair(map, 0x0058, "X");
            addUnicodeGlyphPair(map, 0x0059, "Y");
            addUnicodeGlyphPair(map, 0x005A, "Z");
            addUnicodeGlyphPair(map, 0x005B, "bracketleft");
            addUnicodeGlyphPair(map, 0x005C, "backslash");
            addUnicodeGlyphPair(map, 0x005D, "bracketright");
            addUnicodeGlyphPair(map, 0x005E, "asciicircum");
            addUnicodeGlyphPair(map, 0x005F, "underscore");
            addUnicodeGlyphPair(map, 0x0060, "grave");
            addUnicodeGlyphPair(map, 0x0061, "a");
            addUnicodeGlyphPair(map, 0x0062, "b");
            addUnicodeGlyphPair(map, 0x0063, "c");
            addUnicodeGlyphPair(map, 0x0064, "d");
            addUnicodeGlyphPair(map, 0x0065, "e");
            addUnicodeGlyphPair(map, 0x0066, "f");
            addUnicodeGlyphPair(map, 0x0067, "g");
            addUnicodeGlyphPair(map, 0x0068, "h");
            addUnicodeGlyphPair(map, 0x0069, "i");
            addUnicodeGlyphPair(map, 0x006A, "j");
            addUnicodeGlyphPair(map, 0x006B, "k");
            addUnicodeGlyphPair(map, 0x006C, "l");
            addUnicodeGlyphPair(map, 0x006D, "m");
            addUnicodeGlyphPair(map, 0x006E, "n");
            addUnicodeGlyphPair(map, 0x006F, "o");
            addUnicodeGlyphPair(map, 0x0070, "p");
            addUnicodeGlyphPair(map, 0x0071, "q");
            addUnicodeGlyphPair(map, 0x0072, "r");
            addUnicodeGlyphPair(map, 0x0073, "s");
            addUnicodeGlyphPair(map, 0x0074, "t");
            addUnicodeGlyphPair(map, 0x0075, "u");
            addUnicodeGlyphPair(map, 0x0076, "v");
            addUnicodeGlyphPair(map, 0x0077, "w");
            addUnicodeGlyphPair(map, 0x0078, "x");
            addUnicodeGlyphPair(map, 0x0079, "y");
            addUnicodeGlyphPair(map, 0x007A, "z");
            addUnicodeGlyphPair(map, 0x007B, "braceleft");
            addUnicodeGlyphPair(map, 0x007C, "bar");
            addUnicodeGlyphPair(map, 0x007D, "braceright");
            addUnicodeGlyphPair(map, 0x007E, "asciitilde");
            addUnicodeGlyphPair(map, 0x00A0, "space");
            addUnicodeGlyphPair(map, 0x00A1, "exclamdown");
            addUnicodeGlyphPair(map, 0x00A2, "cent");
            addUnicodeGlyphPair(map, 0x00A3, "sterling");
            addUnicodeGlyphPair(map, 0x00A4, "currency");
            addUnicodeGlyphPair(map, 0x00A5, "yen");
            addUnicodeGlyphPair(map, 0x00A6, "brokenbar");
            addUnicodeGlyphPair(map, 0x00A7, "section");
            addUnicodeGlyphPair(map, 0x00A8, "dieresis");
            addUnicodeGlyphPair(map, 0x00A9, "copyright");
            addUnicodeGlyphPair(map, 0x00AA, "ordfeminine");
            addUnicodeGlyphPair(map, 0x00AB, "guillemotleft");
            addUnicodeGlyphPair(map, 0x00AC, "logicalnot");
            addUnicodeGlyphPair(map, 0x00AD, "hyphen");
            addUnicodeGlyphPair(map, 0x00AE, "registered");
            addUnicodeGlyphPair(map, 0x00AF, "macron");
            addUnicodeGlyphPair(map, 0x00B0, "degree");
            addUnicodeGlyphPair(map, 0x00B1, "plusminus");
            addUnicodeGlyphPair(map, 0x00B2, "twosuperior");
            addUnicodeGlyphPair(map, 0x00B3, "threesuperior");
            addUnicodeGlyphPair(map, 0x00B4, "acute");
            addUnicodeGlyphPair(map, 0x00B5, "mu");
            addUnicodeGlyphPair(map, 0x00B6, "paragraph");
            addUnicodeGlyphPair(map, 0x00B7, "periodcentered");
            addUnicodeGlyphPair(map, 0x00B8, "cedilla");
            addUnicodeGlyphPair(map, 0x00B9, "onesuperior");
            addUnicodeGlyphPair(map, 0x00BA, "ordmasculine");
            addUnicodeGlyphPair(map, 0x00BB, "guillemotright");
            addUnicodeGlyphPair(map, 0x00BC, "onequarter");
            addUnicodeGlyphPair(map, 0x00BD, "onehalf");
            addUnicodeGlyphPair(map, 0x00BE, "threequarters");
            addUnicodeGlyphPair(map, 0x00BF, "questiondown");
            addUnicodeGlyphPair(map, 0x00C0, "Agrave");
            addUnicodeGlyphPair(map, 0x00C1, "Aacute");
            addUnicodeGlyphPair(map, 0x00C2, "Acircumflex");
            addUnicodeGlyphPair(map, 0x00C3, "Atilde");
            addUnicodeGlyphPair(map, 0x00C4, "Adieresis");
            addUnicodeGlyphPair(map, 0x00C5, "Aring");
            addUnicodeGlyphPair(map, 0x00C6, "AE");
            addUnicodeGlyphPair(map, 0x00C7, "Ccedilla");
            addUnicodeGlyphPair(map, 0x00C8, "Egrave");
            addUnicodeGlyphPair(map, 0x00C9, "Eacute");
            addUnicodeGlyphPair(map, 0x00CA, "Ecircumflex");
            addUnicodeGlyphPair(map, 0x00CB, "Edieresis");
            addUnicodeGlyphPair(map, 0x00CC, "Igrave");
            addUnicodeGlyphPair(map, 0x00CD, "Iacute");
            addUnicodeGlyphPair(map, 0x00CE, "Icircumflex");
            addUnicodeGlyphPair(map, 0x00CF, "Idieresis");
            addUnicodeGlyphPair(map, 0x00D0, "Eth");
            addUnicodeGlyphPair(map, 0x00D1, "Ntilde");
            addUnicodeGlyphPair(map, 0x00D2, "Ograve");
            addUnicodeGlyphPair(map, 0x00D3, "Oacute");
            addUnicodeGlyphPair(map, 0x00D4, "Ocircumflex");
            addUnicodeGlyphPair(map, 0x00D5, "Otilde");
            addUnicodeGlyphPair(map, 0x00D6, "Odieresis");
            addUnicodeGlyphPair(map, 0x00D7, "multiply");
            addUnicodeGlyphPair(map, 0x00D8, "Oslash");
            addUnicodeGlyphPair(map, 0x00D9, "Ugrave");
            addUnicodeGlyphPair(map, 0x00DA, "Uacute");
            addUnicodeGlyphPair(map, 0x00DB, "Ucircumflex");
            addUnicodeGlyphPair(map, 0x00DC, "Udieresis");
            addUnicodeGlyphPair(map, 0x00DD, "Yacute");
            addUnicodeGlyphPair(map, 0x00DE, "Thorn");
            addUnicodeGlyphPair(map, 0x00DF, "germandbls");
            addUnicodeGlyphPair(map, 0x00E0, "agrave");
            addUnicodeGlyphPair(map, 0x00E1, "aacute");
            addUnicodeGlyphPair(map, 0x00E2, "acircumflex");
            addUnicodeGlyphPair(map, 0x00E3, "atilde");
            addUnicodeGlyphPair(map, 0x00E4, "adieresis");
            addUnicodeGlyphPair(map, 0x00E5, "aring");
            addUnicodeGlyphPair(map, 0x00E6, "ae");
            addUnicodeGlyphPair(map, 0x00E7, "ccedilla");
            addUnicodeGlyphPair(map, 0x00E8, "egrave");
            addUnicodeGlyphPair(map, 0x00E9, "eacute");
            addUnicodeGlyphPair(map, 0x00EA, "ecircumflex");
            addUnicodeGlyphPair(map, 0x00EB, "edieresis");
            addUnicodeGlyphPair(map, 0x00EC, "igrave");
            addUnicodeGlyphPair(map, 0x00ED, "iacute");
            addUnicodeGlyphPair(map, 0x00EE, "icircumflex");
            addUnicodeGlyphPair(map, 0x00EF, "idieresis");
            addUnicodeGlyphPair(map, 0x00F0, "eth");
            addUnicodeGlyphPair(map, 0x00F1, "ntilde");
            addUnicodeGlyphPair(map, 0x00F2, "ograve");
            addUnicodeGlyphPair(map, 0x00F3, "oacute");
            addUnicodeGlyphPair(map, 0x00F4, "ocircumflex");
            addUnicodeGlyphPair(map, 0x00F5, "otilde");
            addUnicodeGlyphPair(map, 0x00F6, "odieresis");
            addUnicodeGlyphPair(map, 0x00F7, "divide");
            addUnicodeGlyphPair(map, 0x00F8, "oslash");
            addUnicodeGlyphPair(map, 0x00F9, "ugrave");
            addUnicodeGlyphPair(map, 0x00FA, "uacute");
            addUnicodeGlyphPair(map, 0x00FB, "ucircumflex");
            addUnicodeGlyphPair(map, 0x00FC, "udieresis");
            addUnicodeGlyphPair(map, 0x00FD, "yacute");
            addUnicodeGlyphPair(map, 0x00FE, "thorn");
            addUnicodeGlyphPair(map, 0x00FF, "ydieresis");
            addUnicodeGlyphPair(map, 0x0100, "Amacron");
            addUnicodeGlyphPair(map, 0x0101, "amacron");
            addUnicodeGlyphPair(map, 0x0102, "Abreve");
            addUnicodeGlyphPair(map, 0x0103, "abreve");
            addUnicodeGlyphPair(map, 0x0104, "Aogonek");
            addUnicodeGlyphPair(map, 0x0105, "aogonek");
            addUnicodeGlyphPair(map, 0x0106, "Cacute");
            addUnicodeGlyphPair(map, 0x0107, "cacute");
            addUnicodeGlyphPair(map, 0x0108, "Ccircumflex");
            addUnicodeGlyphPair(map, 0x0109, "ccircumflex");
            addUnicodeGlyphPair(map, 0x010A, "Cdotaccent");
            addUnicodeGlyphPair(map, 0x010B, "cdotaccent");
            addUnicodeGlyphPair(map, 0x010C, "Ccaron");
            addUnicodeGlyphPair(map, 0x010D, "ccaron");
            addUnicodeGlyphPair(map, 0x010E, "Dcaron");
            addUnicodeGlyphPair(map, 0x010F, "dcaron");
            addUnicodeGlyphPair(map, 0x0110, "Dcroat");
            addUnicodeGlyphPair(map, 0x0111, "dcroat");
            addUnicodeGlyphPair(map, 0x0112, "Emacron");
            addUnicodeGlyphPair(map, 0x0113, "emacron");
            addUnicodeGlyphPair(map, 0x0114, "Ebreve");
            addUnicodeGlyphPair(map, 0x0115, "ebreve");
            addUnicodeGlyphPair(map, 0x0116, "Edotaccent");
            addUnicodeGlyphPair(map, 0x0117, "edotaccent");
            addUnicodeGlyphPair(map, 0x0118, "Eogonek");
            addUnicodeGlyphPair(map, 0x0119, "eogonek");
            addUnicodeGlyphPair(map, 0x011A, "Ecaron");
            addUnicodeGlyphPair(map, 0x011B, "ecaron");
            addUnicodeGlyphPair(map, 0x011C, "Gcircumflex");
            addUnicodeGlyphPair(map, 0x011D, "gcircumflex");
            addUnicodeGlyphPair(map, 0x011E, "Gbreve");
            addUnicodeGlyphPair(map, 0x011F, "gbreve");
            addUnicodeGlyphPair(map, 0x0120, "Gdotaccent");
            addUnicodeGlyphPair(map, 0x0121, "gdotaccent");
            addUnicodeGlyphPair(map, 0x0122, "Gcommaaccent");
            addUnicodeGlyphPair(map, 0x0123, "gcommaaccent");
            addUnicodeGlyphPair(map, 0x0124, "Hcircumflex");
            addUnicodeGlyphPair(map, 0x0125, "hcircumflex");
            addUnicodeGlyphPair(map, 0x0126, "Hbar");
            addUnicodeGlyphPair(map, 0x0127, "hbar");
            addUnicodeGlyphPair(map, 0x0128, "Itilde");
            addUnicodeGlyphPair(map, 0x0129, "itilde");
            addUnicodeGlyphPair(map, 0x012A, "Imacron");
            addUnicodeGlyphPair(map, 0x012B, "imacron");
            addUnicodeGlyphPair(map, 0x012C, "Ibreve");
            addUnicodeGlyphPair(map, 0x012D, "ibreve");
            addUnicodeGlyphPair(map, 0x012E, "Iogonek");
            addUnicodeGlyphPair(map, 0x012F, "iogonek");
            addUnicodeGlyphPair(map, 0x0130, "Idotaccent");
            addUnicodeGlyphPair(map, 0x0131, "dotlessi");
            addUnicodeGlyphPair(map, 0x0132, "IJ");
            addUnicodeGlyphPair(map, 0x0133, "ij");
            addUnicodeGlyphPair(map, 0x0134, "Jcircumflex");
            addUnicodeGlyphPair(map, 0x0135, "jcircumflex");
            addUnicodeGlyphPair(map, 0x0136, "Kcommaaccent");
            addUnicodeGlyphPair(map, 0x0137, "kcommaaccent");
            addUnicodeGlyphPair(map, 0x0138, "kgreenlandic");
            addUnicodeGlyphPair(map, 0x0139, "Lacute");
            addUnicodeGlyphPair(map, 0x013A, "lacute");
            addUnicodeGlyphPair(map, 0x013B, "Lcommaaccent");
            addUnicodeGlyphPair(map, 0x013C, "lcommaaccent");
            addUnicodeGlyphPair(map, 0x013D, "Lcaron");
            addUnicodeGlyphPair(map, 0x013E, "lcaron");
            addUnicodeGlyphPair(map, 0x013F, "Ldot");
            addUnicodeGlyphPair(map, 0x0140, "ldot");
            addUnicodeGlyphPair(map, 0x0141, "Lslash");
            addUnicodeGlyphPair(map, 0x0142, "lslash");
            addUnicodeGlyphPair(map, 0x0143, "Nacute");
            addUnicodeGlyphPair(map, 0x0144, "nacute");
            addUnicodeGlyphPair(map, 0x0145, "Ncommaaccent");
            addUnicodeGlyphPair(map, 0x0146, "ncommaaccent");
            addUnicodeGlyphPair(map, 0x0147, "Ncaron");
            addUnicodeGlyphPair(map, 0x0148, "ncaron");
            addUnicodeGlyphPair(map, 0x0149, "napostrophe");
            addUnicodeGlyphPair(map, 0x014A, "Eng");
            addUnicodeGlyphPair(map, 0x014B, "eng");
            addUnicodeGlyphPair(map, 0x014C, "Omacron");
            addUnicodeGlyphPair(map, 0x014D, "omacron");
            addUnicodeGlyphPair(map, 0x014E, "Obreve");
            addUnicodeGlyphPair(map, 0x014F, "obreve");
            addUnicodeGlyphPair(map, 0x0150, "Ohungarumlaut");
            addUnicodeGlyphPair(map, 0x0151, "ohungarumlaut");
            addUnicodeGlyphPair(map, 0x0152, "OE");
            addUnicodeGlyphPair(map, 0x0153, "oe");
            addUnicodeGlyphPair(map, 0x0154, "Racute");
            addUnicodeGlyphPair(map, 0x0155, "racute");
            addUnicodeGlyphPair(map, 0x0156, "Rcommaaccent");
            addUnicodeGlyphPair(map, 0x0157, "rcommaaccent");
            addUnicodeGlyphPair(map, 0x0158, "Rcaron");
            addUnicodeGlyphPair(map, 0x0159, "rcaron");
            addUnicodeGlyphPair(map, 0x015A, "Sacute");
            addUnicodeGlyphPair(map, 0x015B, "sacute");
            addUnicodeGlyphPair(map, 0x015C, "Scircumflex");
            addUnicodeGlyphPair(map, 0x015D, "scircumflex");
            addUnicodeGlyphPair(map, 0x015E, "Scedilla");
            addUnicodeGlyphPair(map, 0x015F, "scedilla");
            addUnicodeGlyphPair(map, 0x0160, "Scaron");
            addUnicodeGlyphPair(map, 0x0161, "scaron");
            addUnicodeGlyphPair(map, 0x0162, "Tcommaaccent");
            addUnicodeGlyphPair(map, 0x0163, "tcommaaccent");
            addUnicodeGlyphPair(map, 0x0164, "Tcaron");
            addUnicodeGlyphPair(map, 0x0165, "tcaron");
            addUnicodeGlyphPair(map, 0x0166, "Tbar");
            addUnicodeGlyphPair(map, 0x0167, "tbar");
            addUnicodeGlyphPair(map, 0x0168, "Utilde");
            addUnicodeGlyphPair(map, 0x0169, "utilde");
            addUnicodeGlyphPair(map, 0x016A, "Umacron");
            addUnicodeGlyphPair(map, 0x016B, "umacron");
            addUnicodeGlyphPair(map, 0x016C, "Ubreve");
            addUnicodeGlyphPair(map, 0x016D, "ubreve");
            addUnicodeGlyphPair(map, 0x016E, "Uring");
            addUnicodeGlyphPair(map, 0x016F, "uring");
            addUnicodeGlyphPair(map, 0x0170, "Uhungarumlaut");
            addUnicodeGlyphPair(map, 0x0171, "uhungarumlaut");
            addUnicodeGlyphPair(map, 0x0172, "Uogonek");
            addUnicodeGlyphPair(map, 0x0173, "uogonek");
            addUnicodeGlyphPair(map, 0x0174, "Wcircumflex");
            addUnicodeGlyphPair(map, 0x0175, "wcircumflex");
            addUnicodeGlyphPair(map, 0x0176, "Ycircumflex");
            addUnicodeGlyphPair(map, 0x0177, "ycircumflex");
            addUnicodeGlyphPair(map, 0x0178, "Ydieresis");
            addUnicodeGlyphPair(map, 0x0179, "Zacute");
            addUnicodeGlyphPair(map, 0x017A, "zacute");
            addUnicodeGlyphPair(map, 0x017B, "Zdotaccent");
            addUnicodeGlyphPair(map, 0x017C, "zdotaccent");
            addUnicodeGlyphPair(map, 0x017D, "Zcaron");
            addUnicodeGlyphPair(map, 0x017E, "zcaron");
            addUnicodeGlyphPair(map, 0x017F, "longs");
            addUnicodeGlyphPair(map, 0x0192, "florin");
            addUnicodeGlyphPair(map, 0x01A0, "Ohorn");
            addUnicodeGlyphPair(map, 0x01A1, "ohorn");
            addUnicodeGlyphPair(map, 0x01AF, "Uhorn");
            addUnicodeGlyphPair(map, 0x01B0, "uhorn");
            addUnicodeGlyphPair(map, 0x01E6, "Gcaron");
            addUnicodeGlyphPair(map, 0x01E7, "gcaron");
            addUnicodeGlyphPair(map, 0x01FA, "Aringacute");
            addUnicodeGlyphPair(map, 0x01FB, "aringacute");
            addUnicodeGlyphPair(map, 0x01FC, "AEacute");
            addUnicodeGlyphPair(map, 0x01FD, "aeacute");
            addUnicodeGlyphPair(map, 0x01FE, "Oslashacute");
            addUnicodeGlyphPair(map, 0x01FF, "oslashacute");
            addUnicodeGlyphPair(map, 0x0218, "Scommaaccent");
            addUnicodeGlyphPair(map, 0x0219, "scommaaccent");
            addUnicodeGlyphPair(map, 0x021A, "Tcommaaccent");
            addUnicodeGlyphPair(map, 0x021B, "tcommaaccent");
            addUnicodeGlyphPair(map, 0x02BC, "afii57929");
            addUnicodeGlyphPair(map, 0x02BD, "afii64937");
            addUnicodeGlyphPair(map, 0x02C6, "circumflex");
            addUnicodeGlyphPair(map, 0x02C7, "caron");
            addUnicodeGlyphPair(map, 0x02C9, "macron");
            addUnicodeGlyphPair(map, 0x02D8, "breve");
            addUnicodeGlyphPair(map, 0x02D9, "dotaccent");
            addUnicodeGlyphPair(map, 0x02DA, "ring");
            addUnicodeGlyphPair(map, 0x02DB, "ogonek");
            addUnicodeGlyphPair(map, 0x02DC, "tilde");
            addUnicodeGlyphPair(map, 0x02DD, "hungarumlaut");
            addUnicodeGlyphPair(map, 0x0300, "gravecomb");
            addUnicodeGlyphPair(map, 0x0301, "acutecomb");
            addUnicodeGlyphPair(map, 0x0303, "tildecomb");
            addUnicodeGlyphPair(map, 0x0309, "hookabovecomb");
            addUnicodeGlyphPair(map, 0x0323, "dotbelowcomb");
            addUnicodeGlyphPair(map, 0x0384, "tonos");
            addUnicodeGlyphPair(map, 0x0385, "dieresistonos");
            addUnicodeGlyphPair(map, 0x0386, "Alphatonos");
            addUnicodeGlyphPair(map, 0x0387, "anoteleia");
            addUnicodeGlyphPair(map, 0x0388, "Epsilontonos");
            addUnicodeGlyphPair(map, 0x0389, "Etatonos");
            addUnicodeGlyphPair(map, 0x038A, "Iotatonos");
            addUnicodeGlyphPair(map, 0x038C, "Omicrontonos");
            addUnicodeGlyphPair(map, 0x038E, "Upsilontonos");
            addUnicodeGlyphPair(map, 0x038F, "Omegatonos");
            addUnicodeGlyphPair(map, 0x0390, "iotadieresistonos");
            addUnicodeGlyphPair(map, 0x0391, "Alpha");
            addUnicodeGlyphPair(map, 0x0392, "Beta");
            addUnicodeGlyphPair(map, 0x0393, "Gamma");
            addUnicodeGlyphPair(map, 0x0394, "Delta");
            addUnicodeGlyphPair(map, 0x0395, "Epsilon");
            addUnicodeGlyphPair(map, 0x0396, "Zeta");
            addUnicodeGlyphPair(map, 0x0397, "Eta");
            addUnicodeGlyphPair(map, 0x0398, "Theta");
            addUnicodeGlyphPair(map, 0x0399, "Iota");
            addUnicodeGlyphPair(map, 0x039A, "Kappa");
            addUnicodeGlyphPair(map, 0x039B, "Lambda");
            addUnicodeGlyphPair(map, 0x039C, "Mu");
            addUnicodeGlyphPair(map, 0x039D, "Nu");
            addUnicodeGlyphPair(map, 0x039E, "Xi");
            addUnicodeGlyphPair(map, 0x039F, "Omicron");
            addUnicodeGlyphPair(map, 0x03A0, "Pi");
            addUnicodeGlyphPair(map, 0x03A1, "Rho");
            addUnicodeGlyphPair(map, 0x03A3, "Sigma");
            addUnicodeGlyphPair(map, 0x03A4, "Tau");
            addUnicodeGlyphPair(map, 0x03A5, "Upsilon");
            addUnicodeGlyphPair(map, 0x03A6, "Phi");
            addUnicodeGlyphPair(map, 0x03A7, "Chi");
            addUnicodeGlyphPair(map, 0x03A8, "Psi");
            addUnicodeGlyphPair(map, 0x03A9, "Omega");
            addUnicodeGlyphPair(map, 0x03AA, "Iotadieresis");
            addUnicodeGlyphPair(map, 0x03AB, "Upsilondieresis");
            addUnicodeGlyphPair(map, 0x03AC, "alphatonos");
            addUnicodeGlyphPair(map, 0x03AD, "epsilontonos");
            addUnicodeGlyphPair(map, 0x03AE, "etatonos");
            addUnicodeGlyphPair(map, 0x03AF, "iotatonos");
            addUnicodeGlyphPair(map, 0x03B0, "upsilondieresistonos");
            addUnicodeGlyphPair(map, 0x03B1, "alpha");
            addUnicodeGlyphPair(map, 0x03B2, "beta");
            addUnicodeGlyphPair(map, 0x03B3, "gamma");
            addUnicodeGlyphPair(map, 0x03B4, "delta");
            addUnicodeGlyphPair(map, 0x03B5, "epsilon");
            addUnicodeGlyphPair(map, 0x03B6, "zeta");
            addUnicodeGlyphPair(map, 0x03B7, "eta");
            addUnicodeGlyphPair(map, 0x03B8, "theta");
            addUnicodeGlyphPair(map, 0x03B9, "iota");
            addUnicodeGlyphPair(map, 0x03BA, "kappa");
            addUnicodeGlyphPair(map, 0x03BB, "lambda");
            addUnicodeGlyphPair(map, 0x03BC, "mu");
            addUnicodeGlyphPair(map, 0x03BD, "nu");
            addUnicodeGlyphPair(map, 0x03BE, "xi");
            addUnicodeGlyphPair(map, 0x03BF, "omicron");
            addUnicodeGlyphPair(map, 0x03C0, "pi");
            addUnicodeGlyphPair(map, 0x03C1, "rho");
            addUnicodeGlyphPair(map, 0x03C2, "sigma1");
            addUnicodeGlyphPair(map, 0x03C3, "sigma");
            addUnicodeGlyphPair(map, 0x03C4, "tau");
            addUnicodeGlyphPair(map, 0x03C5, "upsilon");
            addUnicodeGlyphPair(map, 0x03C6, "phi");
            addUnicodeGlyphPair(map, 0x03C7, "chi");
            addUnicodeGlyphPair(map, 0x03C8, "psi");
            addUnicodeGlyphPair(map, 0x03C9, "omega");
            addUnicodeGlyphPair(map, 0x03CA, "iotadieresis");
            addUnicodeGlyphPair(map, 0x03CB, "upsilondieresis");
            addUnicodeGlyphPair(map, 0x03CC, "omicrontonos");
            addUnicodeGlyphPair(map, 0x03CD, "upsilontonos");
            addUnicodeGlyphPair(map, 0x03CE, "omegatonos");
            addUnicodeGlyphPair(map, 0x03D1, "theta1");
            addUnicodeGlyphPair(map, 0x03D2, "Upsilon1");
            addUnicodeGlyphPair(map, 0x03D5, "phi1");
            addUnicodeGlyphPair(map, 0x03D6, "omega1");
            addUnicodeGlyphPair(map, 0x0401, "afii10023");
            addUnicodeGlyphPair(map, 0x0402, "afii10051");
            addUnicodeGlyphPair(map, 0x0403, "afii10052");
            addUnicodeGlyphPair(map, 0x0404, "afii10053");
            addUnicodeGlyphPair(map, 0x0405, "afii10054");
            addUnicodeGlyphPair(map, 0x0406, "afii10055");
            addUnicodeGlyphPair(map, 0x0407, "afii10056");
            addUnicodeGlyphPair(map, 0x0408, "afii10057");
            addUnicodeGlyphPair(map, 0x0409, "afii10058");
            addUnicodeGlyphPair(map, 0x040A, "afii10059");
            addUnicodeGlyphPair(map, 0x040B, "afii10060");
            addUnicodeGlyphPair(map, 0x040C, "afii10061");
            addUnicodeGlyphPair(map, 0x040E, "afii10062");
            addUnicodeGlyphPair(map, 0x040F, "afii10145");
            addUnicodeGlyphPair(map, 0x0410, "afii10017");
            addUnicodeGlyphPair(map, 0x0411, "afii10018");
            addUnicodeGlyphPair(map, 0x0412, "afii10019");
            addUnicodeGlyphPair(map, 0x0413, "afii10020");
            addUnicodeGlyphPair(map, 0x0414, "afii10021");
            addUnicodeGlyphPair(map, 0x0415, "afii10022");
            addUnicodeGlyphPair(map, 0x0416, "afii10024");
            addUnicodeGlyphPair(map, 0x0417, "afii10025");
            addUnicodeGlyphPair(map, 0x0418, "afii10026");
            addUnicodeGlyphPair(map, 0x0419, "afii10027");
            addUnicodeGlyphPair(map, 0x041A, "afii10028");
            addUnicodeGlyphPair(map, 0x041B, "afii10029");
            addUnicodeGlyphPair(map, 0x041C, "afii10030");
            addUnicodeGlyphPair(map, 0x041D, "afii10031");
            addUnicodeGlyphPair(map, 0x041E, "afii10032");
            addUnicodeGlyphPair(map, 0x041F, "afii10033");
            addUnicodeGlyphPair(map, 0x0420, "afii10034");
            addUnicodeGlyphPair(map, 0x0421, "afii10035");
            addUnicodeGlyphPair(map, 0x0422, "afii10036");
            addUnicodeGlyphPair(map, 0x0423, "afii10037");
            addUnicodeGlyphPair(map, 0x0424, "afii10038");
            addUnicodeGlyphPair(map, 0x0425, "afii10039");
            addUnicodeGlyphPair(map, 0x0426, "afii10040");
            addUnicodeGlyphPair(map, 0x0427, "afii10041");
            addUnicodeGlyphPair(map, 0x0428, "afii10042");
            addUnicodeGlyphPair(map, 0x0429, "afii10043");
            addUnicodeGlyphPair(map, 0x042A, "afii10044");
            addUnicodeGlyphPair(map, 0x042B, "afii10045");
            addUnicodeGlyphPair(map, 0x042C, "afii10046");
            addUnicodeGlyphPair(map, 0x042D, "afii10047");
            addUnicodeGlyphPair(map, 0x042E, "afii10048");
            addUnicodeGlyphPair(map, 0x042F, "afii10049");
            addUnicodeGlyphPair(map, 0x0430, "afii10065");
            addUnicodeGlyphPair(map, 0x0431, "afii10066");
            addUnicodeGlyphPair(map, 0x0432, "afii10067");
            addUnicodeGlyphPair(map, 0x0433, "afii10068");
            addUnicodeGlyphPair(map, 0x0434, "afii10069");
            addUnicodeGlyphPair(map, 0x0435, "afii10070");
            addUnicodeGlyphPair(map, 0x0436, "afii10072");
            addUnicodeGlyphPair(map, 0x0437, "afii10073");
            addUnicodeGlyphPair(map, 0x0438, "afii10074");
            addUnicodeGlyphPair(map, 0x0439, "afii10075");
            addUnicodeGlyphPair(map, 0x043A, "afii10076");
            addUnicodeGlyphPair(map, 0x043B, "afii10077");
            addUnicodeGlyphPair(map, 0x043C, "afii10078");
            addUnicodeGlyphPair(map, 0x043D, "afii10079");
            addUnicodeGlyphPair(map, 0x043E, "afii10080");
            addUnicodeGlyphPair(map, 0x043F, "afii10081");
            addUnicodeGlyphPair(map, 0x0440, "afii10082");
            addUnicodeGlyphPair(map, 0x0441, "afii10083");
            addUnicodeGlyphPair(map, 0x0442, "afii10084");
            addUnicodeGlyphPair(map, 0x0443, "afii10085");
            addUnicodeGlyphPair(map, 0x0444, "afii10086");
            addUnicodeGlyphPair(map, 0x0445, "afii10087");
            addUnicodeGlyphPair(map, 0x0446, "afii10088");
            addUnicodeGlyphPair(map, 0x0447, "afii10089");
            addUnicodeGlyphPair(map, 0x0448, "afii10090");
            addUnicodeGlyphPair(map, 0x0449, "afii10091");
            addUnicodeGlyphPair(map, 0x044A, "afii10092");
            addUnicodeGlyphPair(map, 0x044B, "afii10093");
            addUnicodeGlyphPair(map, 0x044C, "afii10094");
            addUnicodeGlyphPair(map, 0x044D, "afii10095");
            addUnicodeGlyphPair(map, 0x044E, "afii10096");
            addUnicodeGlyphPair(map, 0x044F, "afii10097");
            addUnicodeGlyphPair(map, 0x0451, "afii10071");
            addUnicodeGlyphPair(map, 0x0452, "afii10099");
            addUnicodeGlyphPair(map, 0x0453, "afii10100");
            addUnicodeGlyphPair(map, 0x0454, "afii10101");
            addUnicodeGlyphPair(map, 0x0455, "afii10102");
            addUnicodeGlyphPair(map, 0x0456, "afii10103");
            addUnicodeGlyphPair(map, 0x0457, "afii10104");
            addUnicodeGlyphPair(map, 0x0458, "afii10105");
            addUnicodeGlyphPair(map, 0x0459, "afii10106");
            addUnicodeGlyphPair(map, 0x045A, "afii10107");
            addUnicodeGlyphPair(map, 0x045B, "afii10108");
            addUnicodeGlyphPair(map, 0x045C, "afii10109");
            addUnicodeGlyphPair(map, 0x045E, "afii10110");
            addUnicodeGlyphPair(map, 0x045F, "afii10193");
            addUnicodeGlyphPair(map, 0x0462, "afii10146");
            addUnicodeGlyphPair(map, 0x0463, "afii10194");
            addUnicodeGlyphPair(map, 0x0472, "afii10147");
            addUnicodeGlyphPair(map, 0x0473, "afii10195");
            addUnicodeGlyphPair(map, 0x0474, "afii10148");
            addUnicodeGlyphPair(map, 0x0475, "afii10196");
            addUnicodeGlyphPair(map, 0x0490, "afii10050");
            addUnicodeGlyphPair(map, 0x0491, "afii10098");
            addUnicodeGlyphPair(map, 0x04D9, "afii10846");
            addUnicodeGlyphPair(map, 0x05B0, "afii57799");
            addUnicodeGlyphPair(map, 0x05B1, "afii57801");
            addUnicodeGlyphPair(map, 0x05B2, "afii57800");
            addUnicodeGlyphPair(map, 0x05B3, "afii57802");
            addUnicodeGlyphPair(map, 0x05B4, "afii57793");
            addUnicodeGlyphPair(map, 0x05B5, "afii57794");
            addUnicodeGlyphPair(map, 0x05B6, "afii57795");
            addUnicodeGlyphPair(map, 0x05B7, "afii57798");
            addUnicodeGlyphPair(map, 0x05B8, "afii57797");
            addUnicodeGlyphPair(map, 0x05B9, "afii57806");
            addUnicodeGlyphPair(map, 0x05BB, "afii57796");
            addUnicodeGlyphPair(map, 0x05BC, "afii57807");
            addUnicodeGlyphPair(map, 0x05BD, "afii57839");
            addUnicodeGlyphPair(map, 0x05BE, "afii57645");
            addUnicodeGlyphPair(map, 0x05BF, "afii57841");
            addUnicodeGlyphPair(map, 0x05C0, "afii57842");
            addUnicodeGlyphPair(map, 0x05C1, "afii57804");
            addUnicodeGlyphPair(map, 0x05C2, "afii57803");
            addUnicodeGlyphPair(map, 0x05C3, "afii57658");
            addUnicodeGlyphPair(map, 0x05D0, "afii57664");
            addUnicodeGlyphPair(map, 0x05D1, "afii57665");
            addUnicodeGlyphPair(map, 0x05D2, "afii57666");
            addUnicodeGlyphPair(map, 0x05D3, "afii57667");
            addUnicodeGlyphPair(map, 0x05D4, "afii57668");
            addUnicodeGlyphPair(map, 0x05D5, "afii57669");
            addUnicodeGlyphPair(map, 0x05D6, "afii57670");
            addUnicodeGlyphPair(map, 0x05D7, "afii57671");
            addUnicodeGlyphPair(map, 0x05D8, "afii57672");
            addUnicodeGlyphPair(map, 0x05D9, "afii57673");
            addUnicodeGlyphPair(map, 0x05DA, "afii57674");
            addUnicodeGlyphPair(map, 0x05DB, "afii57675");
            addUnicodeGlyphPair(map, 0x05DC, "afii57676");
            addUnicodeGlyphPair(map, 0x05DD, "afii57677");
            addUnicodeGlyphPair(map, 0x05DE, "afii57678");
            addUnicodeGlyphPair(map, 0x05DF, "afii57679");
            addUnicodeGlyphPair(map, 0x05E0, "afii57680");
            addUnicodeGlyphPair(map, 0x05E1, "afii57681");
            addUnicodeGlyphPair(map, 0x05E2, "afii57682");
            addUnicodeGlyphPair(map, 0x05E3, "afii57683");
            addUnicodeGlyphPair(map, 0x05E4, "afii57684");
            addUnicodeGlyphPair(map, 0x05E5, "afii57685");
            addUnicodeGlyphPair(map, 0x05E6, "afii57686");
            addUnicodeGlyphPair(map, 0x05E7, "afii57687");
            addUnicodeGlyphPair(map, 0x05E8, "afii57688");
            addUnicodeGlyphPair(map, 0x05E9, "afii57689");
            addUnicodeGlyphPair(map, 0x05EA, "afii57690");
            addUnicodeGlyphPair(map, 0x05F0, "afii57716");
            addUnicodeGlyphPair(map, 0x05F1, "afii57717");
            addUnicodeGlyphPair(map, 0x05F2, "afii57718");
            addUnicodeGlyphPair(map, 0x060C, "afii57388");
            addUnicodeGlyphPair(map, 0x061B, "afii57403");
            addUnicodeGlyphPair(map, 0x061F, "afii57407");
            addUnicodeGlyphPair(map, 0x0621, "afii57409");
            addUnicodeGlyphPair(map, 0x0622, "afii57410");
            addUnicodeGlyphPair(map, 0x0623, "afii57411");
            addUnicodeGlyphPair(map, 0x0624, "afii57412");
            addUnicodeGlyphPair(map, 0x0625, "afii57413");
            addUnicodeGlyphPair(map, 0x0626, "afii57414");
            addUnicodeGlyphPair(map, 0x0627, "afii57415");
            addUnicodeGlyphPair(map, 0x0628, "afii57416");
            addUnicodeGlyphPair(map, 0x0629, "afii57417");
            addUnicodeGlyphPair(map, 0x062A, "afii57418");
            addUnicodeGlyphPair(map, 0x062B, "afii57419");
            addUnicodeGlyphPair(map, 0x062C, "afii57420");
            addUnicodeGlyphPair(map, 0x062D, "afii57421");
            addUnicodeGlyphPair(map, 0x062E, "afii57422");
            addUnicodeGlyphPair(map, 0x062F, "afii57423");
            addUnicodeGlyphPair(map, 0x0630, "afii57424");
            addUnicodeGlyphPair(map, 0x0631, "afii57425");
            addUnicodeGlyphPair(map, 0x0632, "afii57426");
            addUnicodeGlyphPair(map, 0x0633, "afii57427");
            addUnicodeGlyphPair(map, 0x0634, "afii57428");
            addUnicodeGlyphPair(map, 0x0635, "afii57429");
            addUnicodeGlyphPair(map, 0x0636, "afii57430");
            addUnicodeGlyphPair(map, 0x0637, "afii57431");
            addUnicodeGlyphPair(map, 0x0638, "afii57432");
            addUnicodeGlyphPair(map, 0x0639, "afii57433");
            addUnicodeGlyphPair(map, 0x063A, "afii57434");
            addUnicodeGlyphPair(map, 0x0640, "afii57440");
            addUnicodeGlyphPair(map, 0x0641, "afii57441");
            addUnicodeGlyphPair(map, 0x0642, "afii57442");
            addUnicodeGlyphPair(map, 0x0643, "afii57443");
            addUnicodeGlyphPair(map, 0x0644, "afii57444");
            addUnicodeGlyphPair(map, 0x0645, "afii57445");
            addUnicodeGlyphPair(map, 0x0646, "afii57446");
            addUnicodeGlyphPair(map, 0x0647, "afii57470");
            addUnicodeGlyphPair(map, 0x0648, "afii57448");
            addUnicodeGlyphPair(map, 0x0649, "afii57449");
            addUnicodeGlyphPair(map, 0x064A, "afii57450");
            addUnicodeGlyphPair(map, 0x064B, "afii57451");
            addUnicodeGlyphPair(map, 0x064C, "afii57452");
            addUnicodeGlyphPair(map, 0x064D, "afii57453");
            addUnicodeGlyphPair(map, 0x064E, "afii57454");
            addUnicodeGlyphPair(map, 0x064F, "afii57455");
            addUnicodeGlyphPair(map, 0x0650, "afii57456");
            addUnicodeGlyphPair(map, 0x0651, "afii57457");
            addUnicodeGlyphPair(map, 0x0652, "afii57458");
            addUnicodeGlyphPair(map, 0x0660, "afii57392");
            addUnicodeGlyphPair(map, 0x0661, "afii57393");
            addUnicodeGlyphPair(map, 0x0662, "afii57394");
            addUnicodeGlyphPair(map, 0x0663, "afii57395");
            addUnicodeGlyphPair(map, 0x0664, "afii57396");
            addUnicodeGlyphPair(map, 0x0665, "afii57397");
            addUnicodeGlyphPair(map, 0x0666, "afii57398");
            addUnicodeGlyphPair(map, 0x0667, "afii57399");
            addUnicodeGlyphPair(map, 0x0668, "afii57400");
            addUnicodeGlyphPair(map, 0x0669, "afii57401");
            addUnicodeGlyphPair(map, 0x066A, "afii57381");
            addUnicodeGlyphPair(map, 0x066D, "afii63167");
            addUnicodeGlyphPair(map, 0x0679, "afii57511");
            addUnicodeGlyphPair(map, 0x067E, "afii57506");
            addUnicodeGlyphPair(map, 0x0686, "afii57507");
            addUnicodeGlyphPair(map, 0x0688, "afii57512");
            addUnicodeGlyphPair(map, 0x0691, "afii57513");
            addUnicodeGlyphPair(map, 0x0698, "afii57508");
            addUnicodeGlyphPair(map, 0x06A4, "afii57505");
            addUnicodeGlyphPair(map, 0x06AF, "afii57509");
            addUnicodeGlyphPair(map, 0x06BA, "afii57514");
            addUnicodeGlyphPair(map, 0x06D2, "afii57519");
            addUnicodeGlyphPair(map, 0x06D5, "afii57534");
            addUnicodeGlyphPair(map, 0x1E80, "Wgrave");
            addUnicodeGlyphPair(map, 0x1E81, "wgrave");
            addUnicodeGlyphPair(map, 0x1E82, "Wacute");
            addUnicodeGlyphPair(map, 0x1E83, "wacute");
            addUnicodeGlyphPair(map, 0x1E84, "Wdieresis");
            addUnicodeGlyphPair(map, 0x1E85, "wdieresis");
            addUnicodeGlyphPair(map, 0x1EF2, "Ygrave");
            addUnicodeGlyphPair(map, 0x1EF3, "ygrave");
            addUnicodeGlyphPair(map, 0x200C, "afii61664");
            addUnicodeGlyphPair(map, 0x200D, "afii301");
            addUnicodeGlyphPair(map, 0x200E, "afii299");
            addUnicodeGlyphPair(map, 0x200F, "afii300");
            addUnicodeGlyphPair(map, 0x2012, "figuredash");
            addUnicodeGlyphPair(map, 0x2013, "endash");
            addUnicodeGlyphPair(map, 0x2014, "emdash");
            addUnicodeGlyphPair(map, 0x2015, "afii00208");
            addUnicodeGlyphPair(map, 0x2017, "underscoredbl");
            addUnicodeGlyphPair(map, 0x2018, "quoteleft");
            addUnicodeGlyphPair(map, 0x2019, "quoteright");
            addUnicodeGlyphPair(map, 0x201A, "quotesinglbase");
            addUnicodeGlyphPair(map, 0x201B, "quotereversed");
            addUnicodeGlyphPair(map, 0x201C, "quotedblleft");
            addUnicodeGlyphPair(map, 0x201D, "quotedblright");
            addUnicodeGlyphPair(map, 0x201E, "quotedblbase");
            addUnicodeGlyphPair(map, 0x2020, "dagger");
            addUnicodeGlyphPair(map, 0x2021, "daggerdbl");
            addUnicodeGlyphPair(map, 0x2022, "bullet");
            addUnicodeGlyphPair(map, 0x2024, "onedotenleader");
            addUnicodeGlyphPair(map, 0x2025, "twodotenleader");
            addUnicodeGlyphPair(map, 0x2026, "ellipsis");
            addUnicodeGlyphPair(map, 0x202C, "afii61573");
            addUnicodeGlyphPair(map, 0x202D, "afii61574");
            addUnicodeGlyphPair(map, 0x202E, "afii61575");
            addUnicodeGlyphPair(map, 0x2030, "perthousand");
            addUnicodeGlyphPair(map, 0x2032, "minute");
            addUnicodeGlyphPair(map, 0x2033, "second");
            addUnicodeGlyphPair(map, 0x2039, "guilsinglleft");
            addUnicodeGlyphPair(map, 0x203A, "guilsinglright");
            addUnicodeGlyphPair(map, 0x203C, "exclamdbl");
            addUnicodeGlyphPair(map, 0x2044, "fraction");
            addUnicodeGlyphPair(map, 0x2070, "zerosuperior");
            addUnicodeGlyphPair(map, 0x2074, "foursuperior");
            addUnicodeGlyphPair(map, 0x2075, "fivesuperior");
            addUnicodeGlyphPair(map, 0x2076, "sixsuperior");
            addUnicodeGlyphPair(map, 0x2077, "sevensuperior");
            addUnicodeGlyphPair(map, 0x2078, "eightsuperior");
            addUnicodeGlyphPair(map, 0x2079, "ninesuperior");
            addUnicodeGlyphPair(map, 0x207D, "parenleftsuperior");
            addUnicodeGlyphPair(map, 0x207E, "parenrightsuperior");
            addUnicodeGlyphPair(map, 0x207F, "nsuperior");
            addUnicodeGlyphPair(map, 0x2080, "zeroinferior");
            addUnicodeGlyphPair(map, 0x2081, "oneinferior");
            addUnicodeGlyphPair(map, 0x2082, "twoinferior");
            addUnicodeGlyphPair(map, 0x2083, "threeinferior");
            addUnicodeGlyphPair(map, 0x2084, "fourinferior");
            addUnicodeGlyphPair(map, 0x2085, "fiveinferior");
            addUnicodeGlyphPair(map, 0x2086, "sixinferior");
            addUnicodeGlyphPair(map, 0x2087, "seveninferior");
            addUnicodeGlyphPair(map, 0x2088, "eightinferior");
            addUnicodeGlyphPair(map, 0x2089, "nineinferior");
            addUnicodeGlyphPair(map, 0x208D, "parenleftinferior");
            addUnicodeGlyphPair(map, 0x208E, "parenrightinferior");
            addUnicodeGlyphPair(map, 0x20A1, "colonmonetary");
            addUnicodeGlyphPair(map, 0x20A3, "franc");
            addUnicodeGlyphPair(map, 0x20A4, "lira");
            addUnicodeGlyphPair(map, 0x20A7, "peseta");
            addUnicodeGlyphPair(map, 0x20AA, "afii57636");
            addUnicodeGlyphPair(map, 0x20AB, "dong");
            addUnicodeGlyphPair(map, 0x20AC, "Euro");
            addUnicodeGlyphPair(map, 0x2105, "afii61248");
            addUnicodeGlyphPair(map, 0x2111, "Ifraktur");
            addUnicodeGlyphPair(map, 0x2113, "afii61289");
            addUnicodeGlyphPair(map, 0x2116, "afii61352");
            addUnicodeGlyphPair(map, 0x2118, "weierstrass");
            addUnicodeGlyphPair(map, 0x211C, "Rfraktur");
            addUnicodeGlyphPair(map, 0x211E, "prescription");
            addUnicodeGlyphPair(map, 0x2122, "trademark");
            addUnicodeGlyphPair(map, 0x2126, "Omega");
            addUnicodeGlyphPair(map, 0x212E, "estimated");
            addUnicodeGlyphPair(map, 0x2135, "aleph");
            addUnicodeGlyphPair(map, 0x2153, "onethird");
            addUnicodeGlyphPair(map, 0x2154, "twothirds");
            addUnicodeGlyphPair(map, 0x215B, "oneeighth");
            addUnicodeGlyphPair(map, 0x215C, "threeeighths");
            addUnicodeGlyphPair(map, 0x215D, "fiveeighths");
            addUnicodeGlyphPair(map, 0x215E, "seveneighths");
            addUnicodeGlyphPair(map, 0x2190, "arrowleft");
            addUnicodeGlyphPair(map, 0x2191, "arrowup");
            addUnicodeGlyphPair(map, 0x2192, "arrowright");
            addUnicodeGlyphPair(map, 0x2193, "arrowdown");
            addUnicodeGlyphPair(map, 0x2194, "arrowboth");
            addUnicodeGlyphPair(map, 0x2195, "arrowupdn");
            addUnicodeGlyphPair(map, 0x21A8, "arrowupdnbse");
            addUnicodeGlyphPair(map, 0x21B5, "carriagereturn");
            addUnicodeGlyphPair(map, 0x21D0, "arrowdblleft");
            addUnicodeGlyphPair(map, 0x21D1, "arrowdblup");
            addUnicodeGlyphPair(map, 0x21D2, "arrowdblright");
            addUnicodeGlyphPair(map, 0x21D3, "arrowdbldown");
            addUnicodeGlyphPair(map, 0x21D4, "arrowdblboth");
            addUnicodeGlyphPair(map, 0x2200, "universal");
            addUnicodeGlyphPair(map, 0x2202, "partialdiff");
            addUnicodeGlyphPair(map, 0x2203, "existential");
            addUnicodeGlyphPair(map, 0x2205, "emptyset");
            addUnicodeGlyphPair(map, 0x2206, "Delta");
            addUnicodeGlyphPair(map, 0x2207, "gradient");
            addUnicodeGlyphPair(map, 0x2208, "element");
            addUnicodeGlyphPair(map, 0x2209, "notelement");
            addUnicodeGlyphPair(map, 0x220B, "suchthat");
            addUnicodeGlyphPair(map, 0x220F, "product");
            addUnicodeGlyphPair(map, 0x2211, "summation");
            addUnicodeGlyphPair(map, 0x2212, "minus");
            addUnicodeGlyphPair(map, 0x2215, "fraction");
            addUnicodeGlyphPair(map, 0x2217, "asteriskmath");
            addUnicodeGlyphPair(map, 0x2219, "periodcentered");
            addUnicodeGlyphPair(map, 0x221A, "radical");
            addUnicodeGlyphPair(map, 0x221D, "proportional");
            addUnicodeGlyphPair(map, 0x221E, "infinity");
            addUnicodeGlyphPair(map, 0x221F, "orthogonal");
            addUnicodeGlyphPair(map, 0x2220, "angle");
            addUnicodeGlyphPair(map, 0x2227, "logicaland");
            addUnicodeGlyphPair(map, 0x2228, "logicalor");
            addUnicodeGlyphPair(map, 0x2229, "intersection");
            addUnicodeGlyphPair(map, 0x222A, "union");
            addUnicodeGlyphPair(map, 0x222B, "integral");
            addUnicodeGlyphPair(map, 0x2234, "therefore");
            addUnicodeGlyphPair(map, 0x223C, "similar");
            addUnicodeGlyphPair(map, 0x2245, "congruent");
            addUnicodeGlyphPair(map, 0x2248, "approxequal");
            addUnicodeGlyphPair(map, 0x2260, "notequal");
            addUnicodeGlyphPair(map, 0x2261, "equivalence");
            addUnicodeGlyphPair(map, 0x2264, "lessequal");
            addUnicodeGlyphPair(map, 0x2265, "greaterequal");
            addUnicodeGlyphPair(map, 0x2282, "propersubset");
            addUnicodeGlyphPair(map, 0x2283, "propersuperset");
            addUnicodeGlyphPair(map, 0x2284, "notsubset");
            addUnicodeGlyphPair(map, 0x2286, "reflexsubset");
            addUnicodeGlyphPair(map, 0x2287, "reflexsuperset");
            addUnicodeGlyphPair(map, 0x2295, "circleplus");
            addUnicodeGlyphPair(map, 0x2297, "circlemultiply");
            addUnicodeGlyphPair(map, 0x22A5, "perpendicular");
            addUnicodeGlyphPair(map, 0x22C5, "dotmath");
            addUnicodeGlyphPair(map, 0x2302, "house");
            addUnicodeGlyphPair(map, 0x2310, "revlogicalnot");
            addUnicodeGlyphPair(map, 0x2320, "integraltp");
            addUnicodeGlyphPair(map, 0x2321, "integralbt");
            addUnicodeGlyphPair(map, 0x2329, "angleleft");
            addUnicodeGlyphPair(map, 0x232A, "angleright");
            addUnicodeGlyphPair(map, 0x2500, "SF100000");
            addUnicodeGlyphPair(map, 0x2502, "SF110000");
            addUnicodeGlyphPair(map, 0x250C, "SF010000");
            addUnicodeGlyphPair(map, 0x2510, "SF030000");
            addUnicodeGlyphPair(map, 0x2514, "SF020000");
            addUnicodeGlyphPair(map, 0x2518, "SF040000");
            addUnicodeGlyphPair(map, 0x251C, "SF080000");
            addUnicodeGlyphPair(map, 0x2524, "SF090000");
            addUnicodeGlyphPair(map, 0x252C, "SF060000");
            addUnicodeGlyphPair(map, 0x2534, "SF070000");
            addUnicodeGlyphPair(map, 0x253C, "SF050000");
            addUnicodeGlyphPair(map, 0x2550, "SF430000");
            addUnicodeGlyphPair(map, 0x2551, "SF240000");
            addUnicodeGlyphPair(map, 0x2552, "SF510000");
            addUnicodeGlyphPair(map, 0x2553, "SF520000");
            addUnicodeGlyphPair(map, 0x2554, "SF390000");
            addUnicodeGlyphPair(map, 0x2555, "SF220000");
            addUnicodeGlyphPair(map, 0x2556, "SF210000");
            addUnicodeGlyphPair(map, 0x2557, "SF250000");
            addUnicodeGlyphPair(map, 0x2558, "SF500000");
            addUnicodeGlyphPair(map, 0x2559, "SF490000");
            addUnicodeGlyphPair(map, 0x255A, "SF380000");
            addUnicodeGlyphPair(map, 0x255B, "SF280000");
            addUnicodeGlyphPair(map, 0x255C, "SF270000");
            addUnicodeGlyphPair(map, 0x255D, "SF260000");
            addUnicodeGlyphPair(map, 0x255E, "SF360000");
            addUnicodeGlyphPair(map, 0x255F, "SF370000");
            addUnicodeGlyphPair(map, 0x2560, "SF420000");
            addUnicodeGlyphPair(map, 0x2561, "SF190000");
            addUnicodeGlyphPair(map, 0x2562, "SF200000");
            addUnicodeGlyphPair(map, 0x2563, "SF230000");
            addUnicodeGlyphPair(map, 0x2564, "SF470000");
            addUnicodeGlyphPair(map, 0x2565, "SF480000");
            addUnicodeGlyphPair(map, 0x2566, "SF410000");
            addUnicodeGlyphPair(map, 0x2567, "SF450000");
            addUnicodeGlyphPair(map, 0x2568, "SF460000");
            addUnicodeGlyphPair(map, 0x2569, "SF400000");
            addUnicodeGlyphPair(map, 0x256A, "SF540000");
            addUnicodeGlyphPair(map, 0x256B, "SF530000");
            addUnicodeGlyphPair(map, 0x256C, "SF440000");
            addUnicodeGlyphPair(map, 0x2580, "upblock");
            addUnicodeGlyphPair(map, 0x2584, "dnblock");
            addUnicodeGlyphPair(map, 0x2588, "block");
            addUnicodeGlyphPair(map, 0x258C, "lfblock");
            addUnicodeGlyphPair(map, 0x2590, "rtblock");
            addUnicodeGlyphPair(map, 0x2591, "ltshade");
            addUnicodeGlyphPair(map, 0x2592, "shade");
            addUnicodeGlyphPair(map, 0x2593, "dkshade");
            addUnicodeGlyphPair(map, 0x25A0, "filledbox");
            addUnicodeGlyphPair(map, 0x25A1, "H22073");
            addUnicodeGlyphPair(map, 0x25AA, "H18543");
            addUnicodeGlyphPair(map, 0x25AB, "H18551");
            addUnicodeGlyphPair(map, 0x25AC, "filledrect");
            addUnicodeGlyphPair(map, 0x25B2, "triagup");
            addUnicodeGlyphPair(map, 0x25BA, "triagrt");
            addUnicodeGlyphPair(map, 0x25BC, "triagdn");
            addUnicodeGlyphPair(map, 0x25C4, "triaglf");
            addUnicodeGlyphPair(map, 0x25CA, "lozenge");
            addUnicodeGlyphPair(map, 0x25CB, "circle");
            addUnicodeGlyphPair(map, 0x25CF, "H18533");
            addUnicodeGlyphPair(map, 0x25D8, "invbullet");
            addUnicodeGlyphPair(map, 0x25D9, "invcircle");
            addUnicodeGlyphPair(map, 0x25E6, "openbullet");
            addUnicodeGlyphPair(map, 0x263A, "smileface");
            addUnicodeGlyphPair(map, 0x263B, "invsmileface");
            addUnicodeGlyphPair(map, 0x263C, "sun");
            addUnicodeGlyphPair(map, 0x2640, "female");
            addUnicodeGlyphPair(map, 0x2642, "male");
            addUnicodeGlyphPair(map, 0x2660, "spade");
            addUnicodeGlyphPair(map, 0x2663, "club");
            addUnicodeGlyphPair(map, 0x2665, "heart");
            addUnicodeGlyphPair(map, 0x2666, "diamond");
            addUnicodeGlyphPair(map, 0x266A, "musicalnote");
            addUnicodeGlyphPair(map, 0x266B, "musicalnotedbl");
            addUnicodeGlyphPair(map, 0xF6BE, "dotlessj");
            addUnicodeGlyphPair(map, 0xF6BF, "LL");
            addUnicodeGlyphPair(map, 0xF6C0, "ll");
            addUnicodeGlyphPair(map, 0xF6C1, "Scedilla");
            addUnicodeGlyphPair(map, 0xF6C2, "scedilla");
            addUnicodeGlyphPair(map, 0xF6C3, "commaaccent");
            addUnicodeGlyphPair(map, 0xF6C4, "afii10063");
            addUnicodeGlyphPair(map, 0xF6C5, "afii10064");
            addUnicodeGlyphPair(map, 0xF6C6, "afii10192");
            addUnicodeGlyphPair(map, 0xF6C7, "afii10831");
            addUnicodeGlyphPair(map, 0xF6C8, "afii10832");
            addUnicodeGlyphPair(map, 0xF6C9, "Acute");
            addUnicodeGlyphPair(map, 0xF6CA, "Caron");
            addUnicodeGlyphPair(map, 0xF6CB, "Dieresis");
            addUnicodeGlyphPair(map, 0xF6CC, "DieresisAcute");
            addUnicodeGlyphPair(map, 0xF6CD, "DieresisGrave");
            addUnicodeGlyphPair(map, 0xF6CE, "Grave");
            addUnicodeGlyphPair(map, 0xF6CF, "Hungarumlaut");
            addUnicodeGlyphPair(map, 0xF6D0, "Macron");
            addUnicodeGlyphPair(map, 0xF6D1, "cyrBreve");
            addUnicodeGlyphPair(map, 0xF6D2, "cyrFlex");
            addUnicodeGlyphPair(map, 0xF6D3, "dblGrave");
            addUnicodeGlyphPair(map, 0xF6D4, "cyrbreve");
            addUnicodeGlyphPair(map, 0xF6D5, "cyrflex");
            addUnicodeGlyphPair(map, 0xF6D6, "dblgrave");
            addUnicodeGlyphPair(map, 0xF6D7, "dieresisacute");
            addUnicodeGlyphPair(map, 0xF6D8, "dieresisgrave");
            addUnicodeGlyphPair(map, 0xF6D9, "copyrightserif");
            addUnicodeGlyphPair(map, 0xF6DA, "registerserif");
            addUnicodeGlyphPair(map, 0xF6DB, "trademarkserif");
            addUnicodeGlyphPair(map, 0xF6DC, "onefitted");
            addUnicodeGlyphPair(map, 0xF6DD, "rupiah");
            addUnicodeGlyphPair(map, 0xF6DE, "threequartersemdash");
            addUnicodeGlyphPair(map, 0xF6DF, "centinferior");
            addUnicodeGlyphPair(map, 0xF6E0, "centsuperior");
            addUnicodeGlyphPair(map, 0xF6E1, "commainferior");
            addUnicodeGlyphPair(map, 0xF6E2, "commasuperior");
            addUnicodeGlyphPair(map, 0xF6E3, "dollarinferior");
            addUnicodeGlyphPair(map, 0xF6E4, "dollarsuperior");
            addUnicodeGlyphPair(map, 0xF6E5, "hypheninferior");
            addUnicodeGlyphPair(map, 0xF6E6, "hyphensuperior");
            addUnicodeGlyphPair(map, 0xF6E7, "periodinferior");
            addUnicodeGlyphPair(map, 0xF6E8, "periodsuperior");
            addUnicodeGlyphPair(map, 0xF6E9, "asuperior");
            addUnicodeGlyphPair(map, 0xF6EA, "bsuperior");
            addUnicodeGlyphPair(map, 0xF6EB, "dsuperior");
            addUnicodeGlyphPair(map, 0xF6EC, "esuperior");
            addUnicodeGlyphPair(map, 0xF6ED, "isuperior");
            addUnicodeGlyphPair(map, 0xF6EE, "lsuperior");
            addUnicodeGlyphPair(map, 0xF6EF, "msuperior");
            addUnicodeGlyphPair(map, 0xF6F0, "osuperior");
            addUnicodeGlyphPair(map, 0xF6F1, "rsuperior");
            addUnicodeGlyphPair(map, 0xF6F2, "ssuperior");
            addUnicodeGlyphPair(map, 0xF6F3, "tsuperior");
            addUnicodeGlyphPair(map, 0xF6F4, "Brevesmall");
            addUnicodeGlyphPair(map, 0xF6F5, "Caronsmall");
            addUnicodeGlyphPair(map, 0xF6F6, "Circumflexsmall");
            addUnicodeGlyphPair(map, 0xF6F7, "Dotaccentsmall");
            addUnicodeGlyphPair(map, 0xF6F8, "Hungarumlautsmall");
            addUnicodeGlyphPair(map, 0xF6F9, "Lslashsmall");
            addUnicodeGlyphPair(map, 0xF6FA, "OEsmall");
            addUnicodeGlyphPair(map, 0xF6FB, "Ogoneksmall");
            addUnicodeGlyphPair(map, 0xF6FC, "Ringsmall");
            addUnicodeGlyphPair(map, 0xF6FD, "Scaronsmall");
            addUnicodeGlyphPair(map, 0xF6FE, "Tildesmall");
            addUnicodeGlyphPair(map, 0xF6FF, "Zcaronsmall");
            addUnicodeGlyphPair(map, 0xF721, "exclamsmall");
            addUnicodeGlyphPair(map, 0xF724, "dollaroldstyle");
            addUnicodeGlyphPair(map, 0xF726, "ampersandsmall");
            addUnicodeGlyphPair(map, 0xF730, "zerooldstyle");
            addUnicodeGlyphPair(map, 0xF731, "oneoldstyle");
            addUnicodeGlyphPair(map, 0xF732, "twooldstyle");
            addUnicodeGlyphPair(map, 0xF733, "threeoldstyle");
            addUnicodeGlyphPair(map, 0xF734, "fouroldstyle");
            addUnicodeGlyphPair(map, 0xF735, "fiveoldstyle");
            addUnicodeGlyphPair(map, 0xF736, "sixoldstyle");
            addUnicodeGlyphPair(map, 0xF737, "sevenoldstyle");
            addUnicodeGlyphPair(map, 0xF738, "eightoldstyle");
            addUnicodeGlyphPair(map, 0xF739, "nineoldstyle");
            addUnicodeGlyphPair(map, 0xF73F, "questionsmall");
            addUnicodeGlyphPair(map, 0xF760, "Gravesmall");
            addUnicodeGlyphPair(map, 0xF761, "Asmall");
            addUnicodeGlyphPair(map, 0xF762, "Bsmall");
            addUnicodeGlyphPair(map, 0xF763, "Csmall");
            addUnicodeGlyphPair(map, 0xF764, "Dsmall");
            addUnicodeGlyphPair(map, 0xF765, "Esmall");
            addUnicodeGlyphPair(map, 0xF766, "Fsmall");
            addUnicodeGlyphPair(map, 0xF767, "Gsmall");
            addUnicodeGlyphPair(map, 0xF768, "Hsmall");
            addUnicodeGlyphPair(map, 0xF769, "Ismall");
            addUnicodeGlyphPair(map, 0xF76A, "Jsmall");
            addUnicodeGlyphPair(map, 0xF76B, "Ksmall");
            addUnicodeGlyphPair(map, 0xF76C, "Lsmall");
            addUnicodeGlyphPair(map, 0xF76D, "Msmall");
            addUnicodeGlyphPair(map, 0xF76E, "Nsmall");
            addUnicodeGlyphPair(map, 0xF76F, "Osmall");
            addUnicodeGlyphPair(map, 0xF770, "Psmall");
            addUnicodeGlyphPair(map, 0xF771, "Qsmall");
            addUnicodeGlyphPair(map, 0xF772, "Rsmall");
            addUnicodeGlyphPair(map, 0xF773, "Ssmall");
            addUnicodeGlyphPair(map, 0xF774, "Tsmall");
            addUnicodeGlyphPair(map, 0xF775, "Usmall");
            addUnicodeGlyphPair(map, 0xF776, "Vsmall");
            addUnicodeGlyphPair(map, 0xF777, "Wsmall");
            addUnicodeGlyphPair(map, 0xF778, "Xsmall");
            addUnicodeGlyphPair(map, 0xF779, "Ysmall");
            addUnicodeGlyphPair(map, 0xF77A, "Zsmall");
            addUnicodeGlyphPair(map, 0xF7A1, "exclamdownsmall");
            addUnicodeGlyphPair(map, 0xF7A2, "centoldstyle");
            addUnicodeGlyphPair(map, 0xF7A8, "Dieresissmall");
            addUnicodeGlyphPair(map, 0xF7AF, "Macronsmall");
            addUnicodeGlyphPair(map, 0xF7B4, "Acutesmall");
            addUnicodeGlyphPair(map, 0xF7B8, "Cedillasmall");
            addUnicodeGlyphPair(map, 0xF7BF, "questiondownsmall");
            addUnicodeGlyphPair(map, 0xF7E0, "Agravesmall");
            addUnicodeGlyphPair(map, 0xF7E1, "Aacutesmall");
            addUnicodeGlyphPair(map, 0xF7E2, "Acircumflexsmall");
            addUnicodeGlyphPair(map, 0xF7E3, "Atildesmall");
            addUnicodeGlyphPair(map, 0xF7E4, "Adieresissmall");
            addUnicodeGlyphPair(map, 0xF7E5, "Aringsmall");
            addUnicodeGlyphPair(map, 0xF7E6, "AEsmall");
            addUnicodeGlyphPair(map, 0xF7E7, "Ccedillasmall");
            addUnicodeGlyphPair(map, 0xF7E8, "Egravesmall");
            addUnicodeGlyphPair(map, 0xF7E9, "Eacutesmall");
            addUnicodeGlyphPair(map, 0xF7EA, "Ecircumflexsmall");
            addUnicodeGlyphPair(map, 0xF7EB, "Edieresissmall");
            addUnicodeGlyphPair(map, 0xF7EC, "Igravesmall");
            addUnicodeGlyphPair(map, 0xF7ED, "Iacutesmall");
            addUnicodeGlyphPair(map, 0xF7EE, "Icircumflexsmall");
            addUnicodeGlyphPair(map, 0xF7EF, "Idieresissmall");
            addUnicodeGlyphPair(map, 0xF7F0, "Ethsmall");
            addUnicodeGlyphPair(map, 0xF7F1, "Ntildesmall");
            addUnicodeGlyphPair(map, 0xF7F2, "Ogravesmall");
            addUnicodeGlyphPair(map, 0xF7F3, "Oacutesmall");
            addUnicodeGlyphPair(map, 0xF7F4, "Ocircumflexsmall");
            addUnicodeGlyphPair(map, 0xF7F5, "Otildesmall");
            addUnicodeGlyphPair(map, 0xF7F6, "Odieresissmall");
            addUnicodeGlyphPair(map, 0xF7F8, "Oslashsmall");
            addUnicodeGlyphPair(map, 0xF7F9, "Ugravesmall");
            addUnicodeGlyphPair(map, 0xF7FA, "Uacutesmall");
            addUnicodeGlyphPair(map, 0xF7FB, "Ucircumflexsmall");
            addUnicodeGlyphPair(map, 0xF7FC, "Udieresissmall");
            addUnicodeGlyphPair(map, 0xF7FD, "Yacutesmall");
            addUnicodeGlyphPair(map, 0xF7FE, "Thornsmall");
            addUnicodeGlyphPair(map, 0xF7FF, "Ydieresissmall");
            addUnicodeGlyphPair(map, 0xF8E5, "radicalex");
            addUnicodeGlyphPair(map, 0xF8E6, "arrowvertex");
            addUnicodeGlyphPair(map, 0xF8E7, "arrowhorizex");
            addUnicodeGlyphPair(map, 0xF8E8, "registersans");
            addUnicodeGlyphPair(map, 0xF8E9, "copyrightsans");
            addUnicodeGlyphPair(map, 0xF8EA, "trademarksans");
            addUnicodeGlyphPair(map, 0xF8EB, "parenlefttp");
            addUnicodeGlyphPair(map, 0xF8EC, "parenleftex");
            addUnicodeGlyphPair(map, 0xF8ED, "parenleftbt");
            addUnicodeGlyphPair(map, 0xF8EE, "bracketlefttp");
            addUnicodeGlyphPair(map, 0xF8EF, "bracketleftex");
            addUnicodeGlyphPair(map, 0xF8F0, "bracketleftbt");
            addUnicodeGlyphPair(map, 0xF8F1, "bracelefttp");
            addUnicodeGlyphPair(map, 0xF8F2, "braceleftmid");
            addUnicodeGlyphPair(map, 0xF8F3, "braceleftbt");
            addUnicodeGlyphPair(map, 0xF8F4, "braceex");
            addUnicodeGlyphPair(map, 0xF8F5, "integralex");
            addUnicodeGlyphPair(map, 0xF8F6, "parenrighttp");
            addUnicodeGlyphPair(map, 0xF8F7, "parenrightex");
            addUnicodeGlyphPair(map, 0xF8F8, "parenrightbt");
            addUnicodeGlyphPair(map, 0xF8F9, "bracketrighttp");
            addUnicodeGlyphPair(map, 0xF8FA, "bracketrightex");
            addUnicodeGlyphPair(map, 0xF8FB, "bracketrightbt");
            addUnicodeGlyphPair(map, 0xF8FC, "bracerighttp");
            addUnicodeGlyphPair(map, 0xF8FD, "bracerightmid");
            addUnicodeGlyphPair(map, 0xF8FE, "bracerightbt");
            addUnicodeGlyphPair(map, 0xFB00, "ff");
            addUnicodeGlyphPair(map, 0xFB01, "fi");
            addUnicodeGlyphPair(map, 0xFB02, "fl");
            addUnicodeGlyphPair(map, 0xFB03, "ffi");
            addUnicodeGlyphPair(map, 0xFB04, "ffl");
            addUnicodeGlyphPair(map, 0xFB1F, "afii57705");
            addUnicodeGlyphPair(map, 0xFB2A, "afii57694");
            addUnicodeGlyphPair(map, 0xFB2B, "afii57695");
            addUnicodeGlyphPair(map, 0xFB35, "afii57723");
            addUnicodeGlyphPair(map, 0xFB4B, "afii57700");
            
            HPDF_UNICODE_GLYPH_NAME_MAP = Collections.unmodifiableMap(map);
        }
    }
    
    private static void addUnicodeGlyphPair(Map<String, Character> map, int code, String glyphName) {
        if (map.size() == 0) {
            HPDF_DEFAULT_GLYPH_NAME = glyphName;
        }
        map.put(glyphName, (char) code);
    }
}
