/*
 * JHPdf Free PDF Library : TextDemo.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * original demo/arc_demo.c in 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

import net.sf.jhpdf.*;
import net.sf.jhpdf.pdfobject.*;
import net.sf.jhpdf.pdfobject.HPdfDocument.HPdfCompressionMode;
import net.sf.jhpdf.pdfobject.HPdfPage.HPdfTextRenderingMode;
import net.sf.jhpdf.graphics.*;


public class TextDemo implements DemoEntry {
    public static void main(String[] args) {
        new TextDemo().demo(args);
    }
    
    private static final String F_NAME = "TextDemo.pdf";
    
    @Override
    public void demo(String[] args) {
        final String pageTitle = "Text Demo";
        
        final String sampText = "abcdefgABCDEFG123!#$%&+-@?";
        final String sampText2 = "The quick brown fox jumps over the lazy dog.";
        
        GridSheet grid = new GridSheet();
        
        HPdfDocument pdf = null;
        try {
            pdf = JHPdf.createDocument();
            
            /* set compression mode */
            pdf.setCompressionMode(HPdfCompressionMode.ALL);
            
            /* create default-font */
            HPdfFont font = pdf.getFont("Helvetica", null);
            
            /* add a new page object. */
            HPdfPage page = pdf.addPage();

            /* draw grid to the page */
            grid.printGrid (pdf, page);

            /* print the lines of the page.
            page.setLineWidth(1);
            page.rectangle(50, 50, page.getWidth() - 100,
                        page.getHeight () - 110);
            page.stroke();
            */
            
            /* print the title of the page (with positioning center). */
            page.setFontAndSize(font, 24);
            float tw = page.textWidth(pageTitle);
            page.beginText();
            page.textOut((page.getWidth() - tw) / 2, 
                page.getHeight() - 50, pageTitle);
            page.endText();
            
            page.beginText();
            page.moveTextPos(60, page.getHeight() - 60);
            
            /*
             * font size
             */
            float fsize = 8;
            while (fsize < 60) {
                /* set style and size of font. */
                page.setFontAndSize(font, fsize);
                
                /* set the position of the text. */
                page.moveTextPos(0, -5 - fsize);
                
                /* measure the number of characters which included in the page. */
                int len = page.measureText(sampText, page.getWidth() - 120, false, null);
                
                /* truncate the text. */
                String buf = sampText.substring(0, len);
                
                page.showText(buf);
                
                /* print the description. */
                page.moveTextPos(0, -10);
                page.setFontAndSize(font, 8);
                buf = String.format("FontSize=%.0f", fsize);
                page.showText(buf);
                
                fsize *= 1.5F;
            }
            
            /*
             * font color
             */
            page.setFontAndSize(font, 8);
            page.moveTextPos(0, -30);
            page.showText("Font color");
            
            page.setFontAndSize(font, 18);
            page.moveTextPos(0, -20);
            int len = sampText.length();
            for (int i = 0; i < len; ++i) {
                float r = (float) i / (float) len;
                float g = 1 - r;
                String buf = sampText.substring(i, i + 1);
                
                page.setRGBFill(r, g, 0.0F);
                page.showText(buf);
            }
            page.moveTextPos(0, -25);
            
            for (int i = 0; i < len; ++i) {
                float r = (float) i / (float) len;
                float b = 1 - r;
                String buf = sampText.substring(i, i + 1);
                
                page.setRGBFill(r, 0.0F, b);
                page.showText(buf);
            }
            page.moveTextPos(0, -25);
            
            for (int i = 0; i < len; ++i) {
                float b = (float) i / (float) len;
                float g = 1 - b;
                String buf = sampText.substring(i, i + 1);
                
                page.setRGBFill(0.0F, g, b);
                page.showText(buf);
            }
            page.endText();
            
            float ypos = 450;
            
            /*
             * Font rendering mode
             */
            page.setFontAndSize(font, 32);
            page.setRGBFill(0.5F, 0.5F, 0.0F);
            page.setLineWidth(1.5F);
            
            /* PDF_FILL */
            showDescription (page,  60, ypos,
                        "RenderingMode=PDF_FILL");
            page.setTextRenderingMode(HPdfTextRenderingMode.FILL);
            page.beginText();
            page.textOut(60, ypos, "ABCabc123");
            page.endText();
            
            /* PDF_STROKE */
            showDescription (page,  60, ypos - 50,
                        "RenderingMode=PDF_STROKE");
            page.setTextRenderingMode(HPdfTextRenderingMode.STROKE);
            page.beginText();
            page.textOut(60, ypos - 50, "ABCabc123");
            page.endText();
            
            /* PDF_FILL_THEN_STROKE */
            showDescription (page,  60, ypos - 100,
                        "RenderingMode=PDF_FILL_THEN_STROKE");
            page.setTextRenderingMode(HPdfTextRenderingMode.FILL_THEN_STROKE);
            page.beginText();
            page.textOut(60, ypos - 100, "ABCabc123");
            page.endText();
            
            /* PDF_FILL_CLIPPING */
            showDescription (page,  60, ypos - 150,
                        "RenderingMode=PDF_FILL_CLIPPING");
            page.gSave();
            page.setTextRenderingMode(HPdfTextRenderingMode.FILL_CLIPPING);
            page.beginText();
            page.textOut(60, ypos - 150, "ABCabc123");
            page.endText();
            showStripePattern(page, 60, ypos - 150);
            page.gRestore();
            
            /* PDF_STROKE_CLIPPING */
            showDescription (page,  60, ypos - 200,
                        "RenderingMode=PDF_STROKE_CLIPPING");
            page.gSave();
            page.setTextRenderingMode(HPdfTextRenderingMode.STROKE_CLIPPING);
            page.beginText();
            page.textOut(60, ypos - 200, "ABCabc123");
            page.endText();
            showStripePattern(page, 60, ypos - 200);
            page.gRestore();
            
            /* PDF_FILL_STROKE_CLIPPING */
            showDescription (page,  60, ypos - 250,
                        "RenderingMode=PDF_FILL_STROKE_CLIPPING");
            page.gSave();
            page.setTextRenderingMode(HPdfTextRenderingMode.FILL_STROKE_CLIPPING);
            page.beginText();
            page.textOut(60, ypos - 250, "ABCabc123");
            page.endText();
            showStripePattern(page, 60, ypos - 250);
            page.gRestore();
            
            /* Reset text attributes */
            page.setTextRenderingMode(HPdfTextRenderingMode.FILL);
            page.setRGBFill(0, 0, 0);
            page.setFontAndSize(font, 30);
            
            float angle1;
            float angle2;
            float rad1;
            float rad2;
            /*
             * Rotating text
             */
            angle1 = 30;   /* A rotation of 30 degrees. */
            rad1 = angle1 / 180 * 3.141592F; /* Calcurate the radian value. */
            
            showDescription(page, 320, ypos - 60, "Rotating text");
            page.beginText();
            page.setTextMatrix((float) Math.cos(rad1), (float) Math.sin(rad1), (float) -Math.sin(rad1), (float) Math.cos(rad1),
                330, ypos - 60);
            page.showText("ABCabc123");
            page.endText();
            
            /*
             * Skewing text.
             */
            showDescription(page, 320, ypos - 120, "skew text");
            page.beginText();
            
            angle1 = 10;
            angle2 = 20;
            rad1 = angle1 / 180 * 3.141592F;
            rad2 = angle2 / 180 * 3.141592F;
            
            page.setTextMatrix(1, (float) Math.tan(rad1), (float) Math.tan(rad2), 1, 320, ypos - 120);
            page.showText("ABCabc123");
            page.endText();
            
            /*
             * scaling text (X direction)
             */
            showDescription(page, 320, ypos - 175, "Scaling text (X direction)");
            page.beginText();
            page.setTextMatrix(1.5F, 0, 0, 1, 320, ypos - 175);
            page.showText("ABCabc12");
            page.endText();
            
            /*
             * scaling text (Y direction)
             */
            showDescription(page, 320, ypos - 250, "Scaling text (Y direction)");
            page.beginText();
            page.setTextMatrix(1, 0, 0, 2, 320, ypos - 250);
            page.showText("ABCabc123");
            page.endText();
            
            /*
             * char spacing, word spacing
             */
            showDescription(page, 60, 140, "char-spacing 0");
            showDescription(page, 60, 100, "char-spacing 1.5");
            showDescription(page, 60, 60, "char-spacing 1.5, word-spacing 2.5");
            
            page.setFontAndSize(font, 20);
            page.setRGBFill(0.1F, 0.3F, 0.1F);
            
            /* char-spacing 0 */
            page.beginText();
            page.textOut(60, 140, sampText2);
            page.endText();
            
            /* char-spacing 1.5 */
            page.setCharSpace(1.5F);
            
            page.beginText();
            page.textOut(60, 100, sampText2);
            page.endText();
            
            /* char-spacing 1.5, word-spacing 3.5 */
            page.setWordSpace(2.5F);
            
            page.beginText();
            page.textOut(60, 60, sampText2);
            page.endText();
            
            /* save the document to a file */
            pdf.saveToFile(F_NAME);
            
        } catch (HPdfException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (pdf != null) {
                pdf.freeDocAll();
            }
        }
    }
    
    private void showStripePattern(HPdfPage page, float x, float y) {
        int iy = 0;
        
        while (iy < 50) {
            page.setRGBStroke(0.0F, 0.0F, 0.5F);
            page.setLineWidth(1);
            page.moveTo(x, y + iy);
            page.lineTo(x + page.textWidth("ABCabc123"),
                y + iy);
            page.stroke();
            iy += 3;
        }
        
        page.setLineWidth(2.5F);
    }
    
    private void showDescription(HPdfPage page, float x, float y, String text) {
        float fsize = page.getCurrentFontSize();
        HPdfFont font = page.getCurrentFont();
        HPdfRGBColor c = page.getRGBFill();
        
        page.beginText();
        page.setRGBFill(0, 0, 0);
        page.setTextRenderingMode(HPdfTextRenderingMode.FILL);
        page.setFontAndSize(font, 10);
        page.textOut(x, y - 12, text);
        page.endText();
        
        page.setFontAndSize(font, fsize);
        page.setRGBFill(c.getR(), c.getG(), c.getB());
    }
}
