/*
 * JHPdf Free PDF Library : EncodingList.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * original demo/character_map.c in 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

import java.io.*;

import net.sf.jhpdf.*;
import net.sf.jhpdf.pdfobject.*;
import net.sf.jhpdf.pdfobject.HPdfCatalog.HPdfPageMode;
import net.sf.jhpdf.pdfobject.HPdfDocument.HPdfCompressionMode;
import net.sf.jhpdf.encoder.HPdfEncoder;


public class EncodingList implements DemoEntry {
    public static void main(String[] args) {
        new EncodingList().demo(args);
    }
    
    private static final String F_NAME = "EncodingList.pdf";
    
    private static final int PAGE_WIDTH = 420;
    private static final int PAGE_HEIGHT = 400;
    private static final int CELL_WIDTH = 20;
    private static final int CELL_HEIGHT = 20;
    //private static final int CELL_HEADER = 10;
    
    @Override
    public void demo(String[] args) {
        
        final String[] encodings = {
            "StandardEncoding",
            "MacRomanEncoding",
            "WinAnsiEncoding",
            "ISO8859-2",
            "ISO8859-3",
            "ISO8859-4",
            "ISO8859-5",
            "ISO8859-9",
            "ISO8859-10",
            "ISO8859-13",
            "ISO8859-14",
            "ISO8859-15",
            "ISO8859-16",
            "CP1250",
            "CP1251",
            "CP1252",
            "CP1254",
            "CP1257",
            "KOI8-R",
            "Symbol-Set",
            "ZapfDingbats-Set"
        };
        
        HPdfDocument pdf = null;
        try {
            pdf = JHPdf.createDocument();
            
            /* set compression mode */
            pdf.setCompressionMode(HPdfCompressionMode.ALL);
            
            /* Set page mode to use outlines. */
            pdf.setPageMode(HPdfPageMode.USE_OUTLINE);
            
            /* get default font */
            HPdfFont font = pdf.getFont("Helvetica", null);
            
            String fontName = pdf.loadType1FontFromFile(
                "type1" + File.separator + "a010013l.afm",
                "type1" + File.separator + "a010013l.pfb");
            
            HPdfOutline root = pdf.createOutline(null, "Encoding list", null);
            root.setOpened(true);
            
            for (final String encoding : encodings) {
                HPdfPage page = pdf.addPage();
                
                page.setWidth(PAGE_WIDTH);
                page.setHeight(PAGE_HEIGHT);
                
                HPdfOutline outline = pdf.createOutline(root, encoding, null);
                HPdfDestination dst = page.createDestination();
                dst.setXYZ(0, page.getHeight(), 1);
                /* dst.setFitB(); */ // MEMO: original code /* HPDF_Destination_SetFitB(dst); */
                outline.setDestination(dst);
                
                page.setFontAndSize(font, 15);
                drawGraph(page);
                
                page.beginText();
                page.setFontAndSize(font, 20);
                page.moveTextPos(40, PAGE_HEIGHT - 50);
                page.showText(encoding);
                page.showText(" Encoding");
                page.endText();
                
                HPdfFont font2;
                if (encoding.equals("Symbol-Set")) {
                    font2 = pdf.getFont("Symbol", null);
                } else if (encoding.equals("ZapfDingbats-Set")) {
                    font2 = pdf.getFont("ZapfDingbats", null);
                } else {
                    font2 = pdf.getFont(fontName, encoding);
                }
                
                page.setFontAndSize(font2, 14);
                drawFonts(page);
            }
            
            /* save the document to a file */
            pdf.saveToFile(F_NAME);
            
        } catch (HPdfException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (pdf != null) {
                pdf.freeDocAll();
            }
        }
    }
    
    private void drawGraph(HPdfPage page) {
        
        /* Draw 16 X 15 cells */
        
        /* Draw vertical lines. */
        page.setLineWidth(0.5F);
        
        for (int i = 0; i <= 17; ++i) {
            int x = i * CELL_WIDTH + 40;
            
            page.moveTo(x, PAGE_HEIGHT - 60);
            page.lineTo(x, 40);
            page.stroke();
            
            if (i > 0 && i <= 16) {
                page.beginText();
                page.moveTextPos(x + 5, PAGE_HEIGHT - 75);
                page.showText(String.format("%X", i - 1));
                page.endText();
            }
        }
        
        /* Draw horizontal lines. */
        for (int i = 0; i <= 15; ++i) {
            int y = i * CELL_HEIGHT + 40;
            
            page.moveTo(40, y);
            page.lineTo(PAGE_WIDTH - 40, y);
            page.stroke();
            
            if (i < 14) {
                page.beginText();
                page.moveTextPos(45, y + 5);
                page.showText(String.format("%X", 15 - i));
                page.endText();
            }
        }
    }
    
    private void drawFonts(HPdfPage page) {
        
        page.beginText();
        // memo: HPdfFont#getEncoder() is JHPdf specific method.
        HPdfEncoder encoder = page.getCurrentFont().getEncoder();
        
        /* Draw all character from 0x20 to 0xFF to the canvas. */
        for (int i = 1; i < 17; ++i) {
            for (int j = 1; j < 17; ++i) {
                byte[] bbuf = new byte[1];
                
                bbuf[0] = (byte)((i - 1) * 16 + (j - 1));
                if (bbuf[0] >= 32) {
                    int y = PAGE_HEIGHT - 55 - ((i - 1) * CELL_HEIGHT);
                    int x = j * CELL_WIDTH + 50;
                    // memo: HPdfEncoder#decode() is JHPdf specific method.
                    String buf = encoder.decode(bbuf);
                    
                    float d = x - page.textWidth(buf) / 2;
                    page.textOut(d, y, buf);
                }
            }
        }
        
        page.endText();
    }
}
