/*
 * JHPdf Free PDF Library : HPdfFontDef.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.pdfobject.font;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sf.jhpdf.HPdfException;
import net.sf.jhpdf.graphics.HPdfRect;
import net.sf.jhpdf.io.HPdfMemStream;
import net.sf.jhpdf.pdfobject.HPdfDict;

/**
 * Abstract class represents PDF Font definition.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public abstract class HPdfFontDef {
    // TODO only stub, needs methods definition.
    private static final Logger logger = LoggerFactory.getLogger(HPdfFontDef.class);
    
    private static final long HPDF_FONTDEF_SIG_BYTES = 0x464F4E54L;
    
    public enum HPdfFontDefType {
        TYPE1,
        TRUETYPE,
        CID,
        UNINITIALIZED,
    }
    
    /*------ collection of flags for defining characteristics. ---*/
    protected static final int HPDF_FONT_FIXED_WIDTH    = 1;
    protected static final int HPDF_FONT_SERIF          = 2;
    protected static final int HPDF_FONT_SYMBOLIC       = 4;
    protected static final int HPDF_FONT_SCRIPT         = 8;
      /* Reserved                                         16 */
    protected static final int HPDF_FONT_STD_CHARSET    = 32;
    protected static final int HPDF_FONT_ITALIC         = 64;
      /* Reserved                                         128
         Reserved                                         256
         Reserved                                         512
         Reserved                                         1024
         Reserved                                         2048
         Reserved                                         4096
         Reserved                                         8192
         Reserved                                         16384
         Reserved                                         32768 */
    protected static final int HPDF_FONT_ALL_CAP        = 65536;
    protected static final int HPDF_FONT_SMALL_CAP      = 131072;
    protected static final int HPDF_FONT_FOURCE_BOLD    = 262144;
    
    public final class HPdfCharData implements Cloneable {
        private int charCd;
        public int getCharCd() {
            return this.charCd;
        }
        public void setCharCd(int charCd) {
            this.charCd = charCd;
        }
        private char unicode;
        public char getUnicode() {
            return this.unicode;
        }
        public void setUnicode(char unicode) {
            this.unicode = unicode;
        }
        private int width;
        public int getWidth() {
            return this.width;
        }
        public void setWidth(int width) {
            this.width = width;
        }
        
        private void copy(HPdfCharData org) {
            this.setCharCd(org.getCharCd());
            this.setUnicode(org.getUnicode());
            this.setWidth(org.getWidth());
        }

        @Override
        public HPdfCharData clone()  {
            HPdfCharData o;
            try {
                o = (HPdfCharData) super.clone();
            } catch (CloneNotSupportedException e) {
                throw new RuntimeException("HPdfCharData cloning failed", e);
            }
            o.copy(this);
            return o;
        }
        
    }
    
    protected HPdfFontDef() {
        this.sigBytes = HPDF_FONTDEF_SIG_BYTES;
    }
    
    private long sigBytes;
    
    protected final long getSigBytes() {
        return this.sigBytes;
    }
    
    private String baseFont;
    
    public final String getBaseFont() {
        return this.baseFont;
    }
    
    protected final void setBaseFont(String baseFont) {
        this.baseFont = baseFont;
    }
    
    private HPdfFontDefType type = HPdfFontDefType.UNINITIALIZED;
    
    public final HPdfFontDefType getFontDefType() {
        return this.type;
    }
    
    protected final void setFontDefType(HPdfFontDefType type) {
        this.type = type;
    }
    
    private int ascent;
    
    public final int getAscent() {
        return this.ascent;
    }
    
    protected final void setAscent(int ascent) {
        this.ascent = ascent;
    }
    
    private int descent;
    
    public final int getDescent() {
        return this.descent;
    }
    
    protected final void setDescent(int descent) {
        this.descent = descent;
    }
    
    private int flags;
    
    protected final int getFlags() {
        return this.flags;
    }
    
    /**
     * set fontdef flag. this adds flagVal on current flag setting.
     * @param flagVal flag value to set.
     */
    protected final void setFlag(int flagVal) {
        this.flags |= flagVal;
    }
    
    /**
     * reset fontdef flag. only flagVal flag will be reseted.
     * @param flagVal flag value to reset.
     */
    protected final void resetFlag(int flagVal) {
        this.flags &= ~flagVal;
    }
    
    /**
     * reset all fontdef flag.
     */
    protected final void clearFlag() {
        this.flags = 0;
    }
    
    private HPdfRect fontBbox = new HPdfRect();
    
    public final HPdfRect getFontBbox() {
        return this.fontBbox;
    }
    
    protected final void setFontBbox(HPdfRect fontBbox) {
        this.fontBbox = fontBbox.clone();
    }
    
    private int italicAngle;
    
    protected final int getItalicAngle() {
        return this.italicAngle;
    }
    
    protected final void setItalicAngle(int italicAngle) {
        this.italicAngle = italicAngle;
    }
    
    private int stemv;
    
    protected final int getStemv() {
        return this.stemv;
    }
    
    protected final void setStemv(int stemv) {
        this.stemv = stemv;
    }
    
    private int avgWidth;
    
    protected final int getAvgWidth() {
        return this.avgWidth;
    }
    
    protected final void setAvgWidth(int avgWidth) {
        this.avgWidth = avgWidth;
    }
    
    private int maxWidth;
    
    protected final int getMaxWidth() {
        return this.maxWidth;
    }
    
    protected final void setMaxWidth(int maxWidth) {
        this.maxWidth = maxWidth;
    }
    
    private int missingWidth;
    
    public final int getMissingWidth() {
        return this.missingWidth;
    }
    
    protected final void setMissingWidth(int missingWidth) {
        this.missingWidth = missingWidth;
    }
    
    private int stemh;
    
    protected final int getStemh() {
        return this.stemh;
    }
    
    protected final void setStemh(int stemh) {
        this.stemh = stemh;
    }
    
    private int xHeight;
    
    public final int getXHeight() {
        return this.xHeight;
    }
    
    protected final void setXHeight(int xHeight) {
        this.xHeight = xHeight;
    }
    
    private int capHeight;
    
    public final int getCapHeight() {
        return this.capHeight;
    }
    
    protected final void setCapHeight(int capHeight) {
        this.capHeight = capHeight;
    }
    
    /*  the initial value of descriptor entry is NULL.
     *  when first font-object besed on the fontdef object is created,
     *  the font-descriptor object is created and descriptor entry is set.
     */
    private HPdfDict descriptor;
    
    protected final HPdfDict getDescriptor() {
        return this.descriptor;
    }
    
    protected final void setDescriptor(HPdfDict descriptor) {
        this.descriptor = descriptor;
    }
    // TODO Stream I/O not clear... probably it's MemStream
    private HPdfMemStream data;
    
    protected final HPdfMemStream getData() {
        return this.data;
    }
    
    protected final void setData(HPdfMemStream data) {
        this.data = data;
    }
    
    private boolean valid;
    
    protected final boolean isValid() {
        return this.valid;
    }
    
    protected final void setValid(boolean valid) {
        this.valid = valid;
    }
    
    public final void cleanup() {
        logger.trace("HPdfFontDef#cleanup");
        
        this.clean();
        
        this.setDescriptor(null);
    }
    
    public final boolean validate() {
        logger.trace("HPdfFontDef#validate");
        
        return this.getSigBytes() == HPDF_FONTDEF_SIG_BYTES;
    }
    
    public abstract void init();
    
    public abstract void clean();
    
    public abstract void dispose();

    public abstract int getWidth(char code);
}
