/*
 * JHPdf Free PDF Library : HPdfMemStream.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.io;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.EnumSet;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sf.jhpdf.HPdfConst;
import net.sf.jhpdf.HPdfErrorCode;
import net.sf.jhpdf.encrypt.HPdfEncrypt;

/**
 * In-memory stream class for reading/writing.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public class HPdfMemStream implements HPdfReadStream, HPdfWriteStream {
    
    private static final Logger logger = LoggerFactory.getLogger(HPdfMemStream.class);
    
    private byte[] innerBuffer;
    
    private int currentReadPos;
    
    private int currentWritePos;
    
    /**
     * proxy class for reading.
     *
     */
    private class MemReadStream extends HPdfAbstractReadStream {
        
        private MemReadStream() {
            super();
        }

        @Override
        public int read(byte[] buf, int bufOffset, int siz) {
            logger.trace("HPdfMemStream#read");
            if (currentReadPos >= currentWritePos) {
                return -1;
            }
            if (bufOffset < 0) {
                throw new HPdfIoException(HPdfErrorCode.HPDF_FILE_IO_ERROR, 0,
                        new IndexOutOfBoundsException("bufOffset < 0"));
            } else if (siz < 0) {
                throw new HPdfIoException(HPdfErrorCode.HPDF_FILE_IO_ERROR, 0,
                        new IndexOutOfBoundsException("siz < 0"));
            }
            int actualSize = siz;
            if (currentReadPos + siz >= currentWritePos) {
                actualSize = currentWritePos - currentReadPos;
            }
            if (actualSize > buf.length - bufOffset) {
                throw new HPdfIoException(HPdfErrorCode.HPDF_FILE_IO_ERROR, 0,
                        new IndexOutOfBoundsException(
                                String.format("siz too big, available %d", buf.length - bufOffset)));
            }
            System.arraycopy(innerBuffer, currentReadPos, buf, bufOffset, actualSize);
            currentReadPos += actualSize;
            return actualSize;
        }

        @Override
        protected int readByte() {
            logger.trace("HPdfMemStream#read");
            if (currentReadPos >= innerBuffer.length) {
                return -1;
            } else {
                return 0xFF & innerBuffer[currentReadPos++];
            }
        }

        @Override
        public void seek(long offset, SeekMode mode) {
            logger.trace("HPdfMemStream#seek");
            currentReadPos = (int) this.offsetToAbsolutePos(offset, mode);
        }

        @Override
        public long tell() {
            logger.trace("HPdfMemStream#tell");
            return currentReadPos;
        }

        @Override
        public long getSize() {
            // MEMO: Use WriteStream's getSize().
            return writeDelegate.getSize();
        }

        @Override
        @Deprecated
        public void close() {
            // MEMO: NOP. Actual closing process is defined in Outer class.
        }
    }
    
    private MemReadStream readDelegate = new MemReadStream();
    
    /**
     * proxy class for writing.
     *
     */
    private class MemWriteStream extends HPdfAbstractWriteStream {
        
        private MemWriteStream() {
            super();
        }
        
        @Override
        public void write(byte[] buf, int bufOffset, int siz) {
            logger.trace("HPdfMemStream#write");
            if (bufOffset + siz > buf.length){
                siz = buf.length - bufOffset;
            }
            if (currentWritePos + siz > innerBuffer.length) {
                int tmpSize = innerBuffer.length * 2;
                if (tmpSize < currentWritePos + siz) {
                    tmpSize = currentWritePos + siz;
                }
                innerBuffer = Arrays.copyOf(innerBuffer, tmpSize);
            }
            System.arraycopy(buf, bufOffset, innerBuffer, currentWritePos, siz);
            currentWritePos += siz;
        }
    
        @Override
        @Deprecated
        public void close() {
            // MEMO: NOP. Actual closing process is defined in Outer class.
        }
        
        @Override
        public long getSize() {
            logger.trace("HPdfMemStream#getSize");
            return currentWritePos;
        }
    }
        
    private MemWriteStream writeDelegate = new MemWriteStream();

    /**
     * constructor.
     * @param bufSiz size of internal buffer
     */
    public HPdfMemStream(int bufSiz) {
        super();
        
        logger.trace("HPdfMemStream#ctor");
        
        this.innerBuffer = new byte[bufSiz > 0 ? bufSiz : HPdfConst.HPDF_STREAM_BUF_SIZ];
        this.currentReadPos = 0;
        this.currentWritePos = 0;
    }
    
    private void checkStream() {
        if (this.innerBuffer == null) {
            throw new HPdfIoException(HPdfErrorCode.HPDF_FILE_IO_ERROR, 0,
                    new IOException("MemStream is null."));
        }
    }

    @Override
    public boolean validate() {
        return readDelegate.validate();
    }

    @Override
    public void close() {
        this.innerBuffer = null;
    }
    
    @Override
    public long getSize() {
        checkStream();
        return writeDelegate.getSize();
    }
    
    private int getBufSize() {
        checkStream();
        return this.innerBuffer.length;
    }

    @Override
    public int read(byte[] buf, int siz){
        checkStream();
        return readDelegate.read(buf, siz);
    }

    @Override
    public int read(byte[] buf, int bufOffset, int siz) {
        checkStream();
        return readDelegate.read(buf, bufOffset, siz);
    }

    @Override
    public Charset getCharsetForReadLn() {
        checkStream();
        return readDelegate.getCharsetForReadLn();
    }

    @Override
    public String readLn() {
        checkStream();
        return readDelegate.readLn();
    }

    @Override
    public void seek(long offset, SeekMode mode) {
        checkStream();
        readDelegate.seek(offset, mode);
    }

    @Override
    public long tell(){
        checkStream();
        return readDelegate.tell();
    }

    @Override
    public void writeToStream(HPdfWriteStream dst, EnumSet<HPdfFilterFlag> filter, HPdfEncrypt e) {
        checkStream();
        readDelegate.writeToStream(dst, filter, e);
    }

    @Override
    public void writeToStream(HPdfWriteStream dst, HPdfFilterFlag filter, HPdfEncrypt e) {
        checkStream();
        readDelegate.writeToStream(dst, filter, e);
    }
    
    @Override
    public void write(byte[] buf, int siz) {
        checkStream();
        writeDelegate.write(buf, siz);
    }

    @Override
    public void write(byte[] buf, int bufOffset, int siz) {
        checkStream();
        writeDelegate.write(buf, bufOffset, siz);
    }

    @Override
    public void writeAsBytes(String value) {
        checkStream();
        writeDelegate.writeAsBytes(value);
    }

    @Override
    public void writeStr(String value) {
        checkStream();
        writeDelegate.writeStr(value);
    }

    @Override
    public void writeInt(long value) {
        checkStream();
        writeDelegate.writeInt(value);
    }

    @Override
    public void writeUInt(long value) {
        checkStream();
        writeDelegate.writeUInt(value);
    }
    
    @Override
    public void writeReal(float value) {
        writeDelegate.writeReal(value);
    }

    @Override
    public void writeEscapeName(String value) {
        checkStream();
        writeDelegate.writeEscapeName(value);
    }

    @Override
    public void writeChar(char value) {
        checkStream();
        writeDelegate.writeChar(value);
    }

    @Override
    public void writeUChar(char value) {
        checkStream();
        writeDelegate.writeUChar(value);
    }
    
    @Override
    public void writeBinary(byte[] value, long len, HPdfEncrypt e) {
        checkStream();
        writeDelegate.writeBinary(value, len, e);
    }

    @Override
    public void writeEscapeText(String text) {
        checkStream();
        writeDelegate.writeEscapeText(text);        
    }

    @Override
    public void writeEscapeText2(String text, int len) {
        checkStream();
        writeDelegate.writeEscapeText2(text, len);        
    }
    
    /**
     * do rewrite() with zero offset.
     * @param buf values to be overwritten.
     * @param siz number of values to be overwritten.
     */
    public void rewrite(byte[] buf, int siz) {
        this.rewrite(buf, 0, siz);
    }
    
    /**
     * overwrites stream with given values.
     * overwriting begins at position that tell() returns.<br>
     * This method moves position that tell() returns by siz.<br>
     * if new values overrun existing stream, stream will be expanded
     * (<strong>so following write() begins after written values</strong>).
     * @param buf values to be overwritten.
     * @param bufOffset offset for buf.
     * @param siz number of values to be overwritten.
     */
    public void rewrite(byte[] buf, int bufOffset, int siz) {
        logger.trace("HPdfMemStream#rewrite");
        checkStream();
        if (this.currentReadPos + siz >= this.getBufSize()) {
            // set write pointer to rewriting position.
            this.currentWritePos = this.currentReadPos;
            // write() expands inner buffer and moves currentWritePos to proper position.
            this.write(buf, bufOffset, siz);
            this.currentReadPos = this.currentWritePos;
        } else {
            System.arraycopy(buf, bufOffset, innerBuffer, this.currentReadPos, siz);
            this.currentReadPos += siz;
        }
    }
}
