/*
 * JHPdf Free PDF Library : HPdfBasicEncoder.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.encoder;

import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sf.jhpdf.HPdfErrorCode;
import net.sf.jhpdf.HPdfException;
import net.sf.jhpdf.encoder.HPdfEncoder.HPdfByteType;
import net.sf.jhpdf.io.HPdfWriteStream;
import net.sf.jhpdf.pdfobject.HPdfArray;
import net.sf.jhpdf.pdfobject.HPdfFont.HPdfWritingMode;

/**
 * Class represents PDF Basic encoder.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public class HPdfBasicEncoder extends HPdfEncoder {
    // TODO only stub, needs methods definition.

    private static final Logger logger = LoggerFactory.getLogger(HPdfBasicEncoder.class);
    
    private enum HPdfBaseEncodings {
        STANDARD(HPDF_ENCODING_STANDARD),
        WIN_ANSI(HPDF_ENCODING_WIN_ANSI),
        MAC_ROMAN(HPDF_ENCODING_MAC_ROMAN),
        FONT_SPECIFIC(HPDF_ENCODING_FONT_SPECIFIC),
        ;
        private String encodingName;
        private HPdfBaseEncodings(String encodingName) {
            this.encodingName = encodingName;
        }
        private String getEncodingName() {
            return this.encodingName;
        }
    }
    
    private static class HPdfBuiltinEncodingData {
        private String encodingName;
        private HPdfBaseEncodings baseEncoding;
        private char[] overrideMap;
        
        private HPdfBuiltinEncodingData(
                String encodingName, HPdfBaseEncodings baseEncoding, char[] overrideMap) {
            this.encodingName = encodingName;
            this.baseEncoding = baseEncoding;
            this.overrideMap = overrideMap;
        }
    }
    
    private static Map<String, HPdfBuiltinEncodingData> HPDF_BUILTIN_ENCODINGS;
    
    private static HPdfBuiltinEncodingData findBuiltinData(String encodingName) {
        if (HPDF_BUILTIN_ENCODINGS == null) {
            initBuiltinEncodings();
        }
        return HPDF_BUILTIN_ENCODINGS.get(encodingName);
    }
    
    private static final char FIRST_CHAR = (char) 32;
    private static final char LAST_CHAR = (char) 255;

    public HPdfBasicEncoder(String encodingName) {
        super();
        logger.trace("HPdfBasicEncoder#ctor");
        
        HPdfBuiltinEncodingData data = findBuiltinData(encodingName);
        if (data == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_ENCODING_NAME, 0);
        }
        
        this.setName(data.encodingName);
        this.setType(HPdfEncoderType.SINGLE_BYTE);
        
        // attr initialize
        this.setFirstChar(FIRST_CHAR);
        this.setLastChar(LAST_CHAR);
        this.setHasDifferences(false);
        
        this.setBaseEncoding(data.baseEncoding.getEncodingName());
        
        // TODO: if 'map' is required only for converting byte-to-Unicode,
        // 'map' may be not needed.
    }
    
    private String baseEncoding;
    
    public String getBaseEncoding() {
        return this.baseEncoding;
    }
    
    public void setBaseEncoding(String encoding) {
        this.baseEncoding = encoding;
    }

    private char firstChar;

    /**
     * @return firstChar
     */
    public char getFirstChar() {
        return firstChar;
    }

    /**
     * @param firstChar ݒ肷 firstChar
     */
    public void setFirstChar(char firstChar) {
        this.firstChar = firstChar;
    }

    private char lastChar;

    /**
     * @return lastChar
     */
    public char getLastChar() {
        return lastChar;
    }

    /**
     * @param lastChar ݒ肷 lastChar
     */
    public void setLastChar(char lastChar) {
        this.lastChar = lastChar;
    }
    
    private char[] unicodeMap = new char[256];
    
    public char[] getUnicodeMap() {
        return this.unicodeMap;
    }
    
    public char lookupUnicodeMap(int index) {
        return this.unicodeMap[index];
    }
    
    public void putUnicodeMap(int index, char c) {
        this.unicodeMap[index] = c;
    }
    
    private boolean flgHasDifferences;
    
    public boolean hasDifferences() {
        return this.flgHasDifferences;
    }
    
    public void setHasDifferences(boolean flg) {
        this.flgHasDifferences = flg;
    }
    
    private byte[] differences = new byte[256];
    
    public byte[] getDifferences() {
        return this.differences;
    }
    
    public byte lookupDifference(int index) {
        return this.differences[index];
    }
    
    public void setDifference(int index, byte val) {
        this.differences[index] = val;
        if (val == 1) {
            this.setHasDifferences(true);
        }
    }

    @Override
    protected HPdfByteType getByteTypeByParse(byte[] text, int index) {
        // MEMO: 'single-byte' chars only, no need to parse.
        return HPdfByteType.SINGLE;
    }

    @Override
    public void writeValue(HPdfWriteStream out) {
        // TODO ꂽ\bhEX^u
        logger.trace("HPdfBasicEncoder#write");
        
        /*  if HPDF_ENCODING_FONT_SPECIFIC is selected, no Encoding object will be "
         *  written.
         */        
        if (this.getBaseEncoding().equals(HPDF_ENCODING_FONT_SPECIFIC)) {
            return;
        }
        
        /* if encoder has differences-data, encoding object is written as
           dictionary-object, otherwise it is written as name-object. */
        if (this.hasDifferences()) {
            out.writeStr("/Encoding <<\n"
                    + "/Type /Encoding\n"
                    + "/BaseEncoding ");
        } else {
            out.writeStr("/Encoding ");
        }
        
        out.writeEscapeName(this.getBaseEncoding());
        
        out.writeStr("\n");
        
        /* write differences data */
        if (this.hasDifferences()) {
            out.writeStr("/Differences [");
            
            for (char i = this.getFirstChar(); i <= this.getLastChar(); ++i) {
                if (this.getDifferences()[i] == 1) {
                    
                }
            }
        }
        
    }

    @Override
    public char toUnicode(int code) {
        // MEMO: this is a mere default implementation.
        // if subclasses can do it more efficiently, should do so.
        byte[] buf = new byte[2];
        buf[0] = (byte) ((code >> 8) & 0xFF);
        buf[1] = (byte) (code & 0xFF);
        return this.decode(buf).charAt(0);
    }

    @Override
    public String decode(byte[] buf) {
        // TODO ꂽ\bhEX^u
        return null;
    }

    @Override
    public byte[] encode(String txt) {
        // TODO ꂽ\bhEX^u
        return null;
    }

    @Override
    public void init() {
        // TODO ꂽ\bhEX^u
        
    }
    
    private static void initBuiltinEncodings() {
        // TODO ꂽ\bhEX^u
        HPDF_BUILTIN_ENCODINGS = new HashMap<String, HPdfBuiltinEncodingData>();
    }
    
    
    private static final char[] HPDF_UNICODE_MAP_STANDARD = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x2019,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x2018, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x00A1, (char) 0x00A2, (char) 0x00A3, (char) 0x2044, (char) 0x00A5, (char) 0x0192, (char) 0x00A7,
        (char) 0x00A4, (char) 0x0027, (char) 0x201C, (char) 0x00AB, (char) 0x2039, (char) 0x203A, (char) 0xFB01, (char) 0xFB02,
        (char) 0x0000, (char) 0x2013, (char) 0x2020, (char) 0x2021, (char) 0x00B7, (char) 0x0000, (char) 0x00B6, (char) 0x2022,
        (char) 0x201A, (char) 0x201E, (char) 0x201D, (char) 0x00BB, (char) 0x2026, (char) 0x2030, (char) 0x0000, (char) 0x00BF,
        (char) 0x0000, (char) 0x0060, (char) 0x00B4, (char) 0x02C6, (char) 0x02DC, (char) 0x00AF, (char) 0x02D8, (char) 0x02D9,
        (char) 0x00A8, (char) 0x0000, (char) 0x02DA, (char) 0x00B8, (char) 0x0000, (char) 0x02DD, (char) 0x02DB, (char) 0x02C7,
        (char) 0x2014, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x00C6, (char) 0x0000, (char) 0x00AA, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0141, (char) 0x00D8, (char) 0x0152, (char) 0x00BA, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x00E6, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0131, (char) 0x0000, (char) 0x0000,
        (char) 0x0142, (char) 0x00F8, (char) 0x0153, (char) 0x00DF, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000
    };

    private static final char[] HPDF_UNICODE_MAP_WIN_ANSI = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x0000,
        (char) 0x20AC, (char) 0x0000, (char) 0x201A, (char) 0x0192, (char) 0x201E, (char) 0x2026, (char) 0x2020, (char) 0x2021,
        (char) 0x02C6, (char) 0x2030, (char) 0x0160, (char) 0x2039, (char) 0x0152, (char) 0x0000, (char) 0x017D, (char) 0x0000,
        (char) 0x0000, (char) 0x2018, (char) 0x2019, (char) 0x201C, (char) 0x201D, (char) 0x2022, (char) 0x2013, (char) 0x2014,
        (char) 0x02DC, (char) 0x2122, (char) 0x0161, (char) 0x203A, (char) 0x0153, (char) 0x0000, (char) 0x017E, (char) 0x0178,
        (char) 0x00A0, (char) 0x00A1, (char) 0x00A2, (char) 0x00A3, (char) 0x00A4, (char) 0x00A5, (char) 0x00A6, (char) 0x00A7,
        (char) 0x00A8, (char) 0x00A9, (char) 0x00AA, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x00AE, (char) 0x00AF,
        (char) 0x00B0, (char) 0x00B1, (char) 0x00B2, (char) 0x00B3, (char) 0x00B4, (char) 0x00B5, (char) 0x00B6, (char) 0x00B7,
        (char) 0x00B8, (char) 0x00B9, (char) 0x00BA, (char) 0x00BB, (char) 0x00BC, (char) 0x00BD, (char) 0x00BE, (char) 0x00BF,
        (char) 0x00C0, (char) 0x00C1, (char) 0x00C2, (char) 0x00C3, (char) 0x00C4, (char) 0x00C5, (char) 0x00C6, (char) 0x00C7,
        (char) 0x00C8, (char) 0x00C9, (char) 0x00CA, (char) 0x00CB, (char) 0x00CC, (char) 0x00CD, (char) 0x00CE, (char) 0x00CF,
        (char) 0x00D0, (char) 0x00D1, (char) 0x00D2, (char) 0x00D3, (char) 0x00D4, (char) 0x00D5, (char) 0x00D6, (char) 0x00D7,
        (char) 0x00D8, (char) 0x00D9, (char) 0x00DA, (char) 0x00DB, (char) 0x00DC, (char) 0x00DD, (char) 0x00DE, (char) 0x00DF,
        (char) 0x00E0, (char) 0x00E1, (char) 0x00E2, (char) 0x00E3, (char) 0x00E4, (char) 0x00E5, (char) 0x00E6, (char) 0x00E7,
        (char) 0x00E8, (char) 0x00E9, (char) 0x00EA, (char) 0x00EB, (char) 0x00EC, (char) 0x00ED, (char) 0x00EE, (char) 0x00EF,
        (char) 0x00F0, (char) 0x00F1, (char) 0x00F2, (char) 0x00F3, (char) 0x00F4, (char) 0x00F5, (char) 0x00F6, (char) 0x00F7,
        (char) 0x00F8, (char) 0x00F9, (char) 0x00FA, (char) 0x00FB, (char) 0x00FC, (char) 0x00FD, (char) 0x00FE, (char) 0x00FF
    };

    private static final char[] HPDF_UNICODE_MAP_MAC_ROMAN = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x0000,
        (char) 0x00C4, (char) 0x00C5, (char) 0x00C7, (char) 0x00C9, (char) 0x00D1, (char) 0x00D6, (char) 0x00DC, (char) 0x00E1,
        (char) 0x00E0, (char) 0x00E2, (char) 0x00E4, (char) 0x00E3, (char) 0x00E5, (char) 0x00E7, (char) 0x00E9, (char) 0x00E8,
        (char) 0x00EA, (char) 0x00EB, (char) 0x00ED, (char) 0x00EC, (char) 0x00EE, (char) 0x00EF, (char) 0x00F1, (char) 0x00F3,
        (char) 0x00F2, (char) 0x00F4, (char) 0x00F6, (char) 0x00F5, (char) 0x00FA, (char) 0x00F9, (char) 0x00FB, (char) 0x00FC,
        (char) 0x2020, (char) 0x00B0, (char) 0x00A2, (char) 0x00A3, (char) 0x00A7, (char) 0x2022, (char) 0x00B6, (char) 0x00DF,
        (char) 0x00AE, (char) 0x00A9, (char) 0x2122, (char) 0x00B4, (char) 0x00A8, (char) 0x2260, (char) 0x00C6, (char) 0x00D8,
        (char) 0x221E, (char) 0x00B1, (char) 0x2264, (char) 0x2265, (char) 0x00A5, (char) 0x00B5, (char) 0x2202, (char) 0x2211,
        (char) 0x220F, (char) 0x03C0, (char) 0x222B, (char) 0x00AA, (char) 0x00BA, (char) 0x03A9, (char) 0x00E6, (char) 0x00F8,
        (char) 0x00BF, (char) 0x00A1, (char) 0x00AC, (char) 0x221A, (char) 0x0192, (char) 0x2248, (char) 0x2206, (char) 0x00AB,
        (char) 0x00BB, (char) 0x2026, (char) 0x00A0, (char) 0x00C0, (char) 0x00C3, (char) 0x00D5, (char) 0x0152, (char) 0x0153,
        (char) 0x2013, (char) 0x2014, (char) 0x201C, (char) 0x201D, (char) 0x2018, (char) 0x2019, (char) 0x00F7, (char) 0x25CA,
        (char) 0x00FF, (char) 0x0178, (char) 0x2044, (char) 0x20AC, (char) 0x2039, (char) 0x203A, (char) 0xFB01, (char) 0xFB02,
        (char) 0x2021, (char) 0x00B7, (char) 0x201A, (char) 0x201E, (char) 0x2030, (char) 0x00C2, (char) 0x00CA, (char) 0x00C1,
        (char) 0x00CB, (char) 0x00C8, (char) 0x00CD, (char) 0x00CE, (char) 0x00CF, (char) 0x00CC, (char) 0x00D3, (char) 0x00D4,
        (char) 0xF8FF, (char) 0x00D2, (char) 0x00DA, (char) 0x00DB, (char) 0x00D9, (char) 0x0131, (char) 0x02C6, (char) 0x02DC,
        (char) 0x00AF, (char) 0x02D8, (char) 0x02D9, (char) 0x02DA, (char) 0x00B8, (char) 0x02DD, (char) 0x02DB, (char) 0x02C7
    };

    private static final char[] HPDF_UNICODE_MAP_FONT_SPECIFIC = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0080, (char) 0x0081, (char) 0x0082, (char) 0x0083, (char) 0x0084, (char) 0x0085, (char) 0x0086, (char) 0x0087,
        (char) 0x0088, (char) 0x0089, (char) 0x008A, (char) 0x008B, (char) 0x008C, (char) 0x008D, (char) 0x008E, (char) 0x008F,
        (char) 0x0090, (char) 0x0091, (char) 0x0092, (char) 0x0093, (char) 0x0094, (char) 0x0095, (char) 0x0096, (char) 0x0097,
        (char) 0x0098, (char) 0x0099, (char) 0x009A, (char) 0x009B, (char) 0x009C, (char) 0x009D, (char) 0x009E, (char) 0x009F,
        (char) 0x00A0, (char) 0x00A1, (char) 0x00A2, (char) 0x00A3, (char) 0x00A4, (char) 0x00A5, (char) 0x00A6, (char) 0x00A7,
        (char) 0x00A8, (char) 0x00A9, (char) 0x00AA, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x00AE, (char) 0x00AF,
        (char) 0x00B0, (char) 0x00B1, (char) 0x00B2, (char) 0x00B3, (char) 0x00B4, (char) 0x00B5, (char) 0x00B6, (char) 0x00B7,
        (char) 0x00B8, (char) 0x00B9, (char) 0x00BA, (char) 0x00BB, (char) 0x00BC, (char) 0x00BD, (char) 0x00BE, (char) 0x00BF,
        (char) 0x00C0, (char) 0x00C1, (char) 0x00C2, (char) 0x00C3, (char) 0x00C4, (char) 0x00C5, (char) 0x00C6, (char) 0x00C7,
        (char) 0x00C8, (char) 0x00C9, (char) 0x00CA, (char) 0x00CB, (char) 0x00CC, (char) 0x00CD, (char) 0x00CE, (char) 0x00CF,
        (char) 0x00D0, (char) 0x00D1, (char) 0x00D2, (char) 0x00D3, (char) 0x00D4, (char) 0x00D5, (char) 0x00D6, (char) 0x00D7,
        (char) 0x00D8, (char) 0x00D9, (char) 0x00DA, (char) 0x00DB, (char) 0x00DC, (char) 0x00DD, (char) 0x00DE, (char) 0x00DF,
        (char) 0x00E0, (char) 0x00E1, (char) 0x00E2, (char) 0x00E3, (char) 0x00E4, (char) 0x00E5, (char) 0x00E6, (char) 0x00E7,
        (char) 0x00E8, (char) 0x00E9, (char) 0x00EA, (char) 0x00EB, (char) 0x00EC, (char) 0x00ED, (char) 0x00EE, (char) 0x00EF,
        (char) 0x00F0, (char) 0x00F1, (char) 0x00F2, (char) 0x00F3, (char) 0x00F4, (char) 0x00F5, (char) 0x00F6, (char) 0x00F7,
        (char) 0x00F8, (char) 0x00F9, (char) 0x00FA, (char) 0x00FB, (char) 0x00FC, (char) 0x00FD, (char) 0x00FE, (char) 0x00FF
    };

    //private static final char ISO8859_FIRST_CHAR = (char) 32;
    //private static final char ISO8859_LAST_CHAR = (char) 255;

    /*---------------------------------------------------------------------------*/
    /*------ ISO8859-2 Encoding -------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_ISO8859_2 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x0104, (char) 0x02D8, (char) 0x0141, (char) 0x00A4, (char) 0x013D, (char) 0x015A, (char) 0x00A7,
        (char) 0x00A8, (char) 0x0160, (char) 0x015E, (char) 0x0164, (char) 0x0179, (char) 0x00AD, (char) 0x017D, (char) 0x017B,
        (char) 0x00B0, (char) 0x0105, (char) 0x02DB, (char) 0x0142, (char) 0x00B4, (char) 0x013E, (char) 0x015B, (char) 0x02C7,
        (char) 0x00B8, (char) 0x0161, (char) 0x015F, (char) 0x0165, (char) 0x017A, (char) 0x02DD, (char) 0x017E, (char) 0x017C,
        (char) 0x0154, (char) 0x00C1, (char) 0x00C2, (char) 0x0102, (char) 0x00C4, (char) 0x0139, (char) 0x0106, (char) 0x00C7,
        (char) 0x010C, (char) 0x00C9, (char) 0x0118, (char) 0x00CB, (char) 0x011A, (char) 0x00CD, (char) 0x00CE, (char) 0x010E,
        (char) 0x0110, (char) 0x0143, (char) 0x0147, (char) 0x00D3, (char) 0x00D4, (char) 0x0150, (char) 0x00D6, (char) 0x00D7,
        (char) 0x0158, (char) 0x016E, (char) 0x00DA, (char) 0x0170, (char) 0x00DC, (char) 0x00DD, (char) 0x0162, (char) 0x00DF,
        (char) 0x0155, (char) 0x00E1, (char) 0x00E2, (char) 0x0103, (char) 0x00E4, (char) 0x013A, (char) 0x0107, (char) 0x00E7,
        (char) 0x010D, (char) 0x00E9, (char) 0x0119, (char) 0x00EB, (char) 0x011B, (char) 0x00ED, (char) 0x00EE, (char) 0x010F,
        (char) 0x0111, (char) 0x0144, (char) 0x0148, (char) 0x00F3, (char) 0x00F4, (char) 0x0151, (char) 0x00F6, (char) 0x00F7,
        (char) 0x0159, (char) 0x016F, (char) 0x00FA, (char) 0x0171, (char) 0x00FC, (char) 0x00FD, (char) 0x0163, (char) 0x02D9
    };

    /*---------------------------------------------------------------------------*/
    /*------ ISO8859-3 Encoding -------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_ISO8859_3 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x0126, (char) 0x02D8, (char) 0x00A3, (char) 0x00A4, (char) 0x0000, (char) 0x0124, (char) 0x00A7,
        (char) 0x00A8, (char) 0x0130, (char) 0x015E, (char) 0x011E, (char) 0x0134, (char) 0x00AD, (char) 0x0000, (char) 0x017B,
        (char) 0x00B0, (char) 0x0127, (char) 0x00B2, (char) 0x00B3, (char) 0x00B4, (char) 0x00B5, (char) 0x0125, (char) 0x00B7,
        (char) 0x00B8, (char) 0x0131, (char) 0x015F, (char) 0x011F, (char) 0x0135, (char) 0x00BD, (char) 0x0000, (char) 0x017C,
        (char) 0x00C0, (char) 0x00C1, (char) 0x00C2, (char) 0x0000, (char) 0x00C4, (char) 0x010A, (char) 0x0108, (char) 0x00C7,
        (char) 0x00C8, (char) 0x00C9, (char) 0x00CA, (char) 0x00CB, (char) 0x00CC, (char) 0x00CD, (char) 0x00CE, (char) 0x00CF,
        (char) 0x0000, (char) 0x00D1, (char) 0x00D2, (char) 0x00D3, (char) 0x00D4, (char) 0x0120, (char) 0x00D6, (char) 0x00D7,
        (char) 0x011C, (char) 0x00D9, (char) 0x00DA, (char) 0x00DB, (char) 0x00DC, (char) 0x016C, (char) 0x015C, (char) 0x00DF,
        (char) 0x00E0, (char) 0x00E1, (char) 0x00E2, (char) 0x0000, (char) 0x00E4, (char) 0x010B, (char) 0x0109, (char) 0x00E7,
        (char) 0x00E8, (char) 0x00E9, (char) 0x00EA, (char) 0x00EB, (char) 0x00EC, (char) 0x00ED, (char) 0x00EE, (char) 0x00EF,
        (char) 0x0000, (char) 0x00F1, (char) 0x00F2, (char) 0x00F3, (char) 0x00F4, (char) 0x0121, (char) 0x00F6, (char) 0x00F7,
        (char) 0x011D, (char) 0x00F9, (char) 0x00FA, (char) 0x00FB, (char) 0x00FC, (char) 0x016D, (char) 0x015D, (char) 0x02D9
    };

    /*---------------------------------------------------------------------------*/
    /*------ ISO8859-4 Encoding -------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_ISO8859_4 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x0104, (char) 0x0138, (char) 0x0156, (char) 0x00A4, (char) 0x0128, (char) 0x013B, (char) 0x00A7,
        (char) 0x00A8, (char) 0x0160, (char) 0x0112, (char) 0x0122, (char) 0x0166, (char) 0x00AD, (char) 0x017D, (char) 0x00AF,
        (char) 0x00B0, (char) 0x0105, (char) 0x02DB, (char) 0x0157, (char) 0x00B4, (char) 0x0129, (char) 0x013C, (char) 0x02C7,
        (char) 0x00B8, (char) 0x0161, (char) 0x0113, (char) 0x0123, (char) 0x0167, (char) 0x014A, (char) 0x017E, (char) 0x014B,
        (char) 0x0100, (char) 0x00C1, (char) 0x00C2, (char) 0x00C3, (char) 0x00C4, (char) 0x00C5, (char) 0x00C6, (char) 0x012E,
        (char) 0x010C, (char) 0x00C9, (char) 0x0118, (char) 0x00CB, (char) 0x0116, (char) 0x00CD, (char) 0x00CE, (char) 0x012A,
        (char) 0x0110, (char) 0x0145, (char) 0x014C, (char) 0x0136, (char) 0x00D4, (char) 0x00D5, (char) 0x00D6, (char) 0x00D7,
        (char) 0x00D8, (char) 0x0172, (char) 0x00DA, (char) 0x00DB, (char) 0x00DC, (char) 0x0168, (char) 0x016A, (char) 0x00DF,
        (char) 0x0101, (char) 0x00E1, (char) 0x00E2, (char) 0x00E3, (char) 0x00E4, (char) 0x00E5, (char) 0x00E6, (char) 0x012F,
        (char) 0x010D, (char) 0x00E9, (char) 0x0119, (char) 0x00EB, (char) 0x0117, (char) 0x00ED, (char) 0x00EE, (char) 0x012B,
        (char) 0x0111, (char) 0x0146, (char) 0x014D, (char) 0x0137, (char) 0x00F4, (char) 0x00F5, (char) 0x00F6, (char) 0x00F7,
        (char) 0x00F8, (char) 0x0173, (char) 0x00FA, (char) 0x00FB, (char) 0x00FC, (char) 0x0169, (char) 0x016B, (char) 0x02D9
    };

    /*---------------------------------------------------------------------------*/
    /*------ ISO8859-5 Encoding -------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_ISO8859_5 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x0401, (char) 0x0402, (char) 0x0403, (char) 0x0404, (char) 0x0405, (char) 0x0406, (char) 0x0407,
        (char) 0x0408, (char) 0x0409, (char) 0x040A, (char) 0x040B, (char) 0x040C, (char) 0x00AD, (char) 0x040E, (char) 0x040F,
        (char) 0x0410, (char) 0x0411, (char) 0x0412, (char) 0x0413, (char) 0x0414, (char) 0x0415, (char) 0x0416, (char) 0x0417,
        (char) 0x0418, (char) 0x0419, (char) 0x041A, (char) 0x041B, (char) 0x041C, (char) 0x041D, (char) 0x041E, (char) 0x041F,
        (char) 0x0420, (char) 0x0421, (char) 0x0422, (char) 0x0423, (char) 0x0424, (char) 0x0425, (char) 0x0426, (char) 0x0427,
        (char) 0x0428, (char) 0x0429, (char) 0x042A, (char) 0x042B, (char) 0x042C, (char) 0x042D, (char) 0x042E, (char) 0x042F,
        (char) 0x0430, (char) 0x0431, (char) 0x0432, (char) 0x0433, (char) 0x0434, (char) 0x0435, (char) 0x0436, (char) 0x0437,
        (char) 0x0438, (char) 0x0439, (char) 0x043A, (char) 0x043B, (char) 0x043C, (char) 0x043D, (char) 0x043E, (char) 0x043F,
        (char) 0x0440, (char) 0x0441, (char) 0x0442, (char) 0x0443, (char) 0x0444, (char) 0x0445, (char) 0x0446, (char) 0x0447,
        (char) 0x0448, (char) 0x0449, (char) 0x044A, (char) 0x044B, (char) 0x044C, (char) 0x044D, (char) 0x044E, (char) 0x044F,
        (char) 0x2116, (char) 0x0451, (char) 0x0452, (char) 0x0453, (char) 0x0454, (char) 0x0455, (char) 0x0456, (char) 0x0457,
        (char) 0x0458, (char) 0x0459, (char) 0x045A, (char) 0x045B, (char) 0x045C, (char) 0x00A7, (char) 0x045E, (char) 0x045F
    };

    /*---------------------------------------------------------------------------*/
    /*------ ISO8859-6 Encoding -------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_ISO8859_6 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x00A4, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x060C, (char) 0x00AD, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x061B, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x061F,
        (char) 0x0000, (char) 0x0621, (char) 0x0622, (char) 0x0623, (char) 0x0624, (char) 0x0625, (char) 0x0626, (char) 0x0627,
        (char) 0x0628, (char) 0x0629, (char) 0x062A, (char) 0x062B, (char) 0x062C, (char) 0x062D, (char) 0x062E, (char) 0x062F,
        (char) 0x0630, (char) 0x0631, (char) 0x0632, (char) 0x0633, (char) 0x0634, (char) 0x0635, (char) 0x0636, (char) 0x0637,
        (char) 0x0638, (char) 0x0639, (char) 0x063A, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0640, (char) 0x0641, (char) 0x0642, (char) 0x0643, (char) 0x0644, (char) 0x0645, (char) 0x0646, (char) 0x0647,
        (char) 0x0648, (char) 0x0649, (char) 0x064A, (char) 0x064B, (char) 0x064C, (char) 0x064D, (char) 0x064E, (char) 0x064F,
        (char) 0x0650, (char) 0x0651, (char) 0x0652, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000
    };

    /*---------------------------------------------------------------------------*/
    /*------ ISO8859-7 Encoding -------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_ISO8859_7 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x2018, (char) 0x2019, (char) 0x00A3, (char) 0x20AC, (char) 0x20AF, (char) 0x00A6, (char) 0x00A7,
        (char) 0x00A8, (char) 0x00A9, (char) 0x037A, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x0000, (char) 0x2015,
        (char) 0x00B0, (char) 0x00B1, (char) 0x00B2, (char) 0x00B3, (char) 0x0384, (char) 0x0385, (char) 0x0386, (char) 0x00B7,
        (char) 0x0388, (char) 0x0389, (char) 0x038A, (char) 0x00BB, (char) 0x038C, (char) 0x00BD, (char) 0x038E, (char) 0x038F,
        (char) 0x0390, (char) 0x0391, (char) 0x0392, (char) 0x0393, (char) 0x0394, (char) 0x0395, (char) 0x0396, (char) 0x0397,
        (char) 0x0398, (char) 0x0399, (char) 0x039A, (char) 0x039B, (char) 0x039C, (char) 0x039D, (char) 0x039E, (char) 0x039F,
        (char) 0x03A0, (char) 0x03A1, (char) 0x0000, (char) 0x03A3, (char) 0x03A4, (char) 0x03A5, (char) 0x03A6, (char) 0x03A7,
        (char) 0x03A8, (char) 0x03A9, (char) 0x03AA, (char) 0x03AB, (char) 0x03AC, (char) 0x03AD, (char) 0x03AE, (char) 0x03AF,
        (char) 0x03B0, (char) 0x03B1, (char) 0x03B2, (char) 0x03B3, (char) 0x03B4, (char) 0x03B5, (char) 0x03B6, (char) 0x03B7,
        (char) 0x03B8, (char) 0x03B9, (char) 0x03BA, (char) 0x03BB, (char) 0x03BC, (char) 0x03BD, (char) 0x03BE, (char) 0x03BF,
        (char) 0x03C0, (char) 0x03C1, (char) 0x03C2, (char) 0x03C3, (char) 0x03C4, (char) 0x03C5, (char) 0x03C6, (char) 0x03C7,
        (char) 0x03C8, (char) 0x03C9, (char) 0x03CA, (char) 0x03CB, (char) 0x03CC, (char) 0x03CD, (char) 0x03CE, (char) 0x0000
    };

    /*---------------------------------------------------------------------------*/
    /*------ ISO8859-8 Encoding -------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_ISO8859_8 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x0000, (char) 0x00A2, (char) 0x00A3, (char) 0x00A4, (char) 0x00A5, (char) 0x00A6, (char) 0x00A7,
        (char) 0x00A8, (char) 0x00A9, (char) 0x00D7, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x00AE, (char) 0x00AF,
        (char) 0x00B0, (char) 0x00B1, (char) 0x00B2, (char) 0x00B3, (char) 0x00B4, (char) 0x00B5, (char) 0x00B6, (char) 0x00B7,
        (char) 0x00B8, (char) 0x00B9, (char) 0x00F7, (char) 0x00BB, (char) 0x00BC, (char) 0x00BD, (char) 0x00BE, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x2017,
        (char) 0x05D0, (char) 0x05D1, (char) 0x05D2, (char) 0x05D3, (char) 0x05D4, (char) 0x05D5, (char) 0x05D6, (char) 0x05D7,
        (char) 0x05D8, (char) 0x05D9, (char) 0x05DA, (char) 0x05DB, (char) 0x05DC, (char) 0x05DD, (char) 0x05DE, (char) 0x05DF,
        (char) 0x05E0, (char) 0x05E1, (char) 0x05E2, (char) 0x05E3, (char) 0x05E4, (char) 0x05E5, (char) 0x05E6, (char) 0x05E7,
        (char) 0x05E8, (char) 0x05E9, (char) 0x05EA, (char) 0x0000, (char) 0x0000, (char) 0x200E, (char) 0x200F, (char) 0x0000,
    };

    /*---------------------------------------------------------------------------*/
    /*------ ISO8859-9 Encoding -------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_ISO8859_9 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x00A1, (char) 0x00A2, (char) 0x00A3, (char) 0x00A4, (char) 0x00A5, (char) 0x00A6, (char) 0x00A7,
        (char) 0x00A8, (char) 0x00A9, (char) 0x00AA, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x00AE, (char) 0x00AF,
        (char) 0x00B0, (char) 0x00B1, (char) 0x00B2, (char) 0x00B3, (char) 0x00B4, (char) 0x00B5, (char) 0x00B6, (char) 0x00B7,
        (char) 0x00B8, (char) 0x00B9, (char) 0x00BA, (char) 0x00BB, (char) 0x00BC, (char) 0x00BD, (char) 0x00BE, (char) 0x00BF,
        (char) 0x00C0, (char) 0x00C1, (char) 0x00C2, (char) 0x00C3, (char) 0x00C4, (char) 0x00C5, (char) 0x00C6, (char) 0x00C7,
        (char) 0x00C8, (char) 0x00C9, (char) 0x00CA, (char) 0x00CB, (char) 0x00CC, (char) 0x00CD, (char) 0x00CE, (char) 0x00CF,
        (char) 0x011E, (char) 0x00D1, (char) 0x00D2, (char) 0x00D3, (char) 0x00D4, (char) 0x00D5, (char) 0x00D6, (char) 0x00D7,
        (char) 0x00D8, (char) 0x00D9, (char) 0x00DA, (char) 0x00DB, (char) 0x00DC, (char) 0x0130, (char) 0x015E, (char) 0x00DF,
        (char) 0x00E0, (char) 0x00E1, (char) 0x00E2, (char) 0x00E3, (char) 0x00E4, (char) 0x00E5, (char) 0x00E6, (char) 0x00E7,
        (char) 0x00E8, (char) 0x00E9, (char) 0x00EA, (char) 0x00EB, (char) 0x00EC, (char) 0x00ED, (char) 0x00EE, (char) 0x00EF,
        (char) 0x011F, (char) 0x00F1, (char) 0x00F2, (char) 0x00F3, (char) 0x00F4, (char) 0x00F5, (char) 0x00F6, (char) 0x00F7,
        (char) 0x00F8, (char) 0x00F9, (char) 0x00FA, (char) 0x00FB, (char) 0x00FC, (char) 0x0131, (char) 0x015F, (char) 0x00FF
    };

    /*---------------------------------------------------------------------------*/
    /*------ ISO8859-10 Encoding ------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_ISO8859_10 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x0104, (char) 0x0112, (char) 0x0122, (char) 0x012A, (char) 0x0128, (char) 0x0136, (char) 0x00A7,
        (char) 0x013B, (char) 0x0110, (char) 0x0160, (char) 0x0166, (char) 0x017D, (char) 0x00AD, (char) 0x016A, (char) 0x014A,
        (char) 0x00B0, (char) 0x0105, (char) 0x0113, (char) 0x0123, (char) 0x012B, (char) 0x0129, (char) 0x0137, (char) 0x00B7,
        (char) 0x013C, (char) 0x0111, (char) 0x0161, (char) 0x0167, (char) 0x017E, (char) 0x2015, (char) 0x016B, (char) 0x014B,
        (char) 0x0100, (char) 0x00C1, (char) 0x00C2, (char) 0x00C3, (char) 0x00C4, (char) 0x00C5, (char) 0x00C6, (char) 0x012E,
        (char) 0x010C, (char) 0x00C9, (char) 0x0118, (char) 0x00CB, (char) 0x0116, (char) 0x00CD, (char) 0x00CE, (char) 0x00CF,
        (char) 0x00D0, (char) 0x0145, (char) 0x014C, (char) 0x00D3, (char) 0x00D4, (char) 0x00D5, (char) 0x00D6, (char) 0x0168,
        (char) 0x00D8, (char) 0x0172, (char) 0x00DA, (char) 0x00DB, (char) 0x00DC, (char) 0x00DD, (char) 0x00DE, (char) 0x00DF,
        (char) 0x0101, (char) 0x00E1, (char) 0x00E2, (char) 0x00E3, (char) 0x00E4, (char) 0x00E5, (char) 0x00E6, (char) 0x012F,
        (char) 0x010D, (char) 0x00E9, (char) 0x0119, (char) 0x00EB, (char) 0x0117, (char) 0x00ED, (char) 0x00EE, (char) 0x00EF,
        (char) 0x00F0, (char) 0x0146, (char) 0x014D, (char) 0x00F3, (char) 0x00F4, (char) 0x00F5, (char) 0x00F6, (char) 0x0169,
        (char) 0x00F8, (char) 0x0173, (char) 0x00FA, (char) 0x00FB, (char) 0x00FC, (char) 0x00FD, (char) 0x00FE, (char) 0x0138
    };

    /*---------------------------------------------------------------------------*/
    /*------ ISO8859-11 Encoding ------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_ISO8859_11 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x0E01, (char) 0x0E02, (char) 0x0E03, (char) 0x0E04, (char) 0x0E05, (char) 0x0E06, (char) 0x0E07,
        (char) 0x0E08, (char) 0x0E09, (char) 0x0E0A, (char) 0x0E0B, (char) 0x0E0C, (char) 0x0E0D, (char) 0x0E0E, (char) 0x0E0F,
        (char) 0x0E10, (char) 0x0E11, (char) 0x0E12, (char) 0x0E13, (char) 0x0E14, (char) 0x0E15, (char) 0x0E16, (char) 0x0E17,
        (char) 0x0E18, (char) 0x0E19, (char) 0x0E1A, (char) 0x0E1B, (char) 0x0E1C, (char) 0x0E1D, (char) 0x0E1E, (char) 0x0E1F,
        (char) 0x0E20, (char) 0x0E21, (char) 0x0E22, (char) 0x0E23, (char) 0x0E24, (char) 0x0E25, (char) 0x0E26, (char) 0x0E27,
        (char) 0x0E28, (char) 0x0E29, (char) 0x0E2A, (char) 0x0E2B, (char) 0x0E2C, (char) 0x0E2D, (char) 0x0E2E, (char) 0x0E2F,
        (char) 0x0E30, (char) 0x0E31, (char) 0x0E32, (char) 0x0E33, (char) 0x0E34, (char) 0x0E35, (char) 0x0E36, (char) 0x0E37,
        (char) 0x0E38, (char) 0x0E39, (char) 0x0E3A, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0E3F,
        (char) 0x0E40, (char) 0x0E41, (char) 0x0E42, (char) 0x0E43, (char) 0x0E44, (char) 0x0E45, (char) 0x0E46, (char) 0x0E47,
        (char) 0x0E48, (char) 0x0E49, (char) 0x0E4A, (char) 0x0E4B, (char) 0x0E4C, (char) 0x0E4D, (char) 0x0E4E, (char) 0x0E4F,
        (char) 0x0E50, (char) 0x0E51, (char) 0x0E52, (char) 0x0E53, (char) 0x0E54, (char) 0x0E55, (char) 0x0E56, (char) 0x0E57,
        (char) 0x0E58, (char) 0x0E59, (char) 0x0E5A, (char) 0x0E5B, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000
    };

    /*---------------------------------------------------------------------------*/
    /*------ ISO8859-13 Encoding ------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_ISO8859_13 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x201D, (char) 0x00A2, (char) 0x00A3, (char) 0x00A4, (char) 0x201E, (char) 0x00A6, (char) 0x00A7,
        (char) 0x00D8, (char) 0x00A9, (char) 0x0156, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x00AE, (char) 0x00C6,
        (char) 0x00B0, (char) 0x00B1, (char) 0x00B2, (char) 0x00B3, (char) 0x201C, (char) 0x00B5, (char) 0x00B6, (char) 0x00B7,
        (char) 0x00F8, (char) 0x00B9, (char) 0x0157, (char) 0x00BB, (char) 0x00BC, (char) 0x00BD, (char) 0x00BE, (char) 0x00E6,
        (char) 0x0104, (char) 0x012E, (char) 0x0100, (char) 0x0106, (char) 0x00C4, (char) 0x00C5, (char) 0x0118, (char) 0x0112,
        (char) 0x010C, (char) 0x00C9, (char) 0x0179, (char) 0x0116, (char) 0x0122, (char) 0x0136, (char) 0x012A, (char) 0x013B,
        (char) 0x0160, (char) 0x0143, (char) 0x0145, (char) 0x00D3, (char) 0x014C, (char) 0x00D5, (char) 0x00D6, (char) 0x00D7,
        (char) 0x0172, (char) 0x0141, (char) 0x015A, (char) 0x016A, (char) 0x00DC, (char) 0x017B, (char) 0x017D, (char) 0x00DF,
        (char) 0x0105, (char) 0x012F, (char) 0x0101, (char) 0x0107, (char) 0x00E4, (char) 0x00E5, (char) 0x0119, (char) 0x0113,
        (char) 0x010D, (char) 0x00E9, (char) 0x017A, (char) 0x0117, (char) 0x0123, (char) 0x0137, (char) 0x012B, (char) 0x013C,
        (char) 0x0161, (char) 0x0144, (char) 0x0146, (char) 0x00F3, (char) 0x014D, (char) 0x00F5, (char) 0x00F6, (char) 0x00F7,
        (char) 0x0173, (char) 0x0142, (char) 0x015B, (char) 0x016B, (char) 0x00FC, (char) 0x017C, (char) 0x017E, (char) 0x2019
    };

    /*---------------------------------------------------------------------------*/
    /*------ ISO8859-14 Encoding ------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_ISO8859_14 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x1E02, (char) 0x1E03, (char) 0x00A3, (char) 0x010A, (char) 0x010B, (char) 0x1E0A, (char) 0x00A7,
        (char) 0x1E80, (char) 0x00A9, (char) 0x1E82, (char) 0x1E0B, (char) 0x1EF2, (char) 0x00AD, (char) 0x00AE, (char) 0x0178,
        (char) 0x1E1E, (char) 0x1E1F, (char) 0x0120, (char) 0x0121, (char) 0x1E40, (char) 0x1E41, (char) 0x00B6, (char) 0x1E56,
        (char) 0x1E81, (char) 0x1E57, (char) 0x1E83, (char) 0x1E60, (char) 0x1EF3, (char) 0x1E84, (char) 0x1E85, (char) 0x1E61,
        (char) 0x00C0, (char) 0x00C1, (char) 0x00C2, (char) 0x00C3, (char) 0x00C4, (char) 0x00C5, (char) 0x00C6, (char) 0x00C7,
        (char) 0x00C8, (char) 0x00C9, (char) 0x00CA, (char) 0x00CB, (char) 0x00CC, (char) 0x00CD, (char) 0x00CE, (char) 0x00CF,
        (char) 0x0174, (char) 0x00D1, (char) 0x00D2, (char) 0x00D3, (char) 0x00D4, (char) 0x00D5, (char) 0x00D6, (char) 0x1E6A,
        (char) 0x00D8, (char) 0x00D9, (char) 0x00DA, (char) 0x00DB, (char) 0x00DC, (char) 0x00DD, (char) 0x0176, (char) 0x00DF,
        (char) 0x00E0, (char) 0x00E1, (char) 0x00E2, (char) 0x00E3, (char) 0x00E4, (char) 0x00E5, (char) 0x00E6, (char) 0x00E7,
        (char) 0x00E8, (char) 0x00E9, (char) 0x00EA, (char) 0x00EB, (char) 0x00EC, (char) 0x00ED, (char) 0x00EE, (char) 0x00EF,
        (char) 0x0175, (char) 0x00F1, (char) 0x00F2, (char) 0x00F3, (char) 0x00F4, (char) 0x00F5, (char) 0x00F6, (char) 0x1E6B,
        (char) 0x00F8, (char) 0x00F9, (char) 0x00FA, (char) 0x00FB, (char) 0x00FC, (char) 0x00FD, (char) 0x0177, (char) 0x00FF
    };

    /*---------------------------------------------------------------------------*/
    /*------ ISO8859-15 Encoding ------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_ISO8859_15 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x00A1, (char) 0x00A2, (char) 0x00A3, (char) 0x20AC, (char) 0x00A5, (char) 0x0160, (char) 0x00A7,
        (char) 0x0161, (char) 0x00A9, (char) 0x00AA, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x00AE, (char) 0x00AF,
        (char) 0x00B0, (char) 0x00B1, (char) 0x00B2, (char) 0x00B3, (char) 0x017D, (char) 0x00B5, (char) 0x00B6, (char) 0x00B7,
        (char) 0x017E, (char) 0x00B9, (char) 0x00BA, (char) 0x00BB, (char) 0x0152, (char) 0x0153, (char) 0x0178, (char) 0x00BF,
        (char) 0x00C0, (char) 0x00C1, (char) 0x00C2, (char) 0x00C3, (char) 0x00C4, (char) 0x00C5, (char) 0x00C6, (char) 0x00C7,
        (char) 0x00C8, (char) 0x00C9, (char) 0x00CA, (char) 0x00CB, (char) 0x00CC, (char) 0x00CD, (char) 0x00CE, (char) 0x00CF,
        (char) 0x00D0, (char) 0x00D1, (char) 0x00D2, (char) 0x00D3, (char) 0x00D4, (char) 0x00D5, (char) 0x00D6, (char) 0x00D7,
        (char) 0x00D8, (char) 0x00D9, (char) 0x00DA, (char) 0x00DB, (char) 0x00DC, (char) 0x00DD, (char) 0x00DE, (char) 0x00DF,
        (char) 0x00E0, (char) 0x00E1, (char) 0x00E2, (char) 0x00E3, (char) 0x00E4, (char) 0x00E5, (char) 0x00E6, (char) 0x00E7,
        (char) 0x00E8, (char) 0x00E9, (char) 0x00EA, (char) 0x00EB, (char) 0x00EC, (char) 0x00ED, (char) 0x00EE, (char) 0x00EF,
        (char) 0x00F0, (char) 0x00F1, (char) 0x00F2, (char) 0x00F3, (char) 0x00F4, (char) 0x00F5, (char) 0x00F6, (char) 0x00F7,
        (char) 0x00F8, (char) 0x00F9, (char) 0x00FA, (char) 0x00FB, (char) 0x00FC, (char) 0x00FD, (char) 0x00FE, (char) 0x00FF
    };

    /*---------------------------------------------------------------------------*/
    /*------ ISO8859-16 Encoding ------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_ISO8859_16 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x0104, (char) 0x0105, (char) 0x0141, (char) 0x20AC, (char) 0x201E, (char) 0x0160, (char) 0x00A7,
        (char) 0x0161, (char) 0x00A9, (char) 0x0218, (char) 0x00AB, (char) 0x0179, (char) 0x00AD, (char) 0x017A, (char) 0x017B,
        (char) 0x00B0, (char) 0x00B1, (char) 0x010C, (char) 0x0142, (char) 0x017D, (char) 0x201D, (char) 0x00B6, (char) 0x00B7,
        (char) 0x017E, (char) 0x010D, (char) 0x0219, (char) 0x00BB, (char) 0x0152, (char) 0x0153, (char) 0x0178, (char) 0x017C,
        (char) 0x00C0, (char) 0x00C1, (char) 0x00C2, (char) 0x0102, (char) 0x00C4, (char) 0x0106, (char) 0x00C6, (char) 0x00C7,
        (char) 0x00C8, (char) 0x00C9, (char) 0x00CA, (char) 0x00CB, (char) 0x00CC, (char) 0x00CD, (char) 0x00CE, (char) 0x00CF,
        (char) 0x0110, (char) 0x0143, (char) 0x00D2, (char) 0x00D3, (char) 0x00D4, (char) 0x0150, (char) 0x00D6, (char) 0x015A,
        (char) 0x0170, (char) 0x00D9, (char) 0x00DA, (char) 0x00DB, (char) 0x00DC, (char) 0x0118, (char) 0x021A, (char) 0x00DF,
        (char) 0x00E0, (char) 0x00E1, (char) 0x00E2, (char) 0x0103, (char) 0x00E4, (char) 0x0107, (char) 0x00E6, (char) 0x00E7,
        (char) 0x00E8, (char) 0x00E9, (char) 0x00EA, (char) 0x00EB, (char) 0x00EC, (char) 0x00ED, (char) 0x00EE, (char) 0x00EF,
        (char) 0x0111, (char) 0x0144, (char) 0x00F2, (char) 0x00F3, (char) 0x00F4, (char) 0x0151, (char) 0x00F6, (char) 0x015B,
        (char) 0x0171, (char) 0x00F9, (char) 0x00FA, (char) 0x00FB, (char) 0x00FC, (char) 0x0119, (char) 0x021B, (char) 0x00FF
    };

    //private static final char MSCP_FIRST_CHAR = (char) 32;
    //private static final char MSCP_LAST_CHAR = (char) 255;


    /*---------------------------------------------------------------------------*/
    /*------ CP1250 Encoding ----------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_CP1250 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x20AC, (char) 0x0000, (char) 0x201A, (char) 0x0000, (char) 0x201E, (char) 0x2026, (char) 0x2020, (char) 0x2021,
        (char) 0x0000, (char) 0x2030, (char) 0x0160, (char) 0x2039, (char) 0x015A, (char) 0x0164, (char) 0x017D, (char) 0x0179,
        (char) 0x0000, (char) 0x2018, (char) 0x2019, (char) 0x201C, (char) 0x201D, (char) 0x2022, (char) 0x2013, (char) 0x2014,
        (char) 0x0000, (char) 0x2122, (char) 0x0161, (char) 0x203A, (char) 0x015B, (char) 0x0165, (char) 0x017E, (char) 0x017A,
        (char) 0x00A0, (char) 0x02C7, (char) 0x02D8, (char) 0x0141, (char) 0x00A4, (char) 0x0104, (char) 0x00A6, (char) 0x00A7,
        (char) 0x00A8, (char) 0x00A9, (char) 0x015E, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x00AE, (char) 0x017B,
        (char) 0x00B0, (char) 0x00B1, (char) 0x02DB, (char) 0x0142, (char) 0x00B4, (char) 0x00B5, (char) 0x00B6, (char) 0x00B7,
        (char) 0x00B8, (char) 0x0105, (char) 0x015F, (char) 0x00BB, (char) 0x013D, (char) 0x02DD, (char) 0x013E, (char) 0x017C,
        (char) 0x0154, (char) 0x00C1, (char) 0x00C2, (char) 0x0102, (char) 0x00C4, (char) 0x0139, (char) 0x0106, (char) 0x00C7,
        (char) 0x010C, (char) 0x00C9, (char) 0x0118, (char) 0x00CB, (char) 0x011A, (char) 0x00CD, (char) 0x00CE, (char) 0x010E,
        (char) 0x0110, (char) 0x0143, (char) 0x0147, (char) 0x00D3, (char) 0x00D4, (char) 0x0150, (char) 0x00D6, (char) 0x00D7,
        (char) 0x0158, (char) 0x016E, (char) 0x00DA, (char) 0x0170, (char) 0x00DC, (char) 0x00DD, (char) 0x0162, (char) 0x00DF,
        (char) 0x0155, (char) 0x00E1, (char) 0x00E2, (char) 0x0103, (char) 0x00E4, (char) 0x013A, (char) 0x0107, (char) 0x00E7,
        (char) 0x010D, (char) 0x00E9, (char) 0x0119, (char) 0x00EB, (char) 0x011B, (char) 0x00ED, (char) 0x00EE, (char) 0x010F,
        (char) 0x0111, (char) 0x0144, (char) 0x0148, (char) 0x00F3, (char) 0x00F4, (char) 0x0151, (char) 0x00F6, (char) 0x00F7,
        (char) 0x0159, (char) 0x016F, (char) 0x00FA, (char) 0x0171, (char) 0x00FC, (char) 0x00FD, (char) 0x0163, (char) 0x02D9
    };

    /*---------------------------------------------------------------------------*/
    /*------ CP1251 Encoding ----------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_CP1251 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x0402, (char) 0x0403, (char) 0x201A, (char) 0x0453, (char) 0x201E, (char) 0x2026, (char) 0x2020, (char) 0x2021,
        (char) 0x20AC, (char) 0x2030, (char) 0x0409, (char) 0x2039, (char) 0x040A, (char) 0x040C, (char) 0x040B, (char) 0x040F,
        (char) 0x0452, (char) 0x2018, (char) 0x2019, (char) 0x201C, (char) 0x201D, (char) 0x2022, (char) 0x2013, (char) 0x2014,
        (char) 0x0000, (char) 0x2122, (char) 0x0459, (char) 0x203A, (char) 0x045A, (char) 0x045C, (char) 0x045B, (char) 0x045F,
        (char) 0x00A0, (char) 0x040E, (char) 0x045E, (char) 0x0408, (char) 0x00A4, (char) 0x0490, (char) 0x00A6, (char) 0x00A7,
        (char) 0x0401, (char) 0x00A9, (char) 0x0404, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x00AE, (char) 0x0407,
        (char) 0x00B0, (char) 0x00B1, (char) 0x0406, (char) 0x0456, (char) 0x0491, (char) 0x00B5, (char) 0x00B6, (char) 0x00B7,
        (char) 0x0451, (char) 0x2116, (char) 0x0454, (char) 0x00BB, (char) 0x0458, (char) 0x0405, (char) 0x0455, (char) 0x0457,
        (char) 0x0410, (char) 0x0411, (char) 0x0412, (char) 0x0413, (char) 0x0414, (char) 0x0415, (char) 0x0416, (char) 0x0417,
        (char) 0x0418, (char) 0x0419, (char) 0x041A, (char) 0x041B, (char) 0x041C, (char) 0x041D, (char) 0x041E, (char) 0x041F,
        (char) 0x0420, (char) 0x0421, (char) 0x0422, (char) 0x0423, (char) 0x0424, (char) 0x0425, (char) 0x0426, (char) 0x0427,
        (char) 0x0428, (char) 0x0429, (char) 0x042A, (char) 0x042B, (char) 0x042C, (char) 0x042D, (char) 0x042E, (char) 0x042F,
        (char) 0x0430, (char) 0x0431, (char) 0x0432, (char) 0x0433, (char) 0x0434, (char) 0x0435, (char) 0x0436, (char) 0x0437,
        (char) 0x0438, (char) 0x0439, (char) 0x043A, (char) 0x043B, (char) 0x043C, (char) 0x043D, (char) 0x043E, (char) 0x043F,
        (char) 0x0440, (char) 0x0441, (char) 0x0442, (char) 0x0443, (char) 0x0444, (char) 0x0445, (char) 0x0446, (char) 0x0447,
        (char) 0x0448, (char) 0x0449, (char) 0x044A, (char) 0x044B, (char) 0x044C, (char) 0x044D, (char) 0x044E, (char) 0x044F
    };

    /*---------------------------------------------------------------------------*/
    /*------ CP1252 Encoding ----------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_CP1252 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x20AC, (char) 0x0000, (char) 0x201A, (char) 0x0192, (char) 0x201E, (char) 0x2026, (char) 0x2020, (char) 0x2021,
        (char) 0x02C6, (char) 0x2030, (char) 0x0160, (char) 0x2039, (char) 0x0152, (char) 0x0000, (char) 0x017D, (char) 0x0000,
        (char) 0x0000, (char) 0x2018, (char) 0x2019, (char) 0x201C, (char) 0x201D, (char) 0x2022, (char) 0x2013, (char) 0x2014,
        (char) 0x02DC, (char) 0x2122, (char) 0x0161, (char) 0x203A, (char) 0x0153, (char) 0x0000, (char) 0x017E, (char) 0x0178,
        (char) 0x00A0, (char) 0x00A1, (char) 0x00A2, (char) 0x00A3, (char) 0x00A4, (char) 0x00A5, (char) 0x00A6, (char) 0x00A7,
        (char) 0x00A8, (char) 0x00A9, (char) 0x00AA, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x00AE, (char) 0x00AF,
        (char) 0x00B0, (char) 0x00B1, (char) 0x00B2, (char) 0x00B3, (char) 0x00B4, (char) 0x00B5, (char) 0x00B6, (char) 0x00B7,
        (char) 0x00B8, (char) 0x00B9, (char) 0x00BA, (char) 0x00BB, (char) 0x00BC, (char) 0x00BD, (char) 0x00BE, (char) 0x00BF,
        (char) 0x00C0, (char) 0x00C1, (char) 0x00C2, (char) 0x00C3, (char) 0x00C4, (char) 0x00C5, (char) 0x00C6, (char) 0x00C7,
        (char) 0x00C8, (char) 0x00C9, (char) 0x00CA, (char) 0x00CB, (char) 0x00CC, (char) 0x00CD, (char) 0x00CE, (char) 0x00CF,
        (char) 0x00D0, (char) 0x00D1, (char) 0x00D2, (char) 0x00D3, (char) 0x00D4, (char) 0x00D5, (char) 0x00D6, (char) 0x00D7,
        (char) 0x00D8, (char) 0x00D9, (char) 0x00DA, (char) 0x00DB, (char) 0x00DC, (char) 0x00DD, (char) 0x00DE, (char) 0x00DF,
        (char) 0x00E0, (char) 0x00E1, (char) 0x00E2, (char) 0x00E3, (char) 0x00E4, (char) 0x00E5, (char) 0x00E6, (char) 0x00E7,
        (char) 0x00E8, (char) 0x00E9, (char) 0x00EA, (char) 0x00EB, (char) 0x00EC, (char) 0x00ED, (char) 0x00EE, (char) 0x00EF,
        (char) 0x00F0, (char) 0x00F1, (char) 0x00F2, (char) 0x00F3, (char) 0x00F4, (char) 0x00F5, (char) 0x00F6, (char) 0x00F7,
        (char) 0x00F8, (char) 0x00F9, (char) 0x00FA, (char) 0x00FB, (char) 0x00FC, (char) 0x00FD, (char) 0x00FE, (char) 0x00FF
    };

    /*---------------------------------------------------------------------------*/
    /*------ CP1253 Encoding ----------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_CP1253 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x20AC, (char) 0x0000, (char) 0x201A, (char) 0x0192, (char) 0x201E, (char) 0x2026, (char) 0x2020, (char) 0x2021,
        (char) 0x0000, (char) 0x2030, (char) 0x0000, (char) 0x2039, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x2018, (char) 0x2019, (char) 0x201C, (char) 0x201D, (char) 0x2022, (char) 0x2013, (char) 0x2014,
        (char) 0x0000, (char) 0x2122, (char) 0x0000, (char) 0x203A, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x0385, (char) 0x0386, (char) 0x00A3, (char) 0x00A4, (char) 0x00A5, (char) 0x00A6, (char) 0x00A7,
        (char) 0x00A8, (char) 0x00A9, (char) 0x0000, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x00AE, (char) 0x2015,
        (char) 0x00B0, (char) 0x00B1, (char) 0x00B2, (char) 0x00B3, (char) 0x0384, (char) 0x00B5, (char) 0x00B6, (char) 0x00B7,
        (char) 0x0388, (char) 0x0389, (char) 0x038A, (char) 0x00BB, (char) 0x038C, (char) 0x00BD, (char) 0x038E, (char) 0x038F,
        (char) 0x0390, (char) 0x0391, (char) 0x0392, (char) 0x0393, (char) 0x0394, (char) 0x0395, (char) 0x0396, (char) 0x0397,
        (char) 0x0398, (char) 0x0399, (char) 0x039A, (char) 0x039B, (char) 0x039C, (char) 0x039D, (char) 0x039E, (char) 0x039F,
        (char) 0x03A0, (char) 0x03A1, (char) 0x0000, (char) 0x03A3, (char) 0x03A4, (char) 0x03A5, (char) 0x03A6, (char) 0x03A7,
        (char) 0x03A8, (char) 0x03A9, (char) 0x03AA, (char) 0x03AB, (char) 0x03AC, (char) 0x03AD, (char) 0x03AE, (char) 0x03AF,
        (char) 0x03B0, (char) 0x03B1, (char) 0x03B2, (char) 0x03B3, (char) 0x03B4, (char) 0x03B5, (char) 0x03B6, (char) 0x03B7,
        (char) 0x03B8, (char) 0x03B9, (char) 0x03BA, (char) 0x03BB, (char) 0x03BC, (char) 0x03BD, (char) 0x03BE, (char) 0x03BF,
        (char) 0x03C0, (char) 0x03C1, (char) 0x03C2, (char) 0x03C3, (char) 0x03C4, (char) 0x03C5, (char) 0x03C6, (char) 0x03C7,
        (char) 0x03C8, (char) 0x03C9, (char) 0x03CA, (char) 0x03CB, (char) 0x03CC, (char) 0x03CD, (char) 0x03CE, (char) 0x0000
    };

    /*---------------------------------------------------------------------------*/
    /*------ CP1254 Encoding ----------------------------------------------------*/

    private static final char[]  HPDF_UNICODE_MAP_CP1254 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x20AC, (char) 0x0000, (char) 0x201A, (char) 0x0192, (char) 0x201E, (char) 0x2026, (char) 0x2020, (char) 0x2021,
        (char) 0x02C6, (char) 0x2030, (char) 0x0160, (char) 0x2039, (char) 0x0152, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x2018, (char) 0x2019, (char) 0x201C, (char) 0x201D, (char) 0x2022, (char) 0x2013, (char) 0x2014,
        (char) 0x02DC, (char) 0x2122, (char) 0x0161, (char) 0x203A, (char) 0x0153, (char) 0x0000, (char) 0x0000, (char) 0x0178,
        (char) 0x00A0, (char) 0x00A1, (char) 0x00A2, (char) 0x00A3, (char) 0x00A4, (char) 0x00A5, (char) 0x00A6, (char) 0x00A7,
        (char) 0x00A8, (char) 0x00A9, (char) 0x00AA, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x00AE, (char) 0x00AF,
        (char) 0x00B0, (char) 0x00B1, (char) 0x00B2, (char) 0x00B3, (char) 0x00B4, (char) 0x00B5, (char) 0x00B6, (char) 0x00B7,
        (char) 0x00B8, (char) 0x00B9, (char) 0x00BA, (char) 0x00BB, (char) 0x00BC, (char) 0x00BD, (char) 0x00BE, (char) 0x00BF,
        (char) 0x00C0, (char) 0x00C1, (char) 0x00C2, (char) 0x00C3, (char) 0x00C4, (char) 0x00C5, (char) 0x00C6, (char) 0x00C7,
        (char) 0x00C8, (char) 0x00C9, (char) 0x00CA, (char) 0x00CB, (char) 0x00CC, (char) 0x00CD, (char) 0x00CE, (char) 0x00CF,
        (char) 0x011E, (char) 0x00D1, (char) 0x00D2, (char) 0x00D3, (char) 0x00D4, (char) 0x00D5, (char) 0x00D6, (char) 0x00D7,
        (char) 0x00D8, (char) 0x00D9, (char) 0x00DA, (char) 0x00DB, (char) 0x00DC, (char) 0x0130, (char) 0x015E, (char) 0x00DF,
        (char) 0x00E0, (char) 0x00E1, (char) 0x00E2, (char) 0x00E3, (char) 0x00E4, (char) 0x00E5, (char) 0x00E6, (char) 0x00E7,
        (char) 0x00E8, (char) 0x00E9, (char) 0x00EA, (char) 0x00EB, (char) 0x00EC, (char) 0x00ED, (char) 0x00EE, (char) 0x00EF,
        (char) 0x011F, (char) 0x00F1, (char) 0x00F2, (char) 0x00F3, (char) 0x00F4, (char) 0x00F5, (char) 0x00F6, (char) 0x00F7,
        (char) 0x00F8, (char) 0x00F9, (char) 0x00FA, (char) 0x00FB, (char) 0x00FC, (char) 0x0131, (char) 0x015F, (char) 0x00FF
    };

    /*---------------------------------------------------------------------------*/
    /*------ CP1255 Encoding ----------------------------------------------------*/

    private static final char[]  HPDF_UNICODE_MAP_CP1255 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x20AC, (char) 0x0000, (char) 0x201A, (char) 0x0192, (char) 0x201E, (char) 0x2026, (char) 0x2020, (char) 0x2021,
        (char) 0x02C6, (char) 0x2030, (char) 0x0000, (char) 0x2039, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x2018, (char) 0x2019, (char) 0x201C, (char) 0x201D, (char) 0x2022, (char) 0x2013, (char) 0x2014,
        (char) 0x02DC, (char) 0x2122, (char) 0x0000, (char) 0x203A, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x00A0, (char) 0x00A1, (char) 0x00A2, (char) 0x00A3, (char) 0x20AA, (char) 0x00A5, (char) 0x00A6, (char) 0x00A7,
        (char) 0x00A8, (char) 0x00A9, (char) 0x00D7, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x00AE, (char) 0x00AF,
        (char) 0x00B0, (char) 0x00B1, (char) 0x00B2, (char) 0x00B3, (char) 0x00B4, (char) 0x00B5, (char) 0x00B6, (char) 0x00B7,
        (char) 0x00B8, (char) 0x00B9, (char) 0x00F7, (char) 0x00BB, (char) 0x00BC, (char) 0x00BD, (char) 0x00BE, (char) 0x00BF,
        (char) 0x05B0, (char) 0x05B1, (char) 0x05B2, (char) 0x05B3, (char) 0x05B4, (char) 0x05B5, (char) 0x05B6, (char) 0x05B7,
        (char) 0x05B8, (char) 0x05B9, (char) 0x0000, (char) 0x05BB, (char) 0x05BC, (char) 0x05BD, (char) 0x05BE, (char) 0x05BF,
        (char) 0x05C0, (char) 0x05C1, (char) 0x05C2, (char) 0x05C3, (char) 0x05F0, (char) 0x05F1, (char) 0x05F2, (char) 0x05F3,
        (char) 0x05F4, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x05D0, (char) 0x05D1, (char) 0x05D2, (char) 0x05D3, (char) 0x05D4, (char) 0x05D5, (char) 0x05D6, (char) 0x05D7,
        (char) 0x05D8, (char) 0x05D9, (char) 0x05DA, (char) 0x05DB, (char) 0x05DC, (char) 0x05DD, (char) 0x05DE, (char) 0x05DF,
        (char) 0x05E0, (char) 0x05E1, (char) 0x05E2, (char) 0x05E3, (char) 0x05E4, (char) 0x05E5, (char) 0x05E6, (char) 0x05E7,
        (char) 0x05E8, (char) 0x05E9, (char) 0x05EA, (char) 0x0000, (char) 0x0000, (char) 0x200E, (char) 0x200F, (char) 0x0000
    };

    /*---------------------------------------------------------------------------*/
    /*------ CP1256 Encoding ----------------------------------------------------*/

    private static final char[]  HPDF_UNICODE_MAP_CP1256 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x20AC, (char) 0x067E, (char) 0x201A, (char) 0x0192, (char) 0x201E, (char) 0x2026, (char) 0x2020, (char) 0x2021,
        (char) 0x02C6, (char) 0x2030, (char) 0x0679, (char) 0x2039, (char) 0x0152, (char) 0x0686, (char) 0x0698, (char) 0x0688,
        (char) 0x06AF, (char) 0x2018, (char) 0x2019, (char) 0x201C, (char) 0x201D, (char) 0x2022, (char) 0x2013, (char) 0x2014,
        (char) 0x06A9, (char) 0x2122, (char) 0x0691, (char) 0x203A, (char) 0x0153, (char) 0x200C, (char) 0x200D, (char) 0x06BA,
        (char) 0x00A0, (char) 0x060C, (char) 0x00A2, (char) 0x00A3, (char) 0x00A4, (char) 0x00A5, (char) 0x00A6, (char) 0x00A7,
        (char) 0x00A8, (char) 0x00A9, (char) 0x06BE, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x00AE, (char) 0x00AF,
        (char) 0x00B0, (char) 0x00B1, (char) 0x00B2, (char) 0x00B3, (char) 0x00B4, (char) 0x00B5, (char) 0x00B6, (char) 0x00B7,
        (char) 0x00B8, (char) 0x00B9, (char) 0x061B, (char) 0x00BB, (char) 0x00BC, (char) 0x00BD, (char) 0x00BE, (char) 0x061F,
        (char) 0x06C1, (char) 0x0621, (char) 0x0622, (char) 0x0623, (char) 0x0624, (char) 0x0625, (char) 0x0626, (char) 0x0627,
        (char) 0x0628, (char) 0x0629, (char) 0x062A, (char) 0x062B, (char) 0x062C, (char) 0x062D, (char) 0x062E, (char) 0x062F,
        (char) 0x0630, (char) 0x0631, (char) 0x0632, (char) 0x0633, (char) 0x0634, (char) 0x0635, (char) 0x0636, (char) 0x00D7,
        (char) 0x0637, (char) 0x0638, (char) 0x0639, (char) 0x063A, (char) 0x0640, (char) 0x0641, (char) 0x0642, (char) 0x0643,
        (char) 0x00E0, (char) 0x0644, (char) 0x00E2, (char) 0x0645, (char) 0x0646, (char) 0x0647, (char) 0x0648, (char) 0x00E7,
        (char) 0x00E8, (char) 0x00E9, (char) 0x00EA, (char) 0x00EB, (char) 0x0649, (char) 0x064A, (char) 0x00EE, (char) 0x00EF,
        (char) 0x064B, (char) 0x064C, (char) 0x064D, (char) 0x064E, (char) 0x00F4, (char) 0x064F, (char) 0x0650, (char) 0x00F7,
        (char) 0x0651, (char) 0x00F9, (char) 0x0652, (char) 0x00FB, (char) 0x00FC, (char) 0x200E, (char) 0x200F, (char) 0x06D2
    };

    /*---------------------------------------------------------------------------*/
    /*------ CP1257 Encoding ----------------------------------------------------*/

    private static final char[]  HPDF_UNICODE_MAP_CP1257 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x20AC, (char) 0x0000, (char) 0x201A, (char) 0x0000, (char) 0x201E, (char) 0x2026, (char) 0x2020, (char) 0x2021,
        (char) 0x0000, (char) 0x2030, (char) 0x0000, (char) 0x2039, (char) 0x0000, (char) 0x00A8, (char) 0x02C7, (char) 0x00B8,
        (char) 0x0000, (char) 0x2018, (char) 0x2019, (char) 0x201C, (char) 0x201D, (char) 0x2022, (char) 0x2013, (char) 0x2014,
        (char) 0x0000, (char) 0x2122, (char) 0x0000, (char) 0x203A, (char) 0x0000, (char) 0x00AF, (char) 0x02DB, (char) 0x0000,
        (char) 0x00A0, (char) 0x0000, (char) 0x00A2, (char) 0x00A3, (char) 0x00A4, (char) 0x0000, (char) 0x00A6, (char) 0x00A7,
        (char) 0x00D8, (char) 0x00A9, (char) 0x0156, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x00AE, (char) 0x00C6,
        (char) 0x00B0, (char) 0x00B1, (char) 0x00B2, (char) 0x00B3, (char) 0x00B4, (char) 0x00B5, (char) 0x00B6, (char) 0x00B7,
        (char) 0x00F8, (char) 0x00B9, (char) 0x0157, (char) 0x00BB, (char) 0x00BC, (char) 0x00BD, (char) 0x00BE, (char) 0x00E6,
        (char) 0x0104, (char) 0x012E, (char) 0x0100, (char) 0x0106, (char) 0x00C4, (char) 0x00C5, (char) 0x0118, (char) 0x0112,
        (char) 0x010C, (char) 0x00C9, (char) 0x0179, (char) 0x0116, (char) 0x0122, (char) 0x0136, (char) 0x012A, (char) 0x013B,
        (char) 0x0160, (char) 0x0143, (char) 0x0145, (char) 0x00D3, (char) 0x014C, (char) 0x00D5, (char) 0x00D6, (char) 0x00D7,
        (char) 0x0172, (char) 0x0141, (char) 0x015A, (char) 0x016A, (char) 0x00DC, (char) 0x017B, (char) 0x017D, (char) 0x00DF,
        (char) 0x0105, (char) 0x012F, (char) 0x0101, (char) 0x0107, (char) 0x00E4, (char) 0x00E5, (char) 0x0119, (char) 0x0113,
        (char) 0x010D, (char) 0x00E9, (char) 0x017A, (char) 0x0117, (char) 0x0123, (char) 0x0137, (char) 0x012B, (char) 0x013C,
        (char) 0x0161, (char) 0x0144, (char) 0x0146, (char) 0x00F3, (char) 0x014D, (char) 0x00F5, (char) 0x00F6, (char) 0x00F7,
        (char) 0x0173, (char) 0x0142, (char) 0x015B, (char) 0x016B, (char) 0x00FC, (char) 0x017C, (char) 0x017E, (char) 0x02D9
    };

    /*---------------------------------------------------------------------------*/
    /*------ CP1258 Encoding ----------------------------------------------------*/

    private static final char[]  HPDF_UNICODE_MAP_CP1258 = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x20AC, (char) 0x0000, (char) 0x201A, (char) 0x0192, (char) 0x201E, (char) 0x2026, (char) 0x2020, (char) 0x2021,
        (char) 0x02C6, (char) 0x2030, (char) 0x0000, (char) 0x2039, (char) 0x0152, (char) 0x0000, (char) 0x0000, (char) 0x0000,
        (char) 0x0000, (char) 0x2018, (char) 0x2019, (char) 0x201C, (char) 0x201D, (char) 0x2022, (char) 0x2013, (char) 0x2014,
        (char) 0x02DC, (char) 0x2122, (char) 0x0000, (char) 0x203A, (char) 0x0153, (char) 0x0000, (char) 0x0000, (char) 0x0178,
        (char) 0x00A0, (char) 0x00A1, (char) 0x00A2, (char) 0x00A3, (char) 0x00A4, (char) 0x00A5, (char) 0x00A6, (char) 0x00A7,
        (char) 0x00A8, (char) 0x00A9, (char) 0x00AA, (char) 0x00AB, (char) 0x00AC, (char) 0x00AD, (char) 0x00AE, (char) 0x00AF,
        (char) 0x00B0, (char) 0x00B1, (char) 0x00B2, (char) 0x00B3, (char) 0x00B4, (char) 0x00B5, (char) 0x00B6, (char) 0x00B7,
        (char) 0x00B8, (char) 0x00B9, (char) 0x00BA, (char) 0x00BB, (char) 0x00BC, (char) 0x00BD, (char) 0x00BE, (char) 0x00BF,
        (char) 0x00C0, (char) 0x00C1, (char) 0x00C2, (char) 0x0102, (char) 0x00C4, (char) 0x00C5, (char) 0x00C6, (char) 0x00C7,
        (char) 0x00C8, (char) 0x00C9, (char) 0x00CA, (char) 0x00CB, (char) 0x0300, (char) 0x00CD, (char) 0x00CE, (char) 0x00CF,
        (char) 0x0110, (char) 0x00D1, (char) 0x0309, (char) 0x00D3, (char) 0x00D4, (char) 0x01A0, (char) 0x00D6, (char) 0x00D7,
        (char) 0x00D8, (char) 0x00D9, (char) 0x00DA, (char) 0x00DB, (char) 0x00DC, (char) 0x01AF, (char) 0x0303, (char) 0x00DF,
        (char) 0x00E0, (char) 0x00E1, (char) 0x00E2, (char) 0x0103, (char) 0x00E4, (char) 0x00E5, (char) 0x00E6, (char) 0x00E7,
        (char) 0x00E8, (char) 0x00E9, (char) 0x00EA, (char) 0x00EB, (char) 0x0301, (char) 0x00ED, (char) 0x00EE, (char) 0x00EF,
        (char) 0x0111, (char) 0x00F1, (char) 0x0323, (char) 0x00F3, (char) 0x00F4, (char) 0x01A1, (char) 0x00F6, (char) 0x00F7,
        (char) 0x00F8, (char) 0x00F9, (char) 0x00FA, (char) 0x00FB, (char) 0x00FC, (char) 0x01B0, (char) 0x20AB, (char) 0x00FF
    };

    /*---------------------------------------------------------------------------*/
    /*------ KOI8-R Encoding ----------------------------------------------------*/

    private static final char[] HPDF_UNICODE_MAP_KOI8_R = {
        (char) 0x0020, (char) 0x0021, (char) 0x0022, (char) 0x0023, (char) 0x0024, (char) 0x0025, (char) 0x0026, (char) 0x0027,
        (char) 0x0028, (char) 0x0029, (char) 0x002A, (char) 0x002B, (char) 0x002C, (char) 0x002D, (char) 0x002E, (char) 0x002F,
        (char) 0x0030, (char) 0x0031, (char) 0x0032, (char) 0x0033, (char) 0x0034, (char) 0x0035, (char) 0x0036, (char) 0x0037,
        (char) 0x0038, (char) 0x0039, (char) 0x003A, (char) 0x003B, (char) 0x003C, (char) 0x003D, (char) 0x003E, (char) 0x003F,
        (char) 0x0040, (char) 0x0041, (char) 0x0042, (char) 0x0043, (char) 0x0044, (char) 0x0045, (char) 0x0046, (char) 0x0047,
        (char) 0x0048, (char) 0x0049, (char) 0x004A, (char) 0x004B, (char) 0x004C, (char) 0x004D, (char) 0x004E, (char) 0x004F,
        (char) 0x0050, (char) 0x0051, (char) 0x0052, (char) 0x0053, (char) 0x0054, (char) 0x0055, (char) 0x0056, (char) 0x0057,
        (char) 0x0058, (char) 0x0059, (char) 0x005A, (char) 0x005B, (char) 0x005C, (char) 0x005D, (char) 0x005E, (char) 0x005F,
        (char) 0x0060, (char) 0x0061, (char) 0x0062, (char) 0x0063, (char) 0x0064, (char) 0x0065, (char) 0x0066, (char) 0x0067,
        (char) 0x0068, (char) 0x0069, (char) 0x006A, (char) 0x006B, (char) 0x006C, (char) 0x006D, (char) 0x006E, (char) 0x006F,
        (char) 0x0070, (char) 0x0071, (char) 0x0072, (char) 0x0073, (char) 0x0074, (char) 0x0075, (char) 0x0076, (char) 0x0077,
        (char) 0x0078, (char) 0x0079, (char) 0x007A, (char) 0x007B, (char) 0x007C, (char) 0x007D, (char) 0x007E, (char) 0x007F,
        (char) 0x2500, (char) 0x2502, (char) 0x250C, (char) 0x2510, (char) 0x2514, (char) 0x2518, (char) 0x251C, (char) 0x2524,
        (char) 0x252C, (char) 0x2534, (char) 0x253C, (char) 0x2580, (char) 0x2584, (char) 0x2588, (char) 0x258C, (char) 0x2590,
        (char) 0x2591, (char) 0x2592, (char) 0x2593, (char) 0x2320, (char) 0x25A0, (char) 0x2219, (char) 0x221A, (char) 0x2248,
        (char) 0x2264, (char) 0x2265, (char) 0x00A0, (char) 0x2321, (char) 0x00B0, (char) 0x00B2, (char) 0x00B7, (char) 0x00F7,
        (char) 0x2550, (char) 0x2551, (char) 0x2552, (char) 0x0451, (char) 0x2553, (char) 0x2554, (char) 0x2555, (char) 0x2556,
        (char) 0x2557, (char) 0x2558, (char) 0x2559, (char) 0x255A, (char) 0x255B, (char) 0x255C, (char) 0x255D, (char) 0x255E,
        (char) 0x255F, (char) 0x2560, (char) 0x2561, (char) 0x0401, (char) 0x2562, (char) 0x2563, (char) 0x2564, (char) 0x2565,
        (char) 0x2566, (char) 0x2567, (char) 0x2568, (char) 0x2569, (char) 0x256A, (char) 0x256B, (char) 0x256C, (char) 0x00A9,
        (char) 0x044E, (char) 0x0430, (char) 0x0431, (char) 0x0446, (char) 0x0434, (char) 0x0435, (char) 0x0444, (char) 0x0433,
        (char) 0x0445, (char) 0x0438, (char) 0x0439, (char) 0x043A, (char) 0x043B, (char) 0x043C, (char) 0x043D, (char) 0x043E,
        (char) 0x043F, (char) 0x044F, (char) 0x0440, (char) 0x0441, (char) 0x0442, (char) 0x0443, (char) 0x0436, (char) 0x0432,
        (char) 0x044C, (char) 0x044B, (char) 0x0437, (char) 0x0448, (char) 0x044D, (char) 0x0449, (char) 0x0447, (char) 0x044A,
        (char) 0x042E, (char) 0x0410, (char) 0x0411, (char) 0x0426, (char) 0x0414, (char) 0x0415, (char) 0x0424, (char) 0x0413,
        (char) 0x0425, (char) 0x0418, (char) 0x0419, (char) 0x041A, (char) 0x041B, (char) 0x041C, (char) 0x041D, (char) 0x041E,
        (char) 0x041F, (char) 0x042F, (char) 0x0420, (char) 0x0421, (char) 0x0422, (char) 0x0423, (char) 0x0416, (char) 0x0412,
        (char) 0x042C, (char) 0x042B, (char) 0x0417, (char) 0x0428, (char) 0x042D, (char) 0x0429, (char) 0x0427, (char) 0x042A
    };
    
}
