/*
 * JHPdf Free PDF Library : CharacterMap.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * original demo/character_map.c in 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 * usage character_map <encoding-name> <low-range-from> <low-range-to>
 *              <high-range-from> <high-range-to>
 * ex. character_map 90ms-RKSJ-V 0x80 0x
 *
 */

import net.sf.jhpdf.*;
import net.sf.jhpdf.pdfobject.*;
import net.sf.jhpdf.pdfobject.HPdfCatalog.HPdfPageMode;
import net.sf.jhpdf.pdfobject.HPdfDocument.HPdfCompressionMode;
import net.sf.jhpdf.encoder.HPdfEncoder;
import net.sf.jhpdf.encoder.HPdfEncoder.HPdfByteType;
import net.sf.jhpdf.encoder.HPdfEncoder.HPdfEncoderType;


public class CharacterMap  implements DemoEntry {
    public static void main(String[] args) {
        new CharacterMap().demo(args);
    }
    
    private static final String F_NAME = "CharacterMap.pdf";
    
    @Override
    public void demo(String[] args) {
        
        if (args.length < 2) {
            System.out.println("usage: java CharacterMap <encoding-name> <font-name>");
            return;
        }
        
        HPdfDocument pdf = null;
        try {
            pdf = JHPdf.createDocument();
            
            /* configure pdf-document (showing outline, compression enabled) */
            pdf.setPageMode(HPdfPageMode.USE_OUTLINE);
            pdf.setCompressionMode(HPdfCompressionMode.ALL);
            pdf.setPagesConfiguration(10);
            
            pdf.useJPEncodings();
            pdf.useJPFonts();
            pdf.useKREncodings();
            pdf.useKRFonts();
            pdf.useCNSEncodings();
            pdf.useCNSFonts();
            pdf.useCNTEncodings();
            pdf.useCNTFonts();
            
            HPdfEncoder encoder = pdf.getEncoder(args[0]);
            if (encoder.getType() != HPdfEncoderType.DOUBLE_BYTE) {
                System.out.println("error: " + args[0] + " is not cmap-encoder");
                return;
            }
            
            HPdfFont font = pdf.getFont(args[1], args[0]);
            
            int minL = 255;
            int minH = 256;
            int maxL = 0;
            int maxH = 0;
            
            boolean[] flg = new boolean[256];
            for (int i = 0; i < flg.length; ++i) {
                flg[i] = false;
            }
            
            for (int i = 0; i <= 255; ++i) {
                for (int j = 20; j <= 255; ++j) {
                    byte[] buf = new byte[2];
                    int code = i * 256 + j;
                    
                    buf[0] = (byte)i;
                    buf[1] = (byte)j;
                    
                    HPdfByteType btype = encoder.getByteType(buf, 0);
                    char unicode = encoder.toUnicode(code);
                    
                    if (btype == HPdfByteType.LEAD && unicode != '\u25a1') {
                        if (minL > j) {
                            minL = j;
                        }
                        
                        if (maxL < j) {
                            maxL = j;
                        }
                        
                        if (minH > i) {
                            minH = i;
                        }
                        
                        if (maxH < i) {
                            maxH = i;
                        }
                        
                        flg[i] = true;
                    }
                }
            }
            
            System.out.println(String.format("min_h=%04X max_h=%04X min_l=%04X max_l=%04X",
                minH, maxH, minL, maxL));
            
            /* create outline root. */
            HPdfOutline root = pdf.createOutline(null, args[0], null);
            root.setOpened(true);
            
            for (int i = 0; i <= 255; ++i) {
                if (flg[i]) {
                    HPdfPage page = pdf.addPage();
                    HPdfFont titleFont = pdf.getFont("Helvetica", null);
                    
                    String buf = String.format("0x%04X-0x%04X", 
                        i * 256 + minL, i * 256 + maxL);
                    
                    HPdfOutline outline = pdf.createOutline(root, buf, null);
                    HPdfDestination dst = page.createDestination();
                    outline.setDestination(dst);
                    
                    drawPage(pdf, page, titleFont, font, i, minL);
                    
                    buf = String.format("%s (%s) 0x%04X-0x%04X", args[0], args[1],
                        i * 256 + minL, i * 256 + maxL);
                    
                    page.setFontAndSize(titleFont, 10);
                    page.beginText();
                    page.moveTextPos(40, page.getHeight() - 35);
                    page.showText(buf);
                    page.endText();
                }
            }
            
            /* save the document to a file */
            pdf.saveToFile(F_NAME);
            
        } catch (HPdfException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (pdf != null) {
                pdf.freeDocAll();
            }
        }
    }
    
    private void drawPage(HPdfDocument pdf, HPdfPage page, HPdfFont titleFont, HPdfFont font, int hByte, int lByte) {
        final int PAGE_WIDTH = 420;
        final int CELL_HEIGHT = 20;
        final int CELL_WIDTH = 20;
        
        lByte = (lByte / 16) * 16;
        int hCount = 16 - (lByte / 16);
        int pageHeight = 40 + 40 + (hCount + 1) * CELL_HEIGHT;
        
        page.setHeight(pageHeight);
        page.setWidth(PAGE_WIDTH);
        
        page.setFontAndSize(titleFont, 10);
        
        int ypos = hCount + 1;
        for (;;) {
            int y = ypos * CELL_HEIGHT + 40;
            
            page.moveTo(40, y);
            page.lineTo(380, y);
            page.stroke();
            
            if (ypos < hCount) {
                char c;
                
                int code = 16 - ypos - 1;
                if (code < 10) {
                    c = (char)('0' + code);
                } else {
                    c = (char) ('A' + (code - 10));
                }
                String buf = String.valueOf(c);
                
                float w = page.textWidth(buf);
                page.beginText();
                page.moveTextPos(40 + (20 - w) / 2, y + 5);
                page.showText(buf);
                page.endText();
            }
            
            if (ypos == 0) {
                break;
            }
            
            --ypos;
        }
        
        for (int xpos = 0; xpos <= 17; ++xpos) {
            int y = (hCount + 1) * CELL_HEIGHT + 40;
            int x = xpos * CELL_WIDTH + 40;
            
            page.moveTo(x, 40);
            page.lineTo(x, y);
            page.stroke();
            
            if (xpos > 0 && xpos <= 16) {
                char c;
                
                int code = xpos - 1;
                if (code < 10) {
                    c = (char) ('0' + code);
                } else {
                    c = (char) ('A' + (code - 10));
                }
                String buf = String.valueOf(c);
                
                float w = page.textWidth(buf);
                page.beginText();
                page.moveTextPos(x + (20 - w) / 2, hCount * CELL_HEIGHT + 45);
                page.showText(buf);
                page.endText();
            }
        }
        
        page.setFontAndSize(font, 15);
        
        ypos = hCount;
        for (;;) {
            int y = (ypos - 1) * CELL_HEIGHT + 45;
            
            for (int xpos = 0; xpos < 16; ++xpos) {
                byte[] bbuf = new byte[2]; // memo: assumes double-byte char encoding
                
                int x = xpos * CELL_WIDTH + 40 + CELL_WIDTH;
                
                bbuf[0] = (byte) hByte;
                bbuf[1] = (byte) ((16 - ypos) * 16 + xpos);
                // memo: HPdfFont#getEncoder(), HPdfEncoder#decode()
                // they are both JHPdf specific method.
                String buf = font.getEncoder().decode(bbuf);
                
                float w = page.textWidth(buf);
                if (w > 0) {
                    page.beginText();
                    page.moveTextPos(x + (20 - w) / 2, y);
                    page.showText(buf);
                    page.endText();
                }
            }
            
            if (ypos == 0) {
                break;
            }
            
            --ypos;
        }
    }
}
