﻿#include <stdio.h>
#define _USE_MATH_DEFINES
#include <math.h>

#define ROWS 100			//計算領域の最大行数

//定数
const double CD = 0.5;
const double rho = 1.205;	//[kg/(m^3)]
const double A = 0.0004;	//[m^2]

const double dt = 1.;		//時間刻み幅[s]
const double g = 9.8;		//重力加速度[m/(s^2)]
const double m = 1.0;		//物体質量[kg]
const double initH = 1.0;	//初期高度[m]

typedef struct {
	double t, x, vx, ax, y, vy, ay;
} Tuple;

const Tuple initialTuple = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};	//ゼロ初期化用

typedef struct {
	int theta, v0;					//入力
	Tuple row[ROWS];				//逐次計算領域
	double flyingTime, maxHeight;	//結果
	int resultID;
} DataFrame;

double drag(double v);								//空気抵抗の計算
void inputData(DataFrame *tab);						//データ入力
void initializeState(DataFrame *tab);				//初期状態を設定
int simurate(DataFrame *tab);						//シミュレーション実行
int writeCSV(const char* filename, DataFrame *tab);	//CSVファイルに出力

int main(int argc, char** argv)
{
	DataFrame Table;

	//データ入力
	inputData(&Table);
	//初期状態設定
	initializeState(&Table);

	//シミュレーション実行
	simurate(&Table);

	//結果出力
	printf("滞空時間：%.3f[s]\n", Table.row[Table.resultID].t);
	printf("到達距離：%.3f[m]\n", Table.row[Table.resultID].x);
	printf("最高高度：%.3f[m]\n", Table.maxHeight);

	writeCSV("out.csv", &Table);

	return 0;
}

double drag(double v)
/*-------------------------------------------------------------
速度vから空気抵抗を求める
返り値：抗力F（vの逆向き）
--------------------------------------------------------------*/
{
	return -0.5 * CD * rho * A * v * fabs(v);
}

void inputData(DataFrame *tab)
/*-------------------------------------------------------------
θ[dgree] と v0[km/h] を入力
各適切な範囲の値が入力されるまでループ
入力れた値はtabに登録
--------------------------------------------------------------*/
{
	char buf[64];
	int tmp = -1;
	printf("θ[dgree] と v0[km/h] を入力してください\n");
	
	while(1){
		printf("θ= ") ;
		fgets(buf, 63, stdin);
		if (sscanf(buf, "%d", &tmp) < 1)
			printf("error! \n");
		else if (tmp < 0 || tmp > 90)
			printf("error! \n");
		else
			tab->theta = tmp;
			break;	
	}
	
	tmp = -1;
	while(1){
		printf("v0 = ") ;
		fgets(buf, 63, stdin);
		if (sscanf(buf, "%d", &tmp) < 1)
			printf("error! \n");
		else if (tmp < 0 || tmp > 100)
			printf("error! \n");
		else
			tab->v0 = tmp;
			break;	
	}
}

void initializeState(DataFrame *tab)
/*-------------------------------------------------------------
tabに初期状態を設定
--------------------------------------------------------------*/
{
	int i;
	double v0 = (tab->v0) / 3.6;					//[km/h] -> [m/s]
	double radTheta = (tab->theta) * M_PI / 180.0;	//[dgree] -> [rad]

	//初期状態-----------------------------------------------------
	tab->row[0].t = 0.0;
	tab->row[0].x = 0.0;
	tab->row[0].y = initH;
	tab->row[0].vx = v0 * cos(radTheta);
	tab->row[0].vy = v0 * sin(radTheta);
	tab->row[0].ax = 0.0;
	tab->row[0].ay = -g;
	//-------------------------------------------------------------

	for(i=1; i<ROWS; ++i){
		tab->row[i] = initialTuple;	//ゼロ初期化
	}

	tab->maxHeight = tab->flyingTime = 0.0;
	tab->resultID = 0;
}

int simurate(DataFrame *tab)
/*-------------------------------------------------------------
シミュレーションを実行
dt毎の各時刻tについて x, vx, ax, y, vy, ayを計算
返り値：計算範囲（最大行番号）内で終了すれば 1
--------------------------------------------------------------*/
{
	int i;
	for (i = 1; i < ROWS; ++i) {
		tab->row[i].t = tab->row[i - 1].t + dt;
		// x[n] = x[n-1] + vx[n-1]*dt
		tab->row[i].x = tab->row[i - 1].x + tab->row[i - 1].vx * dt;
		// vx[n] = vx[n-1] + ax[n-1]*dt
		tab->row[i].vx = tab->row[i - 1].vx + tab->row[i - 1].ax * dt;
		// ax[n] = (1/m)*(空気抵抗 ~ vx[n-1]) 
		tab->row[i].ax = (1 / m) * drag(tab->row[i - 1].vx);
		// y[n] = y[n-1] + vy[n-1]*dt
		tab->row[i].y = tab->row[i - 1].y + tab->row[i - 1].vy * dt;
		if (tab->row[i].y > tab->maxHeight){
			//最高高度更新
			tab->maxHeight = tab->row[i].y;
		}
		// vy[n] = vy[n-1] + ay[n-1]*dt
		tab->row[i].vy = tab->row[i - 1].vy + tab->row[i - 1].ay * dt;
		// ay[n] = -g +(1/m)*(空気抵抗 ~ vy[n-1]) 
		tab->row[i].ay = -g + (1 / m) * drag(tab->row[i - 1].vy);

		//debug：逐次各行表示
		printf("%.2f %.2f %.2f %.2f %.2f %.2f %.2f\n", tab->row[i].t, tab->row[i].x, tab->row[i].vx, tab->row[i].ax, tab->row[i].y, tab->row[i].vy, tab->row[i].ay);

		// 物体が地面に接触したとき
		if (tab->row[i].y <= 0.0){
			tab->flyingTime = tab->row[i].t;	//着地時刻
			tab->resultID = i;					//最終行番号
			return 1;
		}
	}
	return 0;
}

int writeCSV(const char* filename, DataFrame *tab)
/*-------------------------------------------------------------
t , x , y をコンマ区切りでfilenameに出力
返り値：書き込み成功で 1
--------------------------------------------------------------*/
{
	FILE *fp = fopen( filename, "w+" );
	int i;
	if( fp == NULL ){
		printf("file open error : %s\n",filename);
		return 0;
	}
	fprintf(fp, "t,x,y\n");	//ヘッダ
	for(i=0; i <= tab->resultID; ++i){
		//結果行まで各行書き出し
		fprintf(fp, "%f,%f,%f\n", tab->row[i].t, tab->row[i].x, tab->row[i].y);
	}
	printf("Output : %s\n",filename);
	fclose(fp);

	return 1;
}
