import os
import os.path
import sys

from rpython.rlib.listsort import make_timsort_class

from cammylib.arrows import BuildProblem
from cammylib.hive import Hive, MissingAtom
from cammylib.parser import parse
from cammylib.types import ConstraintStore, TypeExtractor, UnificationFailed


SortFileNames = make_timsort_class()

def codeblock(code):
    return "```\n" + code + "\n```"

def main(argv):
    hivepath = argv[-1]
    hive = Hive(hivepath)
    if not hivepath.endswith(os.sep):
        hivepath += os.sep
    prefix = len(hivepath)
    doc = []
    for dirpath, dirnames, filenames in os.walk(hivepath):
        SortFileNames(dirnames).sort()
        SortFileNames(filenames).sort()
        section = dirpath[prefix:] or "Top level"
        doc.append("# " + section)

        for filename in filenames:
            if not filename.endswith(".cammy"):
                continue
            fullpath = os.path.join(dirpath, filename)
            suffix = len(fullpath) - 6
            assert suffix >= 0, "proven by .endswith() check above"
            atom = fullpath[prefix:suffix]
            with open(fullpath, "r") as handle:
                sexp, trail = parse(handle.read())

            doc.append("## " + atom)
            doc.append(codeblock(sexp.asStr()))
            try:
                sexp = sexp.canonicalize(hive)
                arrow = sexp.buildArrow()
                cs = ConstraintStore()
                domain, codomain = arrow.types(cs)
                extractor = TypeExtractor(cs)
                for i, (gdom, gcod) in enumerate(cs.knownGivens):
                    doc.append("Given @%d : %s -> %s" % (i,
                            extractor.extract(gdom),
                            extractor.extract(gcod)))
                doc.append("Type: %s -> %s" % (extractor.extract(domain),
                        extractor.extract(codomain)))
            except MissingAtom as ma:
                doc.append(
                    "(Couldn't canonicalize due to missing expression %s)" %
                    ma.atom)
            except BuildProblem:
                doc.append("(Couldn't build arrow)")
            except UnificationFailed as uf:
                doc.append("(Couldn't typecheck: %s)" % uf.message)
            if trail:
                doc.append(trail)
    print "\n\n".join(doc)
    return 0

def target(driver, *args):
    driver.exe_name = "cammy-weave"
    return main, None

if __name__ == "__main__":
    main(sys.argv)
