﻿///////////////////////////////////////////////////////////////////////////////
// This file is part of SharpWebBrowser.
///////////////////////////////////////////////////////////////////////////////
// Copyright (C) 2009-2011 bis5 <bis5@users.sourceforge.jp>
// All rights reserved.
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the Microsoft Public Licence (Ms-PL) as published by 
// Microsoft Corp.
//
// You should have received a copy of the Microsoft Public License along 
// with this program. 
// If not, see <http://www.microsoft.com/opensource/licenses.mspx>
//
///////////////////////////////////////////////////////////////////////////////
//
// File information
// Name: SerializableDictionary.cs
// Author: bis5
//
///////////////////////////////////////////////////////////////////////////////

using System;
using System.Collections.Generic;
using System.Xml;
using System.Xml.Serialization;
using System.Xml.Schema;

namespace Bis5Products.SharpWebBrowser {
    public class SerializableDictionary<Tkey, TValue> :
        Dictionary<Tkey, TValue>, IXmlSerializable {
        /// <summary>
        /// XMLのスキーマ情報を取得します。
        /// </summary>
        /// <returns>常にnull</returns>
        public XmlSchema GetSchema() { return null; }

        /// <summary>
        /// XMLファイルを読み込みます。
        /// </summary>
        /// <param name="reader">XMLファイルからの読み込みに使う<paramref name="System.Xml.XmlReader"/>。</param>
        public void ReadXml(XmlReader reader) {
            var serializer = new XmlSerializer(typeof(KeyValueItem));

            reader.ReadStartElement("configuration");
            while (reader.NodeType != XmlNodeType.EndElement && reader.NodeType != XmlNodeType.None) {
                KeyValueItem item = (KeyValueItem)serializer.Deserialize(reader);
                if (item != null) this.Add(item.Key, item.Value);
            }
            try { reader.ReadEndElement(); } catch { }
        }

        /// <summary>
        /// XMLファイルに設定を保存します。
        /// </summary>
        /// <param name="writer">XMLファイルへの書き込みに使う<paramref name="System.Xml.XmlWeiter"/>クラスのインスタンス。</param>
        public void WriteXml(System.Xml.XmlWriter writer) {
            var serializer = new System.Xml.Serialization.XmlSerializer(typeof(KeyValueItem));
            writer.WriteStartDocument();
            writer.WriteStartElement("configuration");
            writer.WriteString("\r\n");
            foreach (var key in Keys) {
                serializer.Serialize(writer, new KeyValueItem(key, this[key]));
                writer.WriteString("\r\n");
            }

            writer.WriteEndElement();
            writer.Close();
        }


        [Serializable()]
        [XmlRoot("KeyValueItem")]
        public class KeyValueItem {
            [XmlElement("Key")]
            public Tkey Key { get; set; }
            [XmlElement("Value")]
            public TValue Value { get; set; }
            public KeyValueItem() { }
            public KeyValueItem(Tkey key, TValue value) { Key = key; Value = value; }
        }

    }// class SerializableDictionary

}// namespace Bis5Products.SharpWebBrowser