﻿///////////////////////////////////////////////////////////////////////////////
// This file is part of BookmarkManager.
///////////////////////////////////////////////////////////////////////////////
// Copyright (C) 2009-2011 bis5 <bis5@users.sourceforge.jp>
// All rights reserved.
///////////////////////////////////////////////////////////////////////////////
// BookmarkManager is one of modules for SharpWebBrowser.
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the Microsoft Public Licence (Ms-PL) as published by 
// Microsoft Corp.
//
// You should have received a copy of the Microsoft Public License along 
// with this program. 
// If not, see <http://www.microsoft.com/opensource/licenses.mspx>
//
///////////////////////////////////////////////////////////////////////////////
//
// File information
// Name: BookmarkManager.cs
// Author: bis5
// Module: BookmarkManager
//
///////////////////////////////////////////////////////////////////////////////

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Xml;
using System.Windows.Forms;
using System.Text.RegularExpressions;

namespace Bis5Products.SharpWebBrowser {
    /// <remarks>
    /// MainFormでブックマーク項目が更新されたときの動作を指定してください。
    /// </remarks>
    public delegate void BookmarKItemChangedEventHandler();

    public class BookmarkManager {

        private string apppath;

        private int count = 0;

        internal Dictionary<int, List<BookmarkItem>> bm = new Dictionary<int, List<BookmarkItem>>();
        internal ILangMgr lang;
        internal IMainControl main;

        public BookmarkManager(string path, ILangMgr language, IMainControl mainf) {
            apppath = path;
            lang = language;
            main = mainf;
            this.BookmarkItemChanged += new BookmarKItemChangedEventHandler(BookmarkManager_BookmarkItemChanged);
            Load();
        }

        void BookmarkManager_BookmarkItemChanged() {
            var tlist = GetItems().Values;
            //HasChildプロパティの設定
            foreach (var item in tlist)
                if (!bm.ContainsKey(item.Id) || bm[item.Id].Count == 0)
                    item.HasChild = false;
                else
                    item.HasChild = true;
        }


        public event BookmarKItemChangedEventHandler BookmarkItemChanged;

        public void addBookmarkMenu_Click(string name, string url) {
            var manage = new ManageDialog(this);
            var item = new BookmarkItem();
            item.Name = name;
            item.Url = url;
            item.ParentId = -1;//いらない？
            manage.ShowDialog(ManageMode.AddNew, ManageType.Item, item);
        }

        public void Add(BookmarkItem item) {
            count++;
            item.Id = count;
            try {
                if (!bm.ContainsKey(item.ParentId))
                    bm.Add(item.ParentId, new List<BookmarkItem>());
                bm[item.ParentId].Add(item);
            } catch (ArgumentException) {
                count++;
                Add(item);
                return;
            }
            BookmarkItemChanged();
        }
        public void Add(string name, string url) {
            var item = new BookmarkItem();
            item.Name = name;
            item.Url = url;
            item.Type = BookmarkItemType.Item;
            item.ParentId = -1;
            Add(item);
        }
        public void Add(string name, BookmarkItemType type) {
            if (type == BookmarkItemType.Item)
                throw new ArgumentException("type");
            var item = new BookmarkItem();
            item.Name = name;
            item.ParentId = -1;
            item.Type = type;
            item.HasChild = false;
            Add(item);
        }

        public void Set(BookmarkItem item) {
            Set(item.Id, item.Name, item.Url, item.ParentId);
        }
        /// <param name="newname">変更が無ければnullを渡す</param>
        /// <param name="url">変更がなければnullを渡す</param>
        /// <param name="parent">変更がなければ-2を指定する</param>
        public void Set(int id, string newname, string url, int parent) {
            var list = GetItems();
            if (!list.ContainsKey(id))
                throw new KeyNotFoundException("id");
            var item = list[id];
            if (!string.IsNullOrEmpty(newname))
                item.Name = newname;
            if (!string.IsNullOrEmpty(url))
                item.Url = url;
            if (parent != -2)
                item.ParentId = parent;
            BookmarkItemChanged();
        }

        private void Remove(int id) {
            foreach (KeyValuePair<int, List<BookmarkItem>> pair in bm)
                foreach (var item in pair.Value)
                    if (item.Id == id) {
                        pair.Value.Remove(item);
                        return;
                    }
        }
        public void RemoveItem(BookmarkItem item) {
            RemoveItem(item.Id);
        }
        public void RemoveItem(int id) {
            Remove(id);
            BookmarkItemChanged();
        }
        public void RemoveDirectory(BookmarkItem item) {
            RemoveDirectory(item.Id);
        }
        /// <summary>
        /// 子項目含め削除
        /// </summary>
        public void RemoveDirectory(int id) {
            if (bm.ContainsKey(id)) {
                bm.Remove(id);
            }
            Remove(id);
            BookmarkItemChanged();
        }

        public bool IsContains(int id) {
            return GetItems().ContainsKey(id);
        }

        internal Dictionary<int, BookmarkItem> GetItems() {
            var tlist = new Dictionary<int, BookmarkItem>();
            foreach (KeyValuePair<int, List<BookmarkItem>> pair in bm) {
                foreach (BookmarkItem item in pair.Value)
                    tlist.Add(item.Id, item);
            }
            return tlist;
        }

        private void Load() {
            var fs = new FileStream(Path.Combine(apppath, "Bookmark.config"), FileMode.OpenOrCreate, FileAccess.Read);
            var reader = XmlReader.Create(fs);

            try {
                reader.ReadStartElement("BookmarkRoot");
            } catch { }

            var tlist = new List<BookmarkItem>();
            while (true) {
                if (reader.NodeType == XmlNodeType.Whitespace) {
                    reader.Read();
                    continue;
                }
                if (reader.NodeType == XmlNodeType.EndElement ||
                    reader.NodeType == XmlNodeType.None)
                    break;
                if (reader.NodeType == XmlNodeType.Element) {
                    if (reader.Name == "BookmarkRoot") {
                        reader.Read();
                        continue;
                    }

                    if (reader.Name == "dir") {
                        var item = new BookmarkItem();
                        item.Name = reader.GetAttribute("Name");
                        item.Type = BookmarkItemType.Dir;
                        var tmp = reader.GetAttribute("ID");
                        item.Id = int.Parse(reader.GetAttribute("ID"));
                        item.ParentId = int.Parse(reader.GetAttribute("PID"));
                        tlist.Add(item);
                    } else if (reader.Name == "item") {
                        var item = new BookmarkItem();
                        item.Name = reader.GetAttribute("Name");
                        item.Url = reader.GetAttribute("Url");
                        item.Type = BookmarkItemType.Item;
                        item.Id = int.Parse(reader.GetAttribute("ID"));
                        item.ParentId = int.Parse(reader.GetAttribute("PID"));
                        tlist.Add(item);
                    } else if (reader.Name == "Count") { // #23679
                        count = reader.ReadElementContentAsInt();
                    }

                }
                reader.Read();
            }
            reader.Close();
            fs.Close();

            bm.Add(-1, new List<BookmarkItem>());

            foreach (var item in tlist) {
                if (!bm.ContainsKey(item.ParentId))
                    bm.Add(item.ParentId, new List<BookmarkItem>());
                bm[item.ParentId].Add(item);
                //#23679fix
                //旧ver互換維持
                if (count == -1 || item.Id > count)
                    count = item.Id;
            }

            //HasChildプロパティの設定
            foreach (var item in tlist)
                if (!bm.ContainsKey(item.Id) || bm[item.Id].Count == 0)
                    item.HasChild = false;
                else
                    item.HasChild = true;
        }

        public void Save() {
            var fs = new FileStream(Path.Combine(apppath, "Bookmark.config"), FileMode.Create, FileAccess.Write);
            var writer = new StreamWriter(fs, Encoding.UTF8);

            writer.WriteLine("<?xml version=\"1.0\" encoding=\"utf-8\"?>");
            writer.WriteLine("<BookmarkRoot>");

            var tlist = new Dictionary<int, BookmarkItem>();
            foreach (KeyValuePair<int, List<BookmarkItem>> pair in bm) {
                foreach (BookmarkItem item in pair.Value)
                    tlist.Add(item.Id, item);
            }
            //BookmarkItem parent;

            for (int i = 0; i <= count; i++) {

                BookmarkItem item;
                try {
                    item = tlist[i];
                } catch (KeyNotFoundException) { continue; }
                // "<{0} Name=\"{1}\" Url=\"{2}\" HasChild={3} />\r\n"
                var wstring = "<";
                if (item.Type == BookmarkItemType.Dir)
                    wstring += "dir Name=\"";
                else
                    wstring += "item Name=\"";
                wstring += GetUriEncoded(item.Name);
                wstring += "\" Url=\"";
                if (!string.IsNullOrEmpty(item.Url))
                    wstring += GetUriEncoded(item.Url);
                wstring += string.Format("\" ID=\"{0}", item.Id);

                wstring += string.Format("\" PID=\"{0}\" />", item.ParentId);

                writer.WriteLine(wstring);
            }

            writer.Write(String.Format("<Count>{0}</Count>", count)); // #23679

            writer.Write("</BookmarkRoot>");

            writer.Flush();
            writer.Close();

        }

        private string GetUriEncoded(string uri) {
            string ret = uri;
            ret = Regex.Replace(uri, "&", "&amp;");
            ret = Regex.Replace(ret, "\"", "&quot;");
            ret = Regex.Replace(ret, "'", "&apos;");
            ret = Regex.Replace(ret, "<", "&lt;");
            ret = Regex.Replace(ret, ">", "&gt;");
            return ret;
        }

        public ToolStripItemCollection GetMenu(ToolStrip owner) {

            var ret = new ToolStripItemCollection(owner, new ToolStripItem[] { });

            //default menu
            var manage = new ToolStripMenuItem();
            manage.Text = lang.Get("BookmarkManager_BookmarkManage");
            manage.Click += new EventHandler(manage_Click);
            ret.Add(manage);
            ret.Add(new ToolStripSeparator());

            var tlist = new List<BookmarkItem>();
            foreach (KeyValuePair<int, List<BookmarkItem>> pair in bm) {
                foreach (BookmarkItem item in pair.Value)
                    tlist.Add(item);
            }

            var plist = new List<BookmarkItem>();//childを持つdir
            var alist = new List<BookmarkItem>();//aloneなdir
            var ilist = new List<BookmarkItem>();//item
            foreach (var item in tlist)
                if (item.Type == BookmarkItemType.Dir) {
                    if (item.HasChild)
                        plist.Add(item);
                    else
                        alist.Add(item);
                } else if (item.Type == BookmarkItemType.Item)
                    ilist.Add(item);

            var imenu = new List<DropDownItemWithID>();
            foreach (var item in ilist) {
                var menu = new DropDownItemWithID(item.Id);
                ((ToolStripItem)menu).Text = item.Name;
                menu.Click += new EventHandler(bookmarkItem_Click);
                menu.ParentID = item.ParentId;
                imenu.Add(menu);
            }
            var amenu = new List<DropDownItemWithID>();
            foreach (var item in alist) {
                var menu = new DropDownItemWithID(item.Id);
                ((ToolStripItem)menu).Text = item.Name;

                var noitem = new ToolStripMenuItem(lang.Get("strNoItem"));
                noitem.Enabled = false;
                ((ToolStripDropDownItem)menu).DropDownItems.Add(noitem);
                ((ToolStripDropDownItem)menu).DropDownItems.Add(new ToolStripSeparator());
                var cdir = new ToolStripMenuItem(lang.Get("createFolderText"));
                cdir.Click += new EventHandler(createDir_Click);
                menu.DropDownItems.Add(cdir);
                menu.ParentID = item.ParentId;
                amenu.Add(menu);
            }

            var pmenu = new List<DropDownItemWithID>();
            foreach (var item in plist) {
                var menu = new DropDownItemWithID(item.Id);
                ((ToolStripItem)menu).Text = item.Name;

                menu.ParentID = item.ParentId;
                pmenu.Add(menu);
            }

            foreach (var menu in pmenu) {
                foreach (var child in imenu)
                    if (child.ParentID == menu.ID)
                        menu.DropDownItems.Add((ToolStripItem)child);
                foreach (var child in amenu)
                    if (child.ParentID == menu.ID)
                        menu.DropDownItems.Add((ToolStripItem)child);
                foreach (var child in pmenu)
                    if (child.ParentID == menu.ID)
                        menu.DropDownItems.Add(child);
                menu.DropDownItems.Add(new ToolStripSeparator());
                var cdir = new ToolStripMenuItem(lang.Get("createFolderText"));
                cdir.Click += new EventHandler(createDir_Click);
                menu.DropDownItems.Add(cdir);
                if (menu.ParentID == -1)
                    ret.Add((ToolStripItem)menu);
            }
            foreach (var menu in amenu)
                if (menu.ParentID == -1)
                    ret.Add((ToolStripItem)menu);
            foreach (var menu in imenu)
                if (menu.ParentID == -1)
                    ret.Add((ToolStripItem)menu);

            //memo あらかじめ空のListは作っておいて、一つのforeachループのなかでswitchで種類別分岐して処理するとかしないと
            //ディレクトリとアイテムの登録順ばらばらなときに順番復元されない
            //親になり得るディレクトリは必ず子よりも先に生成されるって考えでいいはず

            if (ret.Count == 2) {
                var noitem = new ToolStripMenuItem();
                noitem.Text = lang.Get("strNoItem");
                noitem.Enabled = false;
                ret.Add(noitem);
            }

            ret.Add(new ToolStripSeparator());
            var rcdir = new ToolStripMenuItem(lang.Get("createFolderText"));
            rcdir.Click += new EventHandler(createDir_Click);
            ret.Add(rcdir);

            return ret;
        }

        /// <summary>
        /// BookmarkManageDialog
        /// </summary>
        void manage_Click(object sender, EventArgs e) {
            var manage = new ManageDialog(this);
            manage.ShowDialog(ManageMode.Manage, ManageType.Item);
        }

        /// <summary>
        /// Create new directory
        /// </summary>
        private void createDir_Click(object sender, EventArgs e) {
            var manage = new ManageDialog(this);
            manage.ShowDialog(ManageMode.AddNew, ManageType.Dir);
        }

        /// <summary>
        /// Open bookmark
        /// </summary>
        private void bookmarkItem_Click(object sender, EventArgs e) {
            var Id = ((DropDownItemWithID)sender).ID;
            var tlist = new List<BookmarkItem>();
            foreach (KeyValuePair<int, List<BookmarkItem>> pair in bm) {
                foreach (BookmarkItem item in pair.Value)
                    tlist.Add(item);
            }
            foreach (var item in tlist)
                if (item.Id == Id) {
                    main.WB_Navigate(item.Url);
                    break;
                }
        }

        public string[] GetVersion() {
            var asm = System.Reflection.Assembly.GetExecutingAssembly();
            return new string[]{
                asm.ManifestModule.Name.Replace(".dll","\0"),
                asm.GetName().Version.ToString(),
                "bis5",
                asm.ManifestModule.Name
            };
        }

    }

    public class DropDownItemWithID : ToolStripMenuItem {
        public int ID { get; set; }
        public int ParentID { get; set; }
        public DropDownItemWithID() { }
        public DropDownItemWithID(int Id) { ID = Id; }
    }

    public class BookmarkItem {

        public BookmarkItem() { }

        private bool _hasChild = false;
        private string _url = "";

        public string Name { get; set; }
        public BookmarkItemType Type { get; set; }
        public string Url {
            get { return _url; }
            set { _url = value; }
        }
        public bool HasChild {
            get { return _hasChild; }
            set { _hasChild = value; }
        }
        public int ParentId { get; set; }
        public int Id { get; set; }

    }

    public enum BookmarkItemType { Dir, Item }
}