﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Threading;
using Codeplex.Data;
using LinqToTwitter;

namespace Azyobuzi.UserStreamEx
{
    public class UserStream
        : INotStartedUserStream, IStartedUserStream
    {
        public UserStream(ITwitterAuthorizer auth)
        {
            this.auth = auth;
        }

        private ITwitterAuthorizer auth;
        private HttpWebRequest req;

        public IStartedUserStream Start(string track, bool allReplies)
        {
            if (req != null) req.Abort();
            
            var reqUri = new UriBuilder("https://userstream.twitter.com/2/user.json");
            var param = new Dictionary<string, string>();
            if (!string.IsNullOrEmpty(track)) param.Add("track", track);
            if (allReplies) param.Add("replies", "all");
            reqUri.Query = string.Join("&", param.Select(_ => string.Format("{0}={1}", _.Key, _.Value)));

            req = auth.Get(reqUri.ToString()) as HttpWebRequest;
            req.BeginGetResponse(result =>
            {
                try
                {
                    if (Started != null) Started(this, EventArgs.Empty);
                    var res = req.EndGetResponse(result);
                    using (var sr = new StreamReader(res.GetResponseStream()))
                    {
                        while (!sr.EndOfStream)
                        {
                            var line = sr.ReadLine();

                            if (StreamingWriter != null)
                            {
                                try
                                {
                                    StreamingWriter.WriteLine(line);
                                    StreamingWriter.Flush();
                                }
                                catch { }
                            }

                            if (string.IsNullOrWhiteSpace(line)) continue;

                            var t = new Thread(RaiseEvent);
                            t.Start(line);
                        }
                    }

                    if (Stopped != null)
                        Stopped(this, new StoppedEventArgs(StopReason.CloseResponse, null));
                }
                catch (WebException ex)
                {
                    if (ex.Status == WebExceptionStatus.RequestCanceled)
                    {
                        if (Stopped != null)
                            Stopped(this, new StoppedEventArgs(StopReason.UserStop, ex));
                    }
                    else
                    {
                        throw;
                    }
                }
                catch (Exception ex)
                {
                    if (Stopped != null)
                        Stopped(this, new StoppedEventArgs(StopReason.Error, ex));
                }
                req = null;
            }, null);

            return this;
        }

        private void RaiseEvent(object args)
        {
            var line = (string)args;
            var json = DynamicJson.Parse(line);
            if (json.friends())
            {
                if (ReceiveFriends != null) ReceiveFriends(this, new ReceiveFriendsEventArgs(line));
            }
            else if (json.delete())
            {
                if (DeleteStatus != null) DeleteStatus(this, new DeleteStatusEventArgs(line));
            }
            else if (json.direct_message())
            {
                if (NewDirectMessage != null) NewDirectMessage(this, new NewDirectMessageEventArgs(line));
            }
            else if (json.@event())
            {
                if (ReceiveEvent != null) ReceiveEvent(this, new ReceiveEventEventArgs(line));
            }
            else if (json.limit())
            {
                if (TrackLimit != null) TrackLimit(this, new TrackLimitEventArgs(line));
            }
            else if (json.text())
            {
                if (NewTweet != null) NewTweet(this, new NewTweetEventArgs(line));
            }
            else
            {
                if (ReceiveUnsupportedData != null) ReceiveUnsupportedData(this, new ReceiveJsonEventArgs(line));
            }
        }

        public INotStartedUserStream Stop()
        {
            if (req != null) req.Abort();
            req = null;
            return this;
        }

        public IStartedUserStream SetTimeout(int waitTime)
        {
            if (waitTime < 1)
            {
                Stop();
            }
            else
            {
                var t = new Thread(_ =>
                {
                    Thread.Sleep((int)_);
                    Stop();
                });
                t.IsBackground = true;
                t.Start(waitTime);
            }
            return this;
        }

        public TextWriter StreamingWriter { set; get; }

        #region Events
        public event EventHandler Started;
        public event EventHandler<StoppedEventArgs> Stopped;
        public event EventHandler<ReceiveFriendsEventArgs> ReceiveFriends;
        public event EventHandler<NewTweetEventArgs> NewTweet;
        public event EventHandler<NewDirectMessageEventArgs> NewDirectMessage;
        public event EventHandler<DeleteStatusEventArgs> DeleteStatus;
        public event EventHandler<ReceiveEventEventArgs> ReceiveEvent;
        public event EventHandler<TrackLimitEventArgs> TrackLimit;
        public event EventHandler<ReceiveJsonEventArgs> ReceiveUnsupportedData;
        #endregion

        #region AddHandlerMethod
        public INotStartedUserStream SetStartedEvent(EventHandler handler)
        {
            Started += handler;
            return this;
        }

        public INotStartedUserStream SetStoppedEvent(EventHandler<StoppedEventArgs> handler)
        {
            Stopped += handler;
            return this;
        }

        public INotStartedUserStream SetReceiveFriendsEvent(EventHandler<ReceiveFriendsEventArgs> handler)
        {
            ReceiveFriends += handler;
            return this;
        }

        public INotStartedUserStream SetNewTweetEvent(EventHandler<NewTweetEventArgs> handler)
        {
            NewTweet += handler;
            return this;
        }

        public INotStartedUserStream SetNewDirectMessageEvent(EventHandler<NewDirectMessageEventArgs> handler)
        {
            NewDirectMessage += handler;
            return this;
        }

        public INotStartedUserStream SetDeleteStatusEvent(EventHandler<DeleteStatusEventArgs> handler)
        {
            DeleteStatus += handler;
            return this;
        }

        public INotStartedUserStream SetReceiveEventEvent(EventHandler<ReceiveEventEventArgs> handler)
        {
            ReceiveEvent += handler;
            return this;
        }

        public INotStartedUserStream SetTrackLimitEvent(EventHandler<TrackLimitEventArgs> handler)
        {
            TrackLimit += handler;
            return this;
        }

        public INotStartedUserStream SetReceiveUnsupportedDataEvent(EventHandler<ReceiveJsonEventArgs> handler)
        {
            ReceiveUnsupportedData += handler;
            return this;
        }
        #endregion
    }
}
