#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QTextCodec>
#include <QDir>
#include <QSettings>

#include <QMessageBox>
#include <QDebug>

#include "logstartdialog.h"


MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    // set Codec
    QTextCodec::setCodecForTr(QTextCodec::codecForName("UTF-8")); // source codec is UTF-8

    QCoreApplication::setApplicationName("QtUSBRH and USB Camera Logger win32");
    QCoreApplication::setApplicationVersion("0.1");

    label_statusbar = new QLabel;
    ui->statusBar->addWidget(label_statusbar);

    IniFilepath = QCoreApplication::applicationDirPath() + QDir::separator() + QCoreApplication::applicationName() + ".ini";
    // Load Setting
    LoadSaveSetting(IniFilepath, false);

    ui->lcdTempr->display(0.0);
    ui->lcdHumid->display(0.0);

    ui->label_LogLapsedTime->setText(""); // "Lapsed Time: "
//    ui->label_LogLapsedTime->hide(); // hidden label_LogLapsedTime


    logging = false;

    // Init & Start DAQ Thread
    daqTh = new DAQThread(this);
    // connect signals
    connect(this, SIGNAL(LogStart(QString)),
            daqTh, SIGNAL(LogStart(QString)));
    connect(this, SIGNAL(LogStop()),
            daqTh, SIGNAL(LogStop()));
    connect(daqTh, SIGNAL(LogStarted()),
            this, SLOT(slotLogStarted()));
    connect(daqTh, SIGNAL(LogEnded(int)),
            this, SLOT(slotLogEnded(int)));
    connect(daqTh, SIGNAL(GetData(QDateTime, double, double)),
            this, SLOT(slotGetData(QDateTime, double, double)));
    connect(daqTh, SIGNAL(ChangeLogFilename(QString)),
            this, SLOT(slotChangeLogFilename(QString)));

    connect(daqTh, SIGNAL(SaveCurrentImage(QString, QString)),
            this, SLOT(slotSaveCurrentImage(QString, QString)));

    imageCapture   = new ImageCapture( 0, 640, 480 );
    ui->ImgView->setCapture( imageCapture );
    ui->ImgView->startTimer();

    // DaqTh start
    daqTh->start();

}

MainWindow::~MainWindow()
{   
    ui->ImgView->stopTimer();
    delete imageCapture;

    // Save Setting
    LoadSaveSetting(IniFilepath, true);

    // stop daqTh
    daqTh->stop();
    daqTh->wait();
    delete daqTh;

    delete label_statusbar;

    delete ui;
}

void MainWindow::LoadSaveSetting(QString filepath, bool store)
{
    QSettings settings(filepath, QSettings::IniFormat);
    settings.setIniCodec("UTF-8");
    if (store) { // Store(Save)
        settings.setValue("SETTING/SAVEPATH", logpath);
    } else { // Load
        logpath = settings.value("SETTING/SAVEPATH", QString("")).toString();
    }
}


void MainWindow::slotGetData(QDateTime timestamp, double tmpr, double humid)
{
    // qDebug() << "timestamp=" << timestamp << ", Tempr=" << tmpr << ", humid=" << humid;
    ui->lcdTempr->display(tmpr);
    ui->lcdHumid->display(humid);

    if (logging) {
        int lapsed_time = (QDateTime::currentMSecsSinceEpoch() - logstart_time.toMSecsSinceEpoch()) / 1000;
        if (lapsed_time < 60) { // 1 minute
            ui->label_LogLapsedTime->setText(QString(tr("Lapsed Time: %1 sec")).arg(lapsed_time));
        } else if (lapsed_time < 3600) { // 1 hour
            ui->label_LogLapsedTime->setText(QString(tr("Lapsed Time: %1 min")).arg(lapsed_time/60));
        } else if (lapsed_time < 86400) { // 1 day
            ui->label_LogLapsedTime->setText(QString(tr("Lapsed Time: %1 hour")).arg(lapsed_time/3600));
        } else { // over 1 day
            ui->label_LogLapsedTime->setText(QString(tr("Lapsed Time: %1 day")).arg(lapsed_time/86400));
        }
    }
}
void MainWindow::slotChangeLogFilename(QString filename)
{
//    ui->statusBar->showMessage(QString(tr("logfile: %1")).arg(filename));
    label_statusbar->setText(QString(tr("logfile: %1")).arg(filename));
}

void MainWindow::slotSaveCurrentImage(QString imgfilepath,  QString overlay_text)
{
    ui->ImgView->saveCurrentImage(imgfilepath, overlay_text);
}

void MainWindow::slotLogStarted()
{
    // ui->label_LogState->setText("LogStarted: " + QDateTime::currentDateTime().toString("yyyy/MM/dd hh:mm:ss"));
    ;
}
void MainWindow::slotLogEnded(int status)
{
    logging = false;
    ui->pushButton_LogStart->setText(tr("Log Start"));

    ui->label_LogState->setText(tr("LogEnded: ") + QDateTime::currentDateTime().toString("yyyy/MM/dd hh:mm:ss"));
}


void MainWindow::on_actionExit_triggered()
{
    close();
}

void MainWindow::on_pushButton_LogStart_clicked()
{
    if (logging) {
        // now logging
        // Send LogStop
        emit LogStop();

        logging = false;
        ui->pushButton_LogStart->setText(tr("Log Start"));
        ui->pushButton_LogStart->setIcon(QIcon(":/Images/Play.png"));
        ui->label_LogState->setText(tr("LogEnded: ") + QDateTime::currentDateTime().toString("yyyy/MM/dd hh:mm:ss"));
        ui->label_LogLapsedTime->setText("");
    } else {
        // now idle..
        LogStartDialog *dlg = new LogStartDialog(logpath, this);
        int i = dlg->exec();
        if (i == QDialog::Accepted) {
            qDebug() << "Ok";
            logpath = dlg->getLogSaveFolderPath();
            qDebug() << "logpath is " << logpath;

            logging = true;
            ui->pushButton_LogStart->setText(tr("Log Stop"));
            ui->pushButton_LogStart->setIcon(QIcon(":/Images/Stop.png"));
            logstart_time = QDateTime::currentDateTime();
            ui->label_LogState->setText(tr("LogStarted: ") + logstart_time.toString("yyyy/MM/dd hh:mm:ss"));

            // Send LogStart
            emit LogStart(logpath);

        } else if (i == QDialog::Rejected) {
            qDebug() << "Cancel";
        } else {
            qDebug() << "Unkown";
        }
        delete dlg;
    }
}

void MainWindow::on_actionAbout_triggered()
{
    QMessageBox::about(this, "About", tr("QtUSBRH and USB Camera Logger for win32 v0.1"));
}

void MainWindow::on_actionAboutQt_triggered()
{
    QApplication::aboutQt();
}
