#include <QDebug>

#include "daqthread.h"

#define DAQmxErrChk(functionCall) if( DAQmxFailed(error=(functionCall)) ) goto Error; else

DAQThread::DAQThread(QObject *parent) :
    QThread(parent)
{
    stopped = false;

    state = State_IDLE;
    EvtQue = new QQueue<int>();

    // cannot Init (create) DAQmxTask, when Constructer
}

DAQThread::~DAQThread()
{
    delete EvtQue;
}

void DAQThread::stop()
{
    stopped = true;
}

void DAQThread::startAIAcq()
{
    EvtQue->enqueue(Ev_AI_Start);
}

void DAQThread::stopAIAcq()
{
    EvtQue->enqueue(Ev_AI_Stop);
}

void DAQThread::run()
{
    int32       error = 0;
    char        errBuff[2048] = { '\0' };

    int event = 0;
    state = State_IDLE;

    // DAQmxBase task
    taskHandle = 0;

    // Channel parameters
    char        chan[] = "Dev1/ai0, Dev1/ai1";
    float64     min = -10.0;
    float64     max = 10.0;

    // Timing parameters
    char        clockSource[] = "OnboardClock";
    uInt64      samplesPerChan = 1000;
    float64     sampleRate = 1000.0;

    // Data read parameters
    #define     bufferSize (uInt32)1000
    float64     data[bufferSize * 2];
//    int32       pointsToRead = bufferSize;
    int32       pointsToRead = bufferSize / 5;
    int32       pointsRead;
    float64     timeout = 10.0;
    int32       totalRead = 0;


    uInt32 serialnumber;
    DAQmxBaseGetDevSerialNum("Dev1", &serialnumber);
    qDebug() << QString("%1 SerialNumber is 0x%2").arg("Dev1").arg(serialnumber, 8, 16, QChar('0'));

    // signal - daqmxbase ready
    emit daqmxbase_ready();

    while(!stopped) {
        if (EvtQue->isEmpty()) {
            switch (state) {
            case State_IDLE:
                msleep(100);
                break;
            case State_EXEC:
                DAQmxErrChk (DAQmxBaseReadAnalogF64(taskHandle,pointsToRead,timeout,DAQmx_Val_GroupByScanNumber,data,bufferSize*2,&pointsRead,NULL));
                totalRead += pointsRead;
                // Just print out the first 10 points of the last data read
                for (int i = 0; i < 10; ++i) {
                    qDebug() << QString("data0[%1] = %2\tdata1[%3] = %4").arg(i).arg(data[2*i]).arg(i).arg(data[2*i+1]);
                }
                break;
            default:
                break;
            }
        } else {
            event = EvtQue->dequeue();
            switch (state) {
            case State_IDLE:
                if (event == Ev_AI_Start) {
                    qDebug() << "start AI Acquisition."; // start AI Acq
                    DAQmxErrChk (DAQmxBaseCreateTask("", &taskHandle));
                    DAQmxErrChk (DAQmxBaseCreateAIVoltageChan(taskHandle,chan,"",DAQmx_Val_RSE,min,max,DAQmx_Val_Volts,NULL));
                    DAQmxErrChk (DAQmxBaseCfgSampClkTiming(taskHandle,clockSource,sampleRate,DAQmx_Val_Rising,DAQmx_Val_ContSamps,samplesPerChan));
                    DAQmxErrChk (DAQmxBaseCfgInputBuffer(taskHandle, 200000));
                    // msleep(100);
                    DAQmxErrChk (DAQmxBaseStartTask (taskHandle));
                    totalRead = 0;
                    state = State_EXEC;
                }
                break;
            case State_EXEC:
                if (event == Ev_AI_Stop) {
                    qDebug() << "stop AI Acquisition."; // stop AI Acq
                    DAQmxErrChk (DAQmxBaseStopTask (taskHandle));
                    DAQmxErrChk (DAQmxBaseClearTask (taskHandle));
                    qDebug() << QString("Acquired %1 samples. Total %2").arg(pointsRead).arg(totalRead);
                    state = State_IDLE;
                }
                break;
            default:
                break;
            }
        }
    }

Error:
    if( DAQmxFailed(error) ) {
        DAQmxBaseGetExtendedErrorInfo(errBuff,2048);
        qDebug() << errBuff;
        emit daqmxbase_error(QString(errBuff));
    }
    if( taskHandle!=0 )  {
        /*********************************************/
        // DAQmx Stop Code
        /*********************************************/
        DAQmxBaseStopTask(taskHandle);
        DAQmxBaseClearTask(taskHandle);
        emit daqmxbase_final();
    }
}
