#include <QTextCodec>
#include <QMessageBox>
#include <QFileDialog>
#include <QFileInfo>
#include <QTime>
#include <QSettings>

#include <QDebug>

#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "registholiday_dialog.h"

#define INIFILEPATH "QtSchChime.ini"
#define DEFAULT_CHIME_FILENAME "chime1s.mp3"

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    QTextCodec::setCodecForTr(QTextCodec::codecForName("UTF-8")); // source codec is UTF-8

    status = 0;
    ui->pushButton->setText("Chime Start");

    // load INI file
    IniFilePath = QCoreApplication::applicationDirPath() + QDir::separator() + INIFILEPATH;
    LoadSaveSetting(IniFilePath, false);


    // status bar
    statusbar_label_status = new QLabel;
    statusbar_label_nowtime = new QLabel;
    ui->statusBar->addWidget(statusbar_label_status, 1);
    ui->statusBar->addWidget(statusbar_label_nowtime, 0);
    statusbar_label_status->setText("idle...");
    statusbar_label_nowtime->setText(QTime::currentTime().toString("hh:mm:ss"));
    Timer1sec = new QTimer(); // timer for nowtime display
    connect(Timer1sec, SIGNAL(timeout()), this, SLOT(TimeoutDone()));
    Timer1sec->start(1000);

    if (ui->lineEdit_FilePath->text().isEmpty()) {
        // Set DefaultChimePath
        DefaultChimeFilePath =  QCoreApplication::applicationDirPath() + QDir::separator() + DEFAULT_CHIME_FILENAME;
        QFileInfo fi = QFileInfo(DefaultChimeFilePath);
        if (fi.exists()) {
            ui->lineEdit_FilePath->setText(DefaultChimeFilePath);
            ui->pushButton->setEnabled(true);
            ui->pushButton_TestPlay->setEnabled(true);
        } else {
            ui->pushButton->setEnabled(false);
            ui->pushButton_TestPlay->setEnabled(false);
        }
    }

#if 0
    // Set SchTimeList
    QStringList list;
    list.clear();
    list << "09:00" << "12:00" << "13:00" << "17:30";
    // Set ListWidget, and spinBox hh:mm
    ui->listWidget_SchTimeList->addItems(list);
#endif

    ui->listWidget_SchTimeList->setCurrentRow(0);
    QStringList current_data = ui->listWidget_SchTimeList->currentItem()->text().split(":");
    ui->spinBox_Hour->setValue(current_data.at(0).toInt());
    ui->spinBox_Min->setValue(current_data.at(1).toInt());

    // PlayMusic Thread
    ThPlayMusic = new PlayMusicThread(this->winId());
//    connect(ThPlayMusic, SIGNAL(notify_calc_end(QString)),
//            this, SLOT(slot_calc_end(QString)));
    ThPlayMusic->start();

    if (ui->hSlider_Volume->value() == 0) {
        float vlmlevel = ThPlayMusic->GetSoundVolume() * 100.0;
        // qDebug() << "vlmlevel=" << vlmlevel ;
        ui->hSlider_Volume->setValue((int)vlmlevel);
    }

    // ScheduleManager Thread
    ThSchMngr = new ThSchManager(this);
    connect(ThSchMngr, SIGNAL(sig_notify_nexttime(QString)),
            this, SLOT(slot_notify_nexttime(QString)));
    ThSchMngr->setThreadPtrPlayMusic(ThPlayMusic);
    ThSchMngr->start();

    if (ui->checkBox_AutoStart->isChecked() && ui->pushButton->isEnabled()) {
        // Start SchChime
        on_pushButton_clicked();
    }

}

MainWindow::~MainWindow()
{
    // Save INI file
    LoadSaveSetting(IniFilePath, true);

    ThSchMngr->sendEvent_Stop();
    ThSchMngr->stop();
    ThSchMngr->wait();
    delete ThSchMngr;

    ThPlayMusic->StopMusic();
    ThPlayMusic->stop();
    ThPlayMusic->wait();
    delete ThPlayMusic;

    Timer1sec->stop();
    delete Timer1sec;
    delete statusbar_label_status;
    delete statusbar_label_nowtime;

    delete ui;
}

void MainWindow::LoadSaveSetting(QString filepath, bool store = false)
{
    if (filepath.isEmpty()) return;

    QSettings settings(filepath, QSettings::IniFormat);
    if (store) { // Save
        // Exclude AutoStart flag
        settings.setValue("SETTING/AUTO_START", ui->checkBox_AutoStart->isChecked());

        // Sound Volume Level
        settings.setValue("SETTING/VOLUME_LEVEL", ui->hSlider_Volume->value());

        // Exclude SS flag
        settings.setValue("SETTING/EXCLUDE_SS", ui->checkBox_Exclude_SS->isChecked());

        // Exclude Holidays flag
        settings.setValue("SETTING/EXCLUDE_HOLIDAYS", ui->checkBox_Exclude_Holiday->isChecked());

        // ChimeFilePath
        settings.setValue("SETTING/CHIME_FILEPATH", ui->lineEdit_FilePath->text());

        // SchTimeList
        QStringList list;
        for (int i = 0; i < ui->listWidget_SchTimeList->count(); i++) {
            list << ui->listWidget_SchTimeList->item(i)->text();
        }
        settings.setValue("SETTING/SCHTIME_LIST", list);

        // HolidaysList
        settings.setValue("SETTING/HOLIDAYS_LIST", Holidays);

    } else { // Load
        // Exclude AutoStart flag
        ui->checkBox_AutoStart->setChecked(settings.value("SETTING/AUTO_START", false).toBool());

        // Sound Volume Level
        ui->hSlider_Volume->setValue((settings.value("SETTING/VOLUME_LEVEL", 0).toInt()));

        // Exclude SS flag
        ui->checkBox_Exclude_SS->setChecked(settings.value("SETTING/EXCLUDE_SS", true).toBool());

        // Exclude Holidays flag
        ui->checkBox_Exclude_Holiday->setChecked(settings.value("SETTING/EXCLUDE_HOLIDAYS", false).toBool());

        // ChimeFilePath
        ui->lineEdit_FilePath->setText(settings.value("SETTING/CHIME_FILEPATH", QString("")).toString());

        QStringList list(settings.value("SETTING/SCHTIME_LIST", QStringList()).toStringList());
        if (list.count() == 0) {
            list << "09:00" << "12:00" << "13:00" << "17:30";
        }
        // Set ListWidget, and spinBox hh:mm
        ui->listWidget_SchTimeList->addItems(list);

        QStringList list1(settings.value("SETTING/HOLIDAYS_LIST", QStringList()).toStringList());
        if (list1.count() == 0) {
            Holidays.clear();
            Holidays << "01/01" << "01/15" << "02/11" << "03/21" << "04/29" << "05/01" << "05/03" << "05/04" << "05/05" << "07/21" << "09/15" << "09/23" << "10/10" << "11/03" << "11/23" << "12/23";
        } else {
            Holidays.clear();
            Holidays.append(list1);
        }
        // qDebug() << Holidays;
    }
}

void MainWindow::TimeoutDone()
{
    statusbar_label_nowtime->setText(QTime::currentTime().toString("hh:mm:ss"));
}

void MainWindow::slot_notify_nexttime(QString nexttime)
{
    if (status) {
        QString msg;
        msg = "Next Chime: " + nexttime;
        statusbar_label_status->setText(msg);

        // Search index of nexttime
        QStringList list;
        for (int i = 0; i < ui->listWidget_SchTimeList->count(); i++) {
            list << ui->listWidget_SchTimeList->item(i)->text();
        }
        int r;
        if (0 <= (r = list.indexOf(nexttime, 0))) {
            ui->listWidget_SchTimeList->setCurrentRow(r);
        }
    }
    // MainWindow Popup when minimized
    this->setWindowState(this->windowState() & ~Qt::WindowMinimized | Qt::WindowActive);
}


void MainWindow::on_actionExit_triggered()
{
    close();
}

void MainWindow::on_actionAbout_triggered()
{
    QString AppName = "QtSchChime";
    QString Version = "0.1";
    QString Year = "2014";
    QString Author = "alucky4416@users.sourceforge.ne.jp";
    QString msg;
    msg = QString("<h2>%1 %2</h2>" \
                  "<p>Using Qt4" \
                  "<p>Using BASS audio library (Un4seen Developments)" \
                  "<p>" \
                  "<p>Copyright &copy; %3 %4").arg(AppName).arg(Version).arg(Year).arg(Author);
    QMessageBox::about(this, "About", msg);
}

void MainWindow::on_actionAbout_Qt_triggered()
{
    QApplication::aboutQt();
}

void MainWindow::on_toolButton_FileSelect_clicked()
{
    QString fileName = QFileDialog::getOpenFileName(
                this,
                tr("Select File"), // title
                QDir::homePath(), // CurrentPath
                tr("MP3 (*.mp3);; All Files (*)")); // Filter
    if (!fileName.isEmpty()) {
        ui->lineEdit_FilePath->clear();
        ui->lineEdit_FilePath->setText(fileName);
    }

}

void MainWindow::on_pushButton_clicked()
{
    if (status) {
        // Executing SchChime
        // Stop SchChime
        ThSchMngr->sendEvent_Stop();
        statusbar_label_status->setText("idle...");
        ui->pushButton->setText("Chime Start");

        ui->lineEdit_FilePath->setEnabled(true);
        ui->toolButton_FileSelect->setEnabled(true);
        ui->spinBox_Hour->setEnabled(true);
        ui->spinBox_Min->setEnabled(true);
        ui->listWidget_SchTimeList->setEnabled(true);
        ui->pushButton_Add->setEnabled(true);
        ui->pushButton_Change->setEnabled(true);
        ui->pushButton_Del->setEnabled(true);
        ui->pushButton_TestPlay->setEnabled(true);
        ui->checkBox_Exclude_SS->setEnabled(true);
        ui->checkBox_Exclude_Holiday->setEnabled(true);
        ui->pushButton_RegistHoliday->setEnabled(true);

        status = 0;
    } else {
        // idle
        if (ui->lineEdit_FilePath->text().isEmpty()) return;
        QFileInfo fi = QFileInfo(ui->lineEdit_FilePath->text());
        if (!fi.exists()) {
            QMessageBox::warning(this, "Warning", tr("File Not Found."));
            return;
        }
        if (ui->hSlider_Volume->value() == 0) {
            QMessageBox::warning(this, "Warning", ("Sound Volume = 0"));
            return;
        }

        ThSchMngr->setExcludeSS(ui->checkBox_Exclude_SS->isChecked());
        ThSchMngr->setExcludeHolidays(ui->checkBox_Exclude_Holiday->isChecked());
        ThSchMngr->setChimeFilepath(ui->lineEdit_FilePath->text());
        ThSchMngr->setHolidaysList(Holidays);

        // Update SchTimeList
        QStringList list;
        for (int i = 0; i < ui->listWidget_SchTimeList->count(); i++) {
            list << ui->listWidget_SchTimeList->item(i)->text();
        }
        ThSchMngr->setSchTimeList(list);

        // Start SchChime
        ThSchMngr->sendEvent_Start();
        statusbar_label_status->setText("start");
        ui->pushButton->setText("Chime Stop");

        ui->lineEdit_FilePath->setEnabled(false);
        ui->toolButton_FileSelect->setEnabled(false);
        ui->spinBox_Hour->setEnabled(false);
        ui->spinBox_Min->setEnabled(false);
        ui->listWidget_SchTimeList->setEnabled(false);
        ui->pushButton_Add->setEnabled(false);
        ui->pushButton_Change->setEnabled(false);
        ui->pushButton_Del->setEnabled(false);
        ui->pushButton_TestPlay->setEnabled(false);
        ui->checkBox_Exclude_SS->setEnabled(false);
        ui->checkBox_Exclude_Holiday->setEnabled(false);
        ui->pushButton_RegistHoliday->setEnabled(false);

        status = 1;
    }
}

void MainWindow::on_pushButton_TestPlay_clicked()
{
    if (status) return;
    if (ui->lineEdit_FilePath->text().isEmpty()) return;
    QFileInfo fi = QFileInfo(ui->lineEdit_FilePath->text());
    if (!fi.exists()) {
        QMessageBox::warning(this, "Warning", tr("File Not Found."));
        return;
    }

    ThPlayMusic->PlayMusic(ui->lineEdit_FilePath->text());
}

void MainWindow::on_listWidget_SchTimeList_currentRowChanged(int currentRow)
{
    QStringList current_data = ui->listWidget_SchTimeList->currentItem()->text().split(":");
    ui->spinBox_Hour->setValue(current_data.at(0).toInt());
    ui->spinBox_Min->setValue(current_data.at(1).toInt());
}

void MainWindow::on_pushButton_Change_clicked()
{
    int hour = ui->spinBox_Hour->value();
    int min = ui->spinBox_Min->value();
    QString timestr = QString("%1:%2").arg(hour,2,10,QChar('0')).arg(min, 2, 10, QChar('0'));

    QList<QListWidgetItem*> findresult = ui->listWidget_SchTimeList->findItems(timestr, Qt::MatchExactly);
    if (findresult.isEmpty()) {
        // change item when not exist of SchTimeList
        QListWidgetItem *item = ui->listWidget_SchTimeList->currentItem();
        item->setText(timestr);
        ui->listWidget_SchTimeList->setCurrentItem(item);
    } else {
        ;
    }
}

void MainWindow::on_pushButton_Add_clicked()
{
    int hour = ui->spinBox_Hour->value();
    int min = ui->spinBox_Min->value();
    QString timestr = QString("%1:%2").arg(hour,2,10,QChar('0')).arg(min, 2, 10, QChar('0'));

    QList<QListWidgetItem*> findresult = ui->listWidget_SchTimeList->findItems(timestr, Qt::MatchExactly);
    if (findresult.isEmpty()) {
        // add item when not exist of SchTimeList
        QListWidgetItem *item = new QListWidgetItem();
        item->setText(timestr);
        ui->listWidget_SchTimeList->insertItem(ui->listWidget_SchTimeList->currentRow(), item);
    } else {
        ;
    }
}

void MainWindow::on_pushButton_Del_clicked()
{
    // one of the last?
    if (1 >= ui->listWidget_SchTimeList->count()) return;

    // delete currentRow item
    ui->listWidget_SchTimeList->takeItem(ui->listWidget_SchTimeList->currentRow());
}

void MainWindow::on_pushButton_RegistHoliday_clicked()
{
    RegistHoliday_Dialog *dlg = new RegistHoliday_Dialog(Holidays, this);

    int i = dlg->exec();
    if (i == QDialog::Accepted) {
        qDebug() << "Ok";
        Holidays.clear();
        qDebug() << Holidays;
        Holidays.append(dlg->HolidaysList());
        // qDebug() << Holidays;
    } else if (i == QDialog::Rejected) {
        // qDebug() << "Cancel";
        ;
    } else {
        // qDebug() << "Unkown";
        ;
    }

    delete dlg;
}

void MainWindow::on_lineEdit_FilePath_textChanged(const QString &arg1)
{
    if (arg1.isEmpty()) {
        ui->pushButton->setEnabled(false);
        ui->pushButton_TestPlay->setEnabled(false);
    } else {
        ui->pushButton->setEnabled(true);
        ui->pushButton_TestPlay->setEnabled(true);
    }
}

void MainWindow::on_hSlider_Volume_valueChanged(int value)
{
    float volume = value / 100.0;
    ThPlayMusic->SetSoundVolume(volume);
#if 0
    volume = ThPlayMusic->GetSoundVolume() * 100.0;
    qDebug() << "Volume" << volume;
#endif
}
