package jp.coocan.la.aide.android.snapcard;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.net.Uri;
import android.os.Environment;
import android.util.DisplayMetrics;
import android.view.ViewGroup.LayoutParams;
import android.view.animation.Animation;
import android.view.animation.Animation.AnimationListener;
import android.view.animation.AnimationUtils;
import android.widget.ImageView;


public class MainView extends ImageView implements AnimationListener {

	private Bitmap cardBitmap;
	private float cardPixelWidth  = 0f;
	private float cardPixelHeight = 0f;

	private int   screenBaseX = 0;
	private int   screenBaseY = 0;
	private float dragBaseX = 0f;
	private float dragBaseY = 0f;

	private boolean isSnapouted = false;

	private Animation snapinAnimation;
	private Animation snapoutAnimation;


	public MainView(Context context) {
		super(context);

		DisplayMetrics displayMetrics = ApplicationContainer.getDisplayMetrics();

		//画面DPIから画面上の名刺のピクセルサイズと位置を算出
		//		cardPixelWidth  = displayMetrics.xdpi * displayMetrics.scaledDensity * 2.17f;
		//		cardPixelHeight = displayMetrics.ydpi * displayMetrics.scaledDensity * 3.56f;
		cardPixelWidth  = 2.17f * 160 * displayMetrics.scaledDensity;
		cardPixelHeight = 3.56f * 160 * displayMetrics.scaledDensity;
		screenBaseX = (int)(((float)displayMetrics.widthPixels  - cardPixelWidth)  / 2.0f);
		screenBaseY = (int)(((float)displayMetrics.heightPixels - cardPixelHeight) / 2.0f);

		//Viewの表示位置、サイズの調整
		Matrix matrix = new Matrix();
		matrix.setScale( 1.0f, 1.0f );
		matrix.setTranslate( screenBaseX, screenBaseY );
		this.setImageMatrix( matrix );
		this.setScaleType( ScaleType.MATRIX );

		//名刺画像をロード
		String imageFilename = Environment.getExternalStorageDirectory() + "/SnapCard/card.jpg";
		Uri uri = Uri.fromFile( new File( imageFilename ) );
		loadCardBitmap( uri );
		resetPosition();

		LayoutParams params = new LayoutParams( (int)cardPixelWidth, (int)cardPixelHeight );
		this.setLayoutParams(params);

		//アニメーションのロード
		snapinAnimation  = AnimationUtils.loadAnimation( context, R.anim.snapin  );
		snapinAnimation.setAnimationListener( this );
		snapoutAnimation = AnimationUtils.loadAnimation( context, R.anim.snapout );
		snapoutAnimation.setAnimationListener( this );
	}

	public void loadCardBitmap( Uri uri ){

		Bitmap src = null;

		//現在の名刺画像を除去
		if( cardBitmap != null ){
			cardBitmap.recycle();
			cardBitmap = null;
		}

		//URIから基名刺画像を取得
		if( uri != null ){
			try{
				BitmapFactory.Options opts = new BitmapFactory.Options();
				InputStream is;

				opts.inJustDecodeBounds = true;
				is = ApplicationContainer.getContentResolver().openInputStream( uri );
				try{
					BitmapFactory.decodeStream( is, null, opts );					
				}finally{
					is.close();
				}

				//名刺のピクセルサイズからとBitmapサイズからスケールを計算
				int scaleX = (int)Math.floor( (double)opts.outWidth  / (double)cardPixelWidth  );
				int scaleY = (int)Math.floor( (double)opts.outHeight / (double)cardPixelHeight );

				opts.inJustDecodeBounds = false;
				opts.inSampleSize = Math.min( scaleX, scaleY );

				is = ApplicationContainer.getContentResolver().openInputStream( uri );
				try{
					src = BitmapFactory.decodeStream( is, null, opts );					
				}finally{
					is.close();
				}

			}catch(IOException e){
				src = null;
			}
		}

		//URIから取得できない場合はリソースから取得
		if( src == null ){
			//リソースから画像を取得
			src = BitmapFactory.decodeResource(
					getResources(), 
					R.drawable.dftcard
					);
		}

		//名刺のピクセルサイズからとBitmapサイズからスケールを計算
		float scaleX = cardPixelWidth  / (float)src.getWidth();
		float scaleY = cardPixelHeight / (float)src.getHeight();
		float scale  = Math.min( scaleX, scaleY );

		//名刺画像を作成
		cardBitmap = Bitmap.createScaledBitmap(
				src, 
				(int)((float)src.getWidth()  * scale),
				(int)((float)src.getHeight() * scale),
				true
				);

		src.recycle();
		src = null;

		this.setImageBitmap( cardBitmap );
	}


	public void onStartDrag(float x, float y){
		dragBaseX = x;
		dragBaseY = y;
	}

	public void onDragging(float x, float y){
		float offsetX = x - dragBaseX;
		float offsetY = y - dragBaseY;
		this.layout( 
				(int)(screenBaseX + offsetX), 
				(int)(screenBaseY + offsetY), 
				(int)(screenBaseX + offsetX) + (int)cardPixelWidth, 
				(int)(screenBaseX + offsetY) + (int)cardPixelHeight
				);
	}

	public void onEndDrag(float x, float y){
		if( (cardPixelWidth / 3) <= (x - dragBaseX) ){
			startSnapout();
		}
		else{
			resetPosition();
		}
	}

	public void onShake(){
		startSnapin();
	}


	public void startSnapout(){
		if( ! isSnapouted ){
			this.startAnimation( snapoutAnimation );
		}
	}

	public void endSnapout(){
		this.setVisibility( INVISIBLE );
		isSnapouted = true;
	}


	public void startSnapin(){
		if( isSnapouted ){
			resetPosition();
			this.setVisibility( VISIBLE );
			this.startAnimation( snapinAnimation );
		}
	}

	public void endSnapin(){
		isSnapouted = false;
	}


	public void resetPosition(){
		dragBaseX = 0;
		dragBaseY = 0;

		layout(
				screenBaseX, 
				screenBaseY, 
				screenBaseX + (int)cardPixelWidth, 
				screenBaseY + (int)cardPixelHeight 
				);
	}

	public void onAnimationEnd(Animation animation) {

		if( animation.equals( snapinAnimation ) ){
			endSnapin();
		}
		else if( animation.equals( snapoutAnimation ) ){
			endSnapout();
		}
	}

	public void onAnimationRepeat(Animation animation) {
		// TODO 自動生成されたメソッド・スタブ

	}

	public void onAnimationStart(Animation animation) {
		// TODO 自動生成されたメソッド・スタブ

	}
}
