﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;

namespace Yubeshi.Ubx
{
    public class NavSvInfo : Packet
    {
        #region type definitions

        public enum ChipGeneration : byte
        {
            Antaris = 0,
            Ublox5 = 1,
            Ublox6 = 2,
        }

        public struct SvInfo
        { 

            [Flags]
            public enum Status : byte
            {
                SvUsed = 0x01,
                DifferentialCorrection = 0x02,
                OrbitInformation = 0x04,
                Ephemeris = 0x08,
                Unhealthy = 0x10,
                AlmanacPlus = 0x20,
                AutonomousOrbitPrediction = 0x40,
            }

            private byte[] Raw;

            public int Channel
            {
                get 
                { 
                    return Raw[0];
                }
            }

            public int SatelliteID
            {
                get
                {
                    return Raw[1];
                }
            }

            public Status StatusFlags
            {
                get
                {
                    return (Status)Raw[2];
                }
            }

            public int Quality
            {
                get
                {
                    return Raw[3] & 0x0F;
                }
            }

            public int SignalStrength
            {
                get
                {
                    return Raw[4];
                }
            }

            public Degree Elevation
            {
                get
                {
                    return (sbyte)(Raw[5]);
                }
            }

            public Degree Azimuth
            {
                get
                {
                    return BitConverter.ToInt16(Raw, 6);
                }
            }

            public double PseudoRangeResidual
            {
                get
                {
                    return BitConverter.ToInt32(Raw, 8) * 0.01;
                }
            }

            public void Set(byte[] raw)
            {
                Raw = raw;
            }
        }
        #endregion

        #region constructors

        public NavSvInfo(byte[] sentence, int length)
            : base(sentence, length)
        {
            SvInformation = new SvInfo[NumberOfChannels];
            for (int i = 0; i < NumberOfChannels; ++i)
            {
                SvInformation[i].Set(
                                OctetString.Substring(Raw, 14 + 12 * i, 12));
            }
        }

        #endregion

        #region properties

        public decimal TimeOfWeek
        {
            get
            {
                return BitConverter.ToUInt32(Raw, 6 + 0) * 1e-3m;
            }
        }

        public int NumberOfChannels
        {
            get
            {
                return Raw[6 + 4];
            }
        }

        public ChipGeneration Chip
        {
            get
            {
                return (ChipGeneration)(Raw[6 + 5] & 0x7);
            }
        }

        public SvInfo[] SvInformation
        {
            get;
            private set;
        }

        #endregion

        #region public methods

        public static bool TryParse(byte[] sentence, out UnknownPacket packet)
        {
            packet = null;
            if (sentence.Length < 16)
            {
                return false;
            }
            int length = 8 + 12 * sentence[6 + 4];
            return TryParse(sentence, out packet, 
                                        MessageID.NavSvInfo, length, Build);
        }

        #endregion

        #region private methods

        private static Packet Build(byte[] sentence, int length)
        {
            return new NavSvInfo(sentence, length);
        }

        #endregion
    }
}
