﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;

namespace Yubeshi
{
    /// <summary>
    ///     Height(or Altitude) in metre
    /// </summary>
    public struct Height
    {
        #region type definitions
        public enum Base : int
        {
            Unknown = 0,
            EarthCenter = 1,
            MeanSeaLevel = 2,
            Geoid = 3,
            Ellipsoid = 0x100,
            BesselEllipsoid = Ellipsoid | 41,
            Grs80Ellipsoid = Ellipsoid | 80,
            Wgs84Ellipsoid = Ellipsoid | 84,
        }
        #endregion

        #region fields
        private double height;
        private Base baseLevel;
        #endregion

        #region constructors
        public Height(double height)
            :this(height, Base.Unknown)
        {
        }

        public Height(double height, Base baseLevel)
        {
            
            this.height = height;
            this.baseLevel = baseLevel;
        }
        #endregion

        #region operators
        public static implicit operator Height(double height)
        {
            return new Height(height);
        }

        public static implicit operator double(Height height)
        {
            return height.height;
        }
        
        public static Height operator +(Height left, double right)
        {
            return new Height(left.height + right, left.baseLevel);
        }

        public static Height operator +(double left, Height right)
        {
            return new Height(right.height + left, right.baseLevel);
        }

        public static double operator -(Height left, Height right)
        {
            if (left.baseLevel != right.baseLevel)
            {
                if (left.baseLevel != Base.Unknown &&
                                            right.baseLevel != Base.Unknown)
                {
                    throw new InvalidOperationException(
                                                    "mismatched base level");
                }
            }
            return left.height - right.height;
        }
        #endregion

        #region properties

        public Base BaseLevel
        {
            get
            {
                return baseLevel;
            }
        }

        public bool IsAltitude
        {
            get
            {
                return (baseLevel == Base.MeanSeaLevel) ||
                                                    (baseLevel == Base.Geoid);
            }
        }
        #endregion

        #region public methods
        #endregion
    }
}
