﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;

namespace Yubeshi
{
    public class GeodeticCoordinate
    {
        #region fields
        private const double a = Constants.Wgs84.SemiMajorAxisA;
        private const double b = Constants.Wgs84.SemiMajorAxisB;
        private const double e1sq = Constants.Wgs84.FirstEccentricitySquared;
        private const double pi = Constants.Wgs84.Pi;

        #endregion
        #region constructors

        public GeodeticCoordinate(Degree latitude, Degree longitude)
            : this(latitude, longitude, Double.NaN)
        {
        }

        public GeodeticCoordinate(
                            Degree latitude, Degree longitude, Height altitude)
        {
            Latitude = latitude;
            Longitude = longitude;
            Altitude = altitude;
        }
        #endregion

        #region properties

        /// <summary>
        ///    geographic latitude in degree
        /// </summary>
        public Degree Latitude
        {
            get;
            set;
        }

        /// <summary>
        ///    geocentric latitude in degree
        /// </summary>
        public Degree GeocentricLatitude
        {
            get
            {
                return Math.Atan((1 - e1sq) * Math.Tan(Latitude.Radian));
            }
        }

        /// <summary>
        ///     longitude in degree
        /// </summary>
        public Degree Longitude
        {
            get;
            set;
        }

        /// <summary>
        ///     altitude in metre
        /// </summary>
        public Height Altitude
        {
            get;
            set;
        }
        #endregion

        #region public method
        public EcefCoordinate ToEcefCoordinate()
        {
            double h = Double.IsNaN(Altitude) ? 0 : Altitude;
            double sinPhi = Math.Sin(Latitude.Radian);
            double cosPhi = Math.Cos(Latitude.Radian);
            double khi = Math.Sqrt(1.0 - e1sq * sinPhi * sinPhi);
            double n = a / khi;
            double r = (n + h) * cosPhi;
            double x = r * Math.Cos(Longitude.Radian);
            double y = r * Math.Sin(Longitude.Radian);
            double z = (n * (1 - e1sq) + h) * sinPhi;
            return new EcefCoordinate(x, y, z);
        }


        #endregion

        #region private methods
        #endregion
    }
}
