class Cache < ActiveRecord::Base

 private
	ICON = 0
	PAGELIST = 1
	LOCATION = 2
 
	# キャッシュを探す なければブロックを実行してキャッシュに保存(ブロック無ければnil)
	def self.fetch(user_id, itemtype, path, location, &block)
		rs = where(user_id:user_id, itemtype:itemtype, path:path, location:location).first
		if rs
			data = rs.data
			rs.access_at = Time.now
			rs.save
		elsif block_given?
			data =  block.call
			create(
				user_id: user_id,
				itemtype: itemtype,
				path: path,
				location: location,
				data: data,
				access_at: Time.now
			)
		else
			data = nil
		end
		return data
	end

	# キャッシュを破棄
	def self.dirty(user_id, itemtype, path, location)
		transaction do
			rs = where(user_id:user_id, itemtype:itemtype, path:path, location:location)
			rs.each { |r| r.destroy }
		end
	end

 public
 	# 当該ユーザのキャッシュを全クリア
	def self.do_clear(user_id, vpath=nil, location=nil)
		transaction do
			if vpath.blank?
				if location.blank?
					delete_all(user_id:user_id)
				end
			else
				if location.blank?
					delete_all(user_id:user_id, path:vpath)
				else
					delete_all(user_id:user_id, path:vpath, location:location)
				end
			end
		end
	end

	# キャッシュの保存期限切れ処理
	def self.do_expire(user_id)
		transaction do
			where(user_id:user_id).where(["access_at < ?", 20.day.ago]).delete_all
		end
	end

	# アイコン用キャッシュ
	def self.fetch_icon(user_id, path, location=nil, &block)
		location = ""  if location.blank?
		return fetch(user_id, ICON, path, location, &block)
	end

	# 頁リスト用キャッシュ
	def self.fetch_pagelist(user_id, path, &block)
		if block_given?
			r = fetch(user_id, PAGELIST, path, "") do
				YAML.dump(block.call)
			end
		else
			r = fetch(user_id, PAGELIST, path, "")
		end
		return r ? YAML.load(r) : nil
	end

	# 表示位置用キャッシュ
	def self.fetch_location(user_id, path)
		r = fetch(user_id, LOCATION, path, "")
		return r.nil? ? nil : r.to_i
	end

	# 表示位置用キャッシュ(格納)
	def self.store_location(user_id, path, location)
		location = location.to_s  if !location.nil?
		rs = where(user_id:user_id, itemtype:LOCATION, path:path, location:"").first
		if rs
			rs.data = location
			rs.access_at = Time.now
			rs.save
		else
			create(
				user_id: user_id,
				itemtype: LOCATION,
				path: path,
				location: "",
				data: location,
				access_at: Time.now
			)
		end
	end

end
