class Book
 private
	def initialize(rpath)
		@rpath = rpath
	end

 public
 	attr_reader :rpath

	def self.factory(rpath)
		case type(rpath)
		 when :images
			return BookOfFiles.new(rpath)
		 when :pdf
			return BookOfPdf.new(rpath)
		 when :zip
			return BookOfZip.new(rpath)
		 else
			return Book.new(rpath)
		end
	end
 
	# 本の種類を返す ※本の実在確認はしない
	def self.type(path)
		b = File.basename(path)
		if b == "*img"
			return :images
		elsif extname_match?(YOMEYA_IMAGE_EXT, b)
			return :image
		elsif extname_match?(YOMEYA_PDF_EXT, b)
			return :pdf
		elsif extname_match?(YOMEYA_ZIP_EXT, b)
			return :zip
		elsif extname_match?(YOMEYA_TXT_EXT, b)
			return :text
		elsif extname_match?(YOMEYA_HTML_EXT, b)
			return :html
		elsif extname_match?(YOMEYA_AUDIO_EXT, b)
			return :audio
		elsif extname_match?(YOMEYA_VIDEO_EXT, b)
			return :video
		else
			return nil
		end
	end

#	def bookinfo
#		type = class.type(@rpath)
#		pl = pagelist
#		pages = pl.count
#		if pages > 0
#
#
#		else
#			location = ""
#			page = 0
#			read_start = nil
#			read_end = nil
#		end
#
#		return {
#			type: type,
#			pages: pages,
#			location: location,
#			page: page,
#			read_start: read_start,
#			read_end: read_end,
#		}
#	end

	def pagelist;							[]; end
	def extract(location);					nil; end
	def extract_thumbnail(w, h, l=nil);		nil; end
	def writable?;							false; end
	def delete(location);					false; end
	def update(location, data);				false; end
	def rotate(location, degree);			false; end

	# class methods
	def self.bookinfo(rpath)
		factory(rpath).bookinfo
	end
	def self.pagelist(rpath)
		factory(rpath).pagelist
	end
	def self.extract(rpath, location)
		factory(rpath).extract(location)
	end
	def self.extract_thumbnail(rpath, width, height, location=nil)
		factory(rpath).extract_thumbnail(width, height, location)
	end
	def self.writable?(rpath)
		factory(rpath).writable?
	end
	def self.delete(rpath, location)
		factory(rpath).delete(location)
	end
	def self.update(rpath, location, data)
		factory(rpath).update(location, data)
	end
	def self.rotate(rpath, location, degree)
		factory(rpath).rotate(location, degree)
	end
end

############################################################

class BookOfFiles < Book
 private
	def initialize(rpath)
		if File.basename(rpath) == YOMEYA_IMAGES
			@rpath = File.dirname(rpath)
		else
			@rpath = rpath
		end
	end

 public
	def pagelist
		files = []
		dir = wildcard_escape(@rpath)
		Dir.glob(File.join(dir, wildcard_str(YOMEYA_IMAGE_EXT))) do |child_rpath|
			if File.file?(child_rpath)
				files << File.basename(child_rpath)
			end
		end
		return files.sort_by! { |v| key4sort(v) }
	end

	def extract(location)
		r = File.binread(reg_rpath(@rpath, location))
		return r  if r.present?

		return asset_image_data("icon_file_missing.png")
	end

	def extract_thumbnail(width, height, location=nil)
		if location.present?
			p = location
		else
			p = pagelist.first
		end
		return extract(p)  if p.present?

		return asset_image_data("icon_file_missing.png")
	end

	def delete(location)
		src_filename = reg_rpath(@rpath, location)
		return false  if !File.file?(src_filename)

		n = 1
		while true
			dst_filename = src_filename + ".skip" + ((n == 1) ? "" : n.to_s)
			break  if !File.exist?(dst_filename)
			n += 1
		end

		FileUtils.mv(src_filename, dst_filename)
		return true
	end

	def update(location, data)
		filename = reg_rpath(@rpath, location)
		org_filename = filename + ".org"

		if !File.exist?(org_filename)
			FileUtils.cp(filename, org_filename, preserve:true)
		end

		File.open(filename, "wb:BINARY") do |file|
			file.write(data)
		end
	end

	def rotate(location, degree)
		src_filename = reg_rpath(@rpath, location)
		org_filename = src_filename + ".org"
		return false  if !File.file?(src_filename)

		temp = Tempfile.new("yomeya_temp")
		temp.binmode

		cmd = [JPEGTRAN_CMD]
		cmd += ["-perfect", "-optimize", "-progressive"]
		cmd += ["-rotate", degree.to_s]
		cmd += ["-outfile", temp.path]
		cmd += [src_filename]
		r = system(*cmd)
		if !r
			cmd = [CONVERT_CMD]
			cmd += ["-define", "jpeg:preserve-settings"]
			cmd += ["-interlace", "line"]
			cmd += ["-rotate", degree.to_s]
			cmd += [src_filename]
			cmd += [temp.path]
			r = system(*cmd)
		end
		if r
			if !File.exist?(org_filename)
				FileUtils.cp(src_filename, org_filename, preserve:true)
			end
			FileUtils.mv(temp.path, src_filename)
		end
		temp.close!
		return r
	end

end

############################################################

class BookOfPdf < Book
 public
	def pagelist
		n = PDF::Reader.new(@rpath).page_count
		return (1 .. n).map { |n| "#{n}p" }
	end

	def extract(location)
		filename = "#{@rpath}[#{location.to_i-1}]"
		filename.encode!(Encoding.find("locale"))
		imagelist = Magick::Image.read(filename) do
			self.density = "288"
		end
		image = imagelist[0]

		# 背景を白に
		bg = Magick::Image.new(image.columns, image.rows) do
			self.background_color = "white"
		end
		image.composite!(bg, 0, 0, Magick::DstOverCompositeOp)

		r = image.to_blob do
			self.format = "PNG"
		end
		return r
	end

	def extract_thumbnail(width, height, location=nil)
		if location.present?
			p = location.to_i - 1
		else
			p = 0
		end

		filename = "#{@rpath}[#{p}]"
		filename.encode!(Encoding.find("locale"))
		imagelist = Magick::Image.read(filename) do
			self.density = "36"
		end
		image = imagelist[0]

		# 背景を白に
		bg = Magick::Image.new(image.columns, image.rows) do
			self.background_color = "white"
		end
		image.composite!(bg, 0, 0, Magick::DstOverCompositeOp)

		r = image.to_blob do
			self.format = "PNG"
		end
		return r  if r.present?

		return asset_image_data("icon_file_pdf.png")
	end
end

############################################################

class BookOfZip < Book

 private
	FN_ENC = Encoding::WINDOWS_31J

 public
	def pagelist
		files = []

		Zip::File.open(@rpath) do |zipfile|
			zipfile.each do |entry|
				if entry.ftype == :file
					filename = entry.name.dup
					filename.force_encoding(Encoding::UTF_8)
					if !filename.valid_encoding?
						tmp = filename.dup
						tmp.force_encoding(Encoding::ASCII_8BIT)
						tmp.gsub!("/", "\\")			# なみだぐましい
						tmp.force_encoding(FN_ENC)
						if tmp.valid_encoding?
							tmp.gsub!("\\", "/")			# なみだぐましい
							filename = tmp
						end
					end
#					# gp_flagsを信じる場合(Macで作ったアーカイブが正しく判定されない)
#					if (entry.gp_flags & 0b100000000000) != 0
#						filename.force_encoding(Encoding::UTF_8)
#					else
#						filename.gsub!("/", "\\")			# なみだぐましい
#						filename.force_encoding(FN_ENC)
#						filename.gsub!("\\", "/")			# なみだぐましい
#					end
					filename.encode!
					basename = File.basename(filename)
					if !basename.start_with?("._") && extname_match?(YOMEYA_IMAGE_EXT, basename)
						files << filename
					end
				end
			end
		end

		return files.sort_by! { |v| key4sort(v) }
	end

	def extract(location)
		r = nil

		Zip::File.open(@rpath) do |zipfile|
			begin
				l = location.encode(Encoding::UTF_8).force_encoding(Encoding::ASCII_8BIT)
				r = zipfile.read(l)
			 rescue Errno::ENOENT
				begin
					l = location.encode(FN_ENC).force_encoding(Encoding::ASCII_8BIT)
					l.gsub!("\\", "/")			# なみだぐましい
					r = zipfile.read(l)
				 rescue Errno::ENOENT
					# NOP
				end
			end
		end

		return r
	end

	def extract_thumbnail(width, height, location=nil)
		if location.present?
			p = location
		else
			p = pagelist.first
		end
		return extract(p)  if p.present?

		return asset_image_data("icon_file_zip.png")
	end
end
