public

# app/assets/images 以下のファイルの内容を得る
def asset_image_data(filename)
	filename = File.absolute_path(File.join("app/assets/images", filename), Rails.root)
	return File.binread(filename)
end

# パス名を正規化する(virt_path用)
def reg_vpath(*path)
	path = File.join(*path)
	path.gsub!(/\.\.[\/\\]/, "")
	if os_is_windows?
		return File.absolute_path(path, "/").from(3)
	end
	return File.absolute_path(path, "/").from(1)
end

# パス名を正規化する(real_path用)
def reg_rpath(*path)
	path = File.join(*path)
	return File.absolute_path(path, "/")
end

# locationを正規化する
def reg_location(location)
	location = location.gsub(/\.\.[\/\\]/, "").strip
	return location
end

# ユーザのroots(最上位階層)を作成する
def roots(user_id:nil)
	user_id ||= session[:user_id]
	r = []
	Root.where(user_id:user_id).each do |root|
		r << [
			root.virt_path,
			root.real_path,
		]
	end
	return r
end

# 仮想パスを実際のパスに変換する
def vpath2rpath(vpath, user_id:nil)
	user_id ||= session[:user_id]
	first, rest = vpath.split("/", 2)
	rest ||= ""
	root = Root.where(user_id:user_id, virt_path:first).first
	return nil  if root.nil? # 見つからない場合は何かがおかしいのでnilを返してエラーとする
	return reg_rpath(root.real_path, rest)
end

# 実際のパスを仮想パスに変換する
def rpath2vpath(rpath)
	roots.each do |root_vpath, root_rpath|
		if root_rpath == rpath || rpath.start_with?(root_rpath + "/")
			return reg_vpath(root_vpath, rpath.from(root_rpath.length))
		end
	end
	return "" # 見つからない場合は仮想の世界に現実の世界の情報は返さない
end

# 有効な仮想パスか(ファイルが存在するか)判定する
def valid_vpath?(vpath)
	return true  if vpath.blank?	# 空(root)ならT

	return true  if vpath.is_any_of("*read", "*bookmark", "*tag") # 暫定

	rpath = vpath2rpath(vpath)
	return false  if rpath.blank?	# rpathに変換できなければF

	case Book.type(rpath)
	 when :images
		dir = File.dirname(rpath)
		return true  if File.directory?(dir)	# imagesの場合、ディレクトリが実在すればT
	 else
		return true  if File.exist?(rpath)		# 実在すればT (ディレクトリでもT)
	end

	return false
end

# 有効なディレクトリのパスか判定する
def valid_directory_path?(vpath)
	return true  if vpath.blank?
	return false  if !valid_vpath?(vpath)
	rpath = vpath2rpath(vpath)
return false if rpath.blank?
	return true  if File.directory?(rpath)
	return false
end

# 有効な本のパスか判定する
def valid_book_path?(vpath)
	return false  if vpath.blank?
	return false  if !valid_vpath?(vpath)

	rpath = vpath2rpath(vpath)
return false if rpath.blank?
	case Book.type(rpath)
	 when nil
		return false
	 when :images
		dir = File.dirname(rpath)
		return true  if File.directory?(dir)
	 else
		return true  if File.file?(rpath)
	end

	return false
end

# 有効な頁のパス/locationか判定する
# 頁の実在チェックはしていない
def valid_page_path?(vpath, location)
	return false  if !valid_book_path?(vpath)
	return true  if !location.blank?
	return false
end

# 仮想パス以下のエントリ一覧を仮想パス(key)/実際のパス(val)で得る
def vpath_glob(vpath, user_id:nil)
	user_id ||= session[:user_id]
	first, rest = vpath.split("/", 2)
	rest ||= ""

	root = Root.where(user_id:user_id, virt_path:first).first
	return nil  if root.nil?

	r = {}
	real_path_length = root.real_path.length
	dir = wildcard_escape(reg_rpath(root.real_path, rest))
	Dir.glob(File.join(dir, "*")) do |a_rpath|
		a_vpath = reg_vpath(first, a_rpath.from(real_path_length))
		r[a_vpath] = a_rpath
	end
	return r
end

########################################################################

# イメージファイルしかないディレクトリかどうか判定する
# イメージファイル以外が存在するか、ディレクトリが空の場合はfalseを返す
def image_only_dir?(vpath:nil, rpath:nil)
	rpath = vpath2rpath(vpath)  if rpath.nil?
	dir = wildcard_escape(rpath)

	Dir.glob(File.join(dir, "*/")) do
		return false
	end

	Dir.glob(File.join(dir, wildcard_str(YOMEYA_ARCHIVE_EXT + YOMEYA_OTHER_EXT))) do
		return false
	end

	Dir.glob(File.join(dir, wildcard_str(YOMEYA_IMAGE_EXT))) do
		return true
	end

	return false
end

# pagelist中の何枚目の画像かを得る
# 一致するものがなかった場合、最も近いものを得る
def find_pagelist_idx(pagelist, location)
	return nil  if pagelist.empty?

	r = pagelist.index(location)
	return r  if r

	location_key = key4sort(location)
	pagelist.each.with_index do |page, i|
		if (key4sort(page) <=> location_key) != -1
			return i
		end
	end

	return pagelist.size - 1
end

# vpathで示される本の頁一覧を得る(キャッシュなし)
def _get_pagelist(vpath:nil, rpath:nil)
	rpath = vpath2rpath(vpath)  if rpath.nil?
	return Book.factory(rpath).pagelist
end

# vpathで示される本の頁一覧を得る
def get_pagelist(vpath)
	Cache.transaction do
		return Cache.fetch_pagelist(session[:user_id], vpath) do
			_get_pagelist(vpath:vpath)
		end
	end
end

########################################################################

# ある本のある頁までを読んだことを宣言する
def update_page_read(vpath, location)
	user_id = session[:user_id]

	current_vpath = session[:last_path]
	if current_vpath != vpath # pathが変化した場合、last_path_fromも更新
		session[:last_path] = vpath
		session[:last_path_from] = current_vpath
	end
	session[:last_location] = location

	User.transaction do
		user = User.find(user_id)
		user.last_path = vpath
		user.last_location = location
		user.save!

		reading = Reading.yomeya_find_or_new(user_id, vpath)
		last_page = get_pagelist(vpath).last
		now = Time.now
		reading.location = location
		reading.start_at = now  if reading.start_at.blank?
		reading.touch_at = now
		reading.end_at = now  if reading.end_at.blank? && location == last_page
		reading.save!

		Log.add(user_id, "Moveto: [#{vpath}] [#{location}]")
	end
end

# ファイルリストを表示したことを宣言する
def update_filelist_visit(vpath, location)
	user_id = session[:user_id]

	current_vpath = session[:last_path]
	if current_vpath != vpath # 前回と異なるpathに移動した場合、last_path_fromも更新
		session[:last_path] = vpath
		session[:last_path_from] = current_vpath
	end
	session[:last_location] = location

	User.transaction do
		user = User.find(user_id)
		user.last_path = vpath
		user.last_location = location
		user.save!

		Cache.store_location(user_id, vpath, location)

		Log.add(user_id, "Moveto: [#{vpath}]")
	end
end

class ApplicationController < ActionController::Base
	# Prevent CSRF attacks by raising an exception.
	# For APIs, you may want to use :null_session instead.
	protect_from_forgery with: :exception

 protected
	# 認証済かをチェック
	def auth_check
		return  if params[:controller] == "auth"
		redirect_to(auth_path)  if session[:user_id].blank?
	end
	before_action :auth_check

end
