package jp.gr.java_conf.ykhr.listfindbugs;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

public class Main {

    public static void main(String[] args) {
        try {
            new Main().exec();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void exec() throws ReadException {
        List<DecoratorVo> decorators = loadDetector();
        Map<String, MessageVo> message = loadMessage();
        output(decorators, message);
    }
    
    private void output(List<DecoratorVo> decorators, Map<String, MessageVo> message) {
        System.out.println("Detector,BugType,BugCode,Category,Description,Detail");
        
        Collections.sort(decorators, new Comparator<DecoratorVo>() {
            public int compare(DecoratorVo o1, DecoratorVo o2) {
                return o1.getSimpleName().compareTo(o2.getSimpleName());
            }
        });
        
        for (DecoratorVo decoratorVo : decorators) {
            if (decoratorVo.isHidden()) {
                continue;
            }
            
            String detector = decoratorVo.getSimpleName();
            boolean first = true;
            
            for (BugPatternVo bugPattern : decoratorVo.getBugPatterns()) {
                String type = bugPattern.getType();
                MessageVo messageVo = message.get(type);
                
                if (messageVo == null) {
                    continue;
                }
                
                if (first) {
                    System.out.print(toCsvItemString(detector));
                    first = false;
                }
                System.out.println("," + toCsvItemString(type) 
                        + "," + toCsvItemString(bugPattern.getAbbrev())
                        + "," + toCsvItemString(bugPattern.getCategory())
                        + "," + toCsvItemString(messageVo.getShortDescription())
                        + "," + toCsvItemString(messageVo.getDetail()));
            }
        }
    }

    private List<DecoratorVo> loadDetector() throws ReadException {
        if (!existResource("findbugs.xml")) {
            throw new ReadException("findbugs.xml is not found.");
        }
        
        FindbugsHandler fbHandler = new FindbugsHandler();
        parseXml("findbugs.xml", fbHandler);
        List<DecoratorVo> decorators = fbHandler.getDecorators();
        return decorators;
    }

    private Map<String, MessageVo> loadMessage() throws ReadException {
        if (!existResource("messages.xml")) {
            throw new ReadException("message.xml is not found.");
        }
        
        MessageHandler handler = new MessageHandler();
        parseXml("messages.xml", handler);
        Map<String, MessageVo> message = handler.getMessages();

        Locale locale = Locale.getDefault();
        String localizedResource = "messages_" + locale.getLanguage() + ".xml";
        if (existResource(localizedResource)) {
            handler.reset();
            parseXml(localizedResource, handler);
            Map<String, MessageVo> localizedMessage = handler.getMessages();
            
            // overwrite message.
            message.putAll(localizedMessage);
        }
        
        return message;
    }
    
    private void parseXml(String resource, DefaultHandler handler) throws ReadException {
        InputStream is = null;
        try {
            is = getClass().getClassLoader().getResourceAsStream(resource);
            parseXml(is, handler);
        } finally {
            close(is);
        }
    }
    
    private void parseXml(InputStream is, DefaultHandler handler) throws ReadException {
        try  {
            SAXParser parser = SAXParserFactory.newInstance().newSAXParser();
            parser.parse(is, handler);
        } catch (ParserConfigurationException e) {
            throw new ReadException(e);
        } catch(SAXException e) { 
            throw new ReadException(e);
        } catch (IOException e) {
            throw new ReadException(e);
        }
    }
    
    private String toCsvItemString(String str) {
        return wrapString(escape(str.trim()));
    }
    
    private String escape(String str) {
        if (!str.contains("\"")) {
            return str;
        }
        
        return str.replaceAll("\"", "\"\"");
    }
    
    private String wrapString(String str) {
        return "\"" + str + "\"";
    }
    
    private boolean existResource(String name) {
        URL url = getClass().getClassLoader().getResource(name);
        return url != null;
    }
    
    private void close(InputStream is) {
        if (is != null) {
            try {
                is.close();
            } catch (IOException ignore) {
            }
        }
    }
    
}
