/**
 * The MIT License
 * 
 * Copyright (c) 2009 K.Takahashi
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @author K.Takahashi (kyozi0x7c00@gmail.com)
 */
package jp.sourceforge.geneticalgorithm;

import java.util.ArrayList;
import java.util.List;

import jp.sourceforge.geneticalgorithm.travelingsalesmanproblem.TspIndividual;

public class GeneticPool {
	private int _populationNumber = 100;
	private int _currentGenerationNumber = 0;
	private int _bestGenerationNumber = 0;
	private int _generationNumber = 100;
	private double _crossoverRate = 0.8;
	private double _mutationRate = 0.1;

	private List<Individual> _geenPool;

	private Individual _maxIndividual;
	private Individual _bestIndividual;
	private double _sumFitness = 0.0;
	private double _maxFitness = 0.0;
	private double _averageFitness = 0.0;

	public GeneticPool() {
		_geenPool = new ArrayList<Individual>(_populationNumber);
	}

	public void initializePopulation() {
		_geenPool = new ArrayList<Individual>(_populationNumber);
		// TODO
		_bestIndividual = _maxIndividual = new TspIndividual();
		for (int i = 0; i < _populationNumber; i++) {
			Individual individual = new TspIndividual();
			individual.createGeen();
			_geenPool.add(individual);
		}
	}

	public void takeStatistic() {
		_sumFitness = _maxFitness = 0.0;

		for (Individual individual : _geenPool) {
			_sumFitness += individual.getFitness();
			if (individual.getFitness() >= _maxFitness) {
				_maxFitness = individual.getFitness();
				_maxIndividual = individual;
			}
		}
		_averageFitness = _sumFitness / _geenPool.size();

		if (_bestIndividual.getFitness() < _maxIndividual.getFitness()) {
			_bestIndividual = _maxIndividual;
			_bestGenerationNumber = _currentGenerationNumber;
		}
	}

	public void nextGeneration() {
		int i = 0;
		do {
			if (flip(_crossoverRate)) {
				Individual[] parent = selectParents();
				Individual[] child = crossoverChromosome(parent[0], parent[1]);
				if (flip(_mutationRate)) {
					child[0].mutation();
					child[1].mutation();
				}
				child[0].evaluation();
				child[1].evaluation();
				replace(child[0], child[1]);
			}
			i += 2;
		} while (i < _populationNumber);
		_currentGenerationNumber++;
	}

	private Individual[] selectParents() {
		Individual[] parent = new TspIndividual[2];
		double sum;
		double roulette1 = Math.random() * _sumFitness;
		double roulette2 = Math.random() * _sumFitness;
		
		sum = 0.0;
		for (Individual individual : _geenPool) {
			sum += individual.getFitness();
			parent[0] = individual;
			if (sum >= roulette1) {
				break;
			}
		}
		
		sum = 0.0;
		for (Individual individual : _geenPool) {
			sum += individual.getFitness();
			parent[1] = individual;
			if (sum >= roulette2) {
				break;
			}
		}

		return parent;
	}

	private Individual[] crossoverChromosome(Individual individual0, Individual individual1) {
		Individual[] child = new Individual[2];
		child[0] = new TspIndividual();
		child[0].createGeen();
		child[1] = new TspIndividual();
		child[1].createGeen();

		return child;
	}

	private void replace(Individual individual0, Individual individual1) {
		int minIndividual0 = 0;
		int minIndividual1 = 1;
		double minFitness0 = _geenPool.get(0).getFitness();
		double minFitness1 = _geenPool.get(1).getFitness();

		for (int i = 0; i < _geenPool.size(); i++) {
			Individual individual = _geenPool.get(i);
			if (minFitness0 > individual.getFitness()) {
				minIndividual0 = i;
			} else if (minFitness1 > individual.getFitness()) {
				minIndividual1 = i;
			}
		}
		_geenPool.set(minIndividual0, individual0);
		_geenPool.set(minIndividual1, individual1);
	}

	private boolean flip(double probability) {
		if (Math.random() > probability) {
			return false;
		} else {
			return true;
		}
	}

	public int getCurrentGenerationNumber() {
		return _currentGenerationNumber;
	}

	public int getGenerationNumber() {
		return _generationNumber;
	}
}
