package hiro.yoshioka.sdh;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

public class StringDataHolder {
	// ****************************************************************************
	// publicフィールド
	// ****************************************************************************
	/** 処理結果NG */
	public static final int NG_VALUE = -1;

	// ****************************************************************************
	// privateフィールド
	// ****************************************************************************
	/** データ保持用ハッシュマップ */
	private Map<String, List<String>> hash = new HashMap<String, List<String>>();

	/** キー保持用配列 */
	String[] key;

	// ****************************************************************************
	// コンストラクタ
	// ****************************************************************************
	// ****************************************************************************
	/**
	 * ハッシュテーブルのキー設定用コンストラクタ
	 * 
	 * @param argNames
	 *            キー名
	 * @throws NullPointerException
	 *             キー名にNULLが含まれる場合
	 */
	// ****************************************************************************
	public StringDataHolder(String[] argNames) throws NullPointerException {
		if (argNames == null) {
			throw new NullPointerException("argNames must be not null");
		}
		key = argNames;
		for (int iName = 0; iName < argNames.length; iName++) {
			if (argNames[iName] == null) {
				throw new NullPointerException("argNames[" + iName
						+ "] must be not null");
			}
			hash.put(argNames[iName], new ArrayList<String>());
		}
	}

	// ****************************************************************************
	// publicメソッド
	// ****************************************************************************
	// ****************************************************************************
	/**
	 * リストの指定された位置に、指定された要素を挿入します。 <BR>
	 * 現在その位置にある要素と後続の要素は右に移動します <BR>
	 * (インデックス値に 1 を加算)。
	 * 
	 * @param argName
	 *            キー名
	 * @param argIndex
	 *            インデックス
	 * @param argValue
	 *            値
	 * @throws IndexOutOfBoundsException
	 *             インデックスが範囲外の場合 (argIndex < 0 || argIndex > size())
	 * @throws NullPointerException
	 *             キー名にNULLが含まれる場合
	 */
	// ****************************************************************************
	public void addData(String argName, int argIndex, String argValue)
			throws IndexOutOfBoundsException, NullPointerException {
		if (argName == null) {
			throw new NullPointerException("argName must be not null");
		}
		List<String> list = hash.get(argName);
		try {
			list.add(argIndex, argValue);
		} catch (IndexOutOfBoundsException e) {
			throw new IndexOutOfBoundsException("IndexOutOfBoundsException "
					+ argIndex);
		}
	}

	// ****************************************************************************
	/**
	 * 最後尾に文字列データを追加します。
	 * 
	 * @param argName
	 *            キー名
	 * @param argValue
	 *            値
	 * @throws NullPointerException
	 *             キー名にNULLが含まれる場合
	 */
	// ****************************************************************************
	public void setData(String argName, String argValue)
			throws NullPointerException {
		if (argName == null) {
			throw new NullPointerException("argName must be not null");
		}
		(hash.get(argName)).add(argValue);
	}

	// ****************************************************************************
	/**
	 * 指定位置に文字列データを設定します。
	 * 
	 * @param argName
	 *            キー名
	 * @param argIndex
	 *            インデックス
	 * @param argValue
	 *            値
	 * @throws IndexOutOfBoundsException
	 *             不正なインデックス指定の場合
	 * @throws NullPointerException
	 *             キー名にNULLが含まれる場合
	 */
	// ****************************************************************************
	public void setData(String argName, int argIndex, String argValue)
			throws IndexOutOfBoundsException, NullPointerException {
		if (argName == null) {
			throw new NullPointerException("argName must be not null");
		}
		List<String> list = hash.get(argName);
		if (argIndex < 0 || list.size() < argIndex) {
			throw new IndexOutOfBoundsException("IndexOutOfBoundsException "
					+ argIndex);
		} else if (list.size() == argIndex) {
			list.add(argValue);
		} else {
			list.set(argIndex, argValue);
		}
	}

	// ****************************************************************************
	/**
	 * 最初の文字列データを取得します。
	 * 
	 * @param argName
	 *            キー名
	 * @return 値
	 * @throws NullPointerException
	 *             キー名にNULLが含まれる場合
	 */
	// ****************************************************************************
	public String getData(String argName) throws NullPointerException {
		if (argName == null) {
			throw new NullPointerException("argName must be not null");
		}
		List<String> list = hash.get(argName);
		return list.get(0);
	}

	public String[][] getAllData() {
		int cnt = getDataCount(key[0]);
		String[][] ret = new String[cnt][key.length];
		for (int irow = 0; irow < cnt; irow++) {
			for (int i = 0; i < key.length; i++) {
				ret[irow][i] = getData(key[i], irow);
			}
		}
		return ret;
	}

	// ****************************************************************************
	/**
	 * 指定した位置の文字列データを取得します。
	 * 
	 * @param argName
	 *            キー名
	 * @param argIndex
	 *            インデックス
	 * @return 値
	 * @throws IndexOutOfBoundsException
	 *             不正なインデックス指定の場合
	 * @throws NullPointerException
	 *             キー名にNULLが含まれる場合
	 */
	// ****************************************************************************
	public String getData(String argName, int argIndex)
			throws IndexOutOfBoundsException, NullPointerException {
		if (argName == null) {
			throw new NullPointerException("argName must be not null");
		}
		List<String> list = hash.get(argName);
		try {
			return list.get(argIndex);
		} catch (IndexOutOfBoundsException e) {
			throw new IndexOutOfBoundsException("IndexOutOfBoundsException " + argIndex);
		}
	}

	public int getMaximumLength(String argName) {
		List<String> list = hash.get(argName);
		int max = 0;
		for (int i = 0; i < list.size(); i++) {
			String value = list.get(i);
			if (max < value.length()) {
				max = value.length();
			}
		}
		return max;
	}

	// ****************************************************************************
	/**
	 * 指定キーの要素数を取得します。
	 * 
	 * @param argName
	 *            キー名
	 * @return 要素数
	 * @throws NullPointerException
	 *             キー名にNULLが含まれる場合
	 */
	// ****************************************************************************
	public int getDataCount(String argName) throws NullPointerException {
		if (argName == null) {
			throw new NullPointerException("argName must be not null");
		}
		List list = hash.get(argName);
		return list.size();
	}

	// ****************************************************************************
	/**
	 * 複数の指定キーの要素数を比較後に返却します。 例 <BR>
	 * �KEY {"K1","K2"} VALUE {{"1","2"},{"3","4"}} <BR>
	 * の状態で{"K1","K2"}を指定した場合は2が返る。 <BR>
	 * �KEY {"K1","K2","K3"} VALUE {{"1","2"},{"4"},{"5","5","3"}} <BR>
	 * の状態で{"K1","K2"}を指定した場合は-1が返る。 <BR>
	 * �KEY {"K1","K4","K7"} VALUE {{"1","2"},{"4"},{"5","5","3"}} <BR>
	 * の状態で{"K1","K2"}を指定した場合はNullPointerExceptionが返る。
	 * 
	 * @param argName
	 *            キー名
	 * @return -1 又は 指定キー要素数
	 * @throws NullPointerException
	 *             キー名にNULLが含まれる場合
	 */
	// ****************************************************************************
	public int getDataCount(String[] argName) throws NullPointerException {
		if (argName == null || argName.length <= 0) {
			return NG_VALUE;
		}
		int lastCnt = getDataCount(argName[0]);
		for (int i0 = 1; i0 < argName.length; i0++) {
			if (lastCnt != getDataCount(argName[i0])) {
				return NG_VALUE;
			}
		}
		return lastCnt;
	}

	// ****************************************************************************
	/**
	 * 指定キーの要素を全て削除します。
	 * 
	 * @param argName
	 *            キー名
	 */
	// ****************************************************************************
	public void initData(String argName) {
		List list = hash.get(argName);
		if (list == null) {
			return;
		}
		list.clear();
	}

	// ****************************************************************************
	/**
	 * 全てのキーの要素を全て削除します。
	 */
	// ****************************************************************************
	public void initData() {
		// ハッシュテーブルのキーに対応する値を全て取得する。
		Iterator<String> ite = hash.keySet().iterator();
		while (ite.hasNext()) {
			initData(ite.next()); // キー値を取得して全要素を削除
		}
	}

	// ****************************************************************************
	/**
	 * 内部で保持しているHashMap オブジェクトの文字列表現を返します。
	 * 
	 * @return 文字列表現
	 */
	// ****************************************************************************
	public String toString() {
		return hash.toString();
	}

	// ****************************************************************************
	/**
	 * 指定キーに指定された値がある場合に true を返します
	 * 
	 * @param argName
	 *            キー名
	 * @param argValue
	 *            値
	 * @return true/false
	 */
	// ****************************************************************************
	public boolean contains(String argName, String argValue) {
		List list = hash.get(argName);
		if (list == null) {
			return false;
		}
		return list.contains(argValue);
	}

	// ****************************************************************************
	/**
	 * equals メソッドを使って等しいかどうかを判定しながら、指定キーに対し <BR>
	 * 指定された値と同じ内容の要素を先頭から検索します。
	 * 
	 * @param argName
	 *            キー名
	 * @param argValue
	 *            値
	 * @return リスト内で引数が最初に現れるインデックス。 <BR>
	 *         オブジェクトが見つからない場合は -1
	 */
	// ****************************************************************************
	public int indexOf(String argName, String argValue) {
		List list = hash.get(argName);
		if (list == null) {
			return NG_VALUE;
		}
		return list.indexOf(argValue);
	}

	// ****************************************************************************
	/**
	 * 指定キーに対し要素がないかどうかを判定します
	 * 
	 * @param argName
	 *            キー名
	 * @return リストに要素がない場合は true、そうでない場合は false
	 */
	// ****************************************************************************
	public boolean isEmpty(String argName) {
		List list = hash.get(argName);
		if (list == null) {
			return false;
		}
		return list.isEmpty();
	}

	// ****************************************************************************
	/**
	 * 指定キーのリスト内の指定された位置から要素を削除します。 <BR>
	 * そして、後続の要素を左側に移動し、 <BR>
	 * それぞれのインデックスから 1 を減算します。
	 * 
	 * @param argName
	 *            キー名
	 * @param argIndex
	 *            インデックス
	 * @return リストから削除した文字列
	 * @throws IndexOutOfBoundsException
	 *             不正なインデックス指定の場合
	 * @throws NullPointerException
	 *             キー名にNULLが含まれる場合
	 */
	// ****************************************************************************
	public String remove(String argName, int argIndex)
			throws IndexOutOfBoundsException, NullPointerException {
		if (argName == null) {
			throw new NullPointerException("argName must be not null");
		}
		try {
			return hash.get(argName).remove(argIndex);
		} catch (IndexOutOfBoundsException e) {
			throw new IndexOutOfBoundsException("IndexOutOfBoundsException " + argIndex);
		}
	}

	// ****************************************************************************
	/**
	 * 指定キーの指定されたオブジェクトがリスト内で最後に現れるインデックスを返します。
	 * 
	 * @param argName
	 *            キー名
	 * @param argValue
	 *            値
	 * @return 指定キーのリストで指定のオブジェクトと一致する最後の 文字列のインデックス。 <BR>
	 *         指定キーの指定オブジェクトが見つからない場合は -1
	 */
	// ****************************************************************************
	public int lastIndexOf(String argName, String argValue) {
		List list = hash.get(argName);
		if (list == null) {
			return NG_VALUE;
		}
		return list.lastIndexOf(argValue);
	}

	// ****************************************************************************
	/**
	 * キーの中身をはかいしないこと。！ そのうち破壊できないように修正する。。。
	 * 
	 * @return キー
	 */
	// ****************************************************************************
	public String[] getKey() {
		return key;
	}
}