/*
  win32_win.cpp
  base class of screen driver

  Made by Studio Breeze. 2002

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
 */

#include "sted_win32.h"
#include "sted_screen_win32.h"

/* accessor */

CSTedScreenBase*
CreateScreenDriver(int in_size)
{
  return (CSTedScreenBase *)CSTedScreenWin32::New(in_size);
}

void
DestroyScreenDriver(CSTedScreenBase* in_driver)
{
  CSTedScreenWin32* self = (CSTedScreenWin32 *)in_driver;
  self->Tidy();
  delete self;
}

/* class */

CSTedScreenWin32::CSTedScreenWin32()
{
  fWindow = NULL;
  fMenu = NULL;
  fWindowDC = NULL;

  fText = NULL;
  fTextMask = NULL;
  fGraphicsWindow[0] = NULL;
  fGraphicsWindow[1] = NULL;
  fOffScreen = NULL;
  fOffScreenIsDirty = FALSE;
  fCanUpdateWindow = TRUE;

  fTextDC = NULL;
  fTextMaskDC = NULL;
  fGraphicsWindowDC[0] = NULL;
  fGraphicsWindowDC[1] = NULL;
  fOffScreenDC = NULL;

  fCurrentTextX = 0;
  fCurrentTextY = 0;
  fTextOriginY = 0;
  fTextHeight = fX68TextHeight;
  fIsCursorOn = FALSE;
  fCurrentGraphics = 0;

  fWindowWidth = 768;
  fWindowHeight = 512;
  fTextLineHeight = 12;

  fUCS2Buf = NULL;
  fSJISBuf = NULL;
  const int fUCS2BufLen = 1024;
  const int fSJISBufLen = 1024;

  fCurrentTextColor = 0;
  fCurrentTextBackColor = 0;
  fCurrentGraphicsColor = 0;

  fBrushMaskOff = NULL;
  fBrushMaskOn = NULL;

  fFileDirHandle = NULL;

  fMidiInDevice = NULL;
  fMidiInPort = 0;
  for (int i=0; i<fMidiOutPorts; i++) {
	fMidiOutDevice[i] = NULL;
	fMidiOutPort[i] = 0; // default device
  }
  fMidiOutPacket[0] = -1;
  fMidiOutPacket[1] = -1;
  fMidiOutPacket[2] = -1;
  fMidiOutPacket[3] = -1;

  fTimerEventNotified = FALSE;

  fConstructed = FALSE;
}

void
CSTedScreenWin32::Construct(int in_size)
{
  int fw, fh;
  int w,h;
  int i;
  HFONT font;
  LOGFONT lf;

  ::ZeroMemory((void *)&lf, sizeof(LOGFONT));
  lf.lfHeight = in_size;
  lf.lfWidth = 0;
  lf.lfWeight = 0;
  lf.lfItalic = 0;
  lf.lfUnderline = 0;
  lf.lfCharSet = SHIFTJIS_CHARSET;
  lf.lfPitchAndFamily = FIXED_PITCH | FF_DONTCARE;

  font = ::CreateFontIndirect(&lf);

  if (in_size<0) return;
  fh = in_size;
  fw = in_size/2;
  w = fw * fX68TextWidth;
  h = fh * fX68TextHeight;

  // main window resource
  fWindow = STedGetWindow();
  fMenu = STedGetMenu();
  fWindowDC = STedGetWindowDC();
  if (!fWindow || !fMenu || !fWindowDC) return;

  // off-screen bitmaps
  fText = ::CreateCompatibleBitmap(fWindowDC, w, h);
  if (!fText) return;
  fTextMask = ::CreateBitmap(w, h, 1, 1, NULL);
  if (!fTextMask) return;
  fGraphicsWindow[0] = ::CreateCompatibleBitmap(fWindowDC, w, h);
  if (!fGraphicsWindow[0]) return;
  fGraphicsWindow[1] = ::CreateCompatibleBitmap(fWindowDC, w, h);
  if (!fGraphicsWindow[1]) return;
  fOffScreen = ::CreateCompatibleBitmap(fWindowDC, w, h);
  if (!fOffScreen) return;

  // device contexts
  fTextDC = ::CreateCompatibleDC(fWindowDC);
  if (!fTextDC) return;
  ::SelectObject(fTextDC, fText);
  fTextMaskDC = ::CreateCompatibleDC(fWindowDC);
  if (!fTextMaskDC) return;
  ::SelectObject(fTextMaskDC, fTextMask);
  fGraphicsWindowDC[0] = ::CreateCompatibleDC(fWindowDC);
  if (!fGraphicsWindowDC[0]) return;
  ::SelectObject(fGraphicsWindowDC[0], fGraphicsWindow[0]);
  fGraphicsWindowDC[1] = ::CreateCompatibleDC(fWindowDC);
  if (!fGraphicsWindowDC[1]) return;
  ::SelectObject(fGraphicsWindowDC[1], fGraphicsWindow[1]);
  fOffScreenDC = ::CreateCompatibleDC(fWindowDC);
  if (!fOffScreenDC) return;
  ::SelectObject(fOffScreenDC, fOffScreen);

  ::SelectObject(fTextDC, font);
  ::SelectObject(fTextMaskDC, font);
  ::SelectObject(fGraphicsWindowDC[0], font);
  ::SelectObject(fGraphicsWindowDC[1], font);
  ::SelectObject(fOffScreenDC, font);
  ::SetBkMode(fTextDC, TRANSPARENT);
  ::SetBkMode(fTextMaskDC, TRANSPARENT);
  ::SetBkMode(fGraphicsWindowDC[0], TRANSPARENT);
  ::SetBkMode(fGraphicsWindowDC[1], TRANSPARENT);
  ::SetBkMode(fOffScreenDC, TRANSPARENT);
  ::SetTextColor(fTextDC, RGB(0xff, 0xff, 0xff));
  ::SetTextColor(fTextMaskDC, RGB(0x00, 0x00, 0x00));
  
  fWindowWidth = w;
  fWindowHeight = h;
  fTextLineHeight = fh;

  fUCS2Buf = (LPTSTR)malloc(sizeof(WCHAR)*fUCS2BufLen);
  if (!fUCS2Buf) return;
  fSJISBuf = (char *)malloc(sizeof(char)*fSJISBufLen);
  if (!fSJISBuf) return;

  fBrushMaskOn = ::CreateSolidBrush(RGB(0xff,0xff,0xff));
  if (!fBrushMaskOn) return;
  fBrushMaskOff = ::CreateSolidBrush(RGB(0x00,0x00,0x00));
  if (!fBrushMaskOff) return;
  fBrushClear = ::CreateSolidBrush(RGB(0x00,0x00,0x00));
  if (!fBrushClear) return;
  fBrushTextClear = ::CreateSolidBrush(RGB(0xff,0xff,0xff));
  if (!fBrushTextClear) return;

  for (i=0; i<3; i++) {
	  SetTextPalette(i, 0);
  }

  h+=::GetSystemMetrics(SM_CYCAPTION);
  h+=::GetSystemMetrics(SM_CYMENU);
  h+=(::GetSystemMetrics(SM_CYSIZEFRAME)*2);

  ::SetWindowPos(fWindow, HWND_TOP, 150, 150, w, h,
		 SWP_DRAWFRAME | SWP_NOMOVE | SWP_NOZORDER);
  
  fConstructed = TRUE;
}

CSTedScreenWin32*
CSTedScreenWin32::New(int in_screensize)
{
  CSTedScreenWin32* self;
  self = new CSTedScreenWin32;
  if (!self) return NULL;

  self->Construct(in_screensize);
  if (!self->fConstructed) {
    self->Tidy();
    delete self;
    return NULL;
  }
  return self;
}

CSTedScreenWin32::~CSTedScreenWin32()
{
}

void
CSTedScreenWin32::Tidy(void)
{
  if (fTextDC) ::DeleteDC(fTextDC);
  fTextDC = NULL;
  if (fTextMaskDC) ::DeleteDC(fTextMaskDC);
  fTextMaskDC = NULL;
  if (fGraphicsWindowDC[0]) ::DeleteDC(fGraphicsWindowDC[0]);
  fGraphicsWindowDC[0] = NULL;
  if (fGraphicsWindowDC[1]) ::DeleteDC(fGraphicsWindowDC[1]);
  fGraphicsWindowDC[2] = NULL;
  if (fOffScreenDC) ::DeleteDC(fOffScreenDC);
  fOffScreenDC = NULL;

  if (fText) ::DeleteObject(fText);
  fText = NULL;
  if (fTextMask) ::DeleteObject(fTextMask);
  fTextMask = NULL;
  if (fGraphicsWindow[0]) ::DeleteObject(fGraphicsWindow[0]);
  fGraphicsWindow[0] = NULL;
  if (fGraphicsWindow[1]) ::DeleteObject(fGraphicsWindow[1]);
  fGraphicsWindow[1] = NULL;
  if (fOffScreen) ::DeleteObject(fOffScreen);
  fOffScreen = NULL;

  if (fUCS2Buf) free((void *)fUCS2Buf);
  fUCS2Buf = NULL;
  if (fSJISBuf) free((void *)fSJISBuf);
  fSJISBuf = NULL;

  if (fBrushMaskOn) ::DeleteObject(fBrushMaskOn);
  fBrushMaskOn = NULL;
  if (fBrushMaskOff) ::DeleteObject(fBrushMaskOff);
  fBrushMaskOff = NULL;
  if (fBrushClear) ::DeleteObject(fBrushClear);
  fBrushClear = NULL;
  if (fBrushTextClear) ::DeleteObject(fBrushTextClear);
  fBrushTextClear = NULL;

  if (fFileDirHandle) {
	  ::FindClose(fFileDirHandle);
	  fFileDirHandle = NULL;
  }

  ::SetRectEmpty(&fInvalidRect);

  fCanUpdateWindow = TRUE;

  fTimerEventNotified = FALSE;

  fConstructed = FALSE;
}

// window
void
CSTedScreenWin32::InitWindow(void)
{
  RECT r;

  ::SetRect(&r, 0, 0, fWindowWidth, fWindowHeight);

  ::FillRect(fTextDC, &r, fBrushTextClear);
  ::FillRect(fTextMaskDC, &r, MASK_OFF);
  ::FillRect(fGraphicsWindowDC[0], &r, fBrushClear);
  ::FillRect(fGraphicsWindowDC[1], &r, fBrushClear);
  ::FillRect(fOffScreenDC, &r, fBrushClear);

  ::SetTextColor(fTextDC, RGB(0xff, 0xff, 0xff));
  ::SetTextColor(fTextMaskDC, RGB(0xff, 0xff, 0xff));
  ::SetBkColor(fTextDC, RGB(0x00, 0x00, 0x00));
  ::SetBkColor(fTextMaskDC, RGB(0xff, 0xff, 0xff));
  ::SetBkMode(fTextDC, OPAQUE);
  ::SetBkMode(fTextMaskDC, OPAQUE);

  ::SetTextColor(fGraphicsWindowDC[0], RGB(0xff, 0xff, 0xff));
  ::SetTextColor(fGraphicsWindowDC[1], RGB(0xff, 0xff, 0xff));
  ::SetBkColor(fGraphicsWindowDC[0], RGB(0x00, 0x00, 0x00));
  ::SetBkColor(fGraphicsWindowDC[1], RGB(0x00, 0x00, 0x00));
  ::SetBkMode(fGraphicsWindowDC[0], OPAQUE);
  ::SetBkMode(fGraphicsWindowDC[1], OPAQUE);

  ::SetTextColor(fOffScreenDC, RGB(0x00, 0x00, 0x00));
  ::SetBkColor(fOffScreenDC, RGB(0xff, 0xff, 0xff));
  ::SetBkMode(fOffScreenDC, OPAQUE);

  ::SetRectEmpty(&fInvalidRect);

  ::memset((void *)fUCS2Buf, 0, sizeof(WCHAR)*fUCS2BufLen);
  ::memset((void *)fSJISBuf, 0, sizeof(char)*fSJISBufLen);

  fCurrentTextX = 0;
  fCurrentTextY = 0;
  fIsCursorOn = FALSE;
  fCurrentGraphics = 0;
  fCurrentTextColor = 3;
  fCurrentTextBackColor = 0;
  fCurrentGraphicsColor = 0;

  InvalidateRect(&r);
}

void
CSTedScreenWin32::CloseWindow(void)
{
}

void
CSTedScreenWin32::RepaintWindow(void)
{
  InvalidateRect(NULL);
}

void
CSTedScreenWin32::UpdateWindow(void)
{
	fCanUpdateWindow = TRUE;
	::UpdateWindow(fWindow);
}

void
CSTedScreenWin32::DisableUpdateWindow(void)
{
	fCanUpdateWindow = FALSE;
}


// private
void
CSTedScreenWin32::InvalidateRect(const RECT *in_rect)
{
  RECT r;
  int w,h;

  if (::IsRectEmpty(&fInvalidRect)) {
	  ::CopyRect(&r, in_rect);
  } else {
	  ::UnionRect(&r, &fInvalidRect, in_rect);
  }
  w = r.right - r.left;
  h = r.bottom - r.top;
  if (fCanUpdateWindow && (w*h > fWindowHeight * fWindowWidth / 8)) {
	  ::UpdateWindow(fWindow);
  } else {
	  ::CopyRect(&fInvalidRect, &r);
  }

  ::InvalidateRect(fWindow, in_rect, FALSE);
  fOffScreenIsDirty = TRUE;
}

void
CSTedScreenWin32::InvalidateRect(const int in_x, const int in_y, const int in_w, const int in_h)
{
  RECT r;
  ::SetRect(&r, in_x, in_y, in_x+in_w, in_y+in_h);
  InvalidateRect(&r);
}

// public
void
CSTedScreenWin32::Paint(RECT *in_rect)
{
  int x,y;
  int w,h;

  x = in_rect->left;
  y = in_rect->top;
  w = in_rect->right-x;
  h = in_rect->bottom-y;

  if (fOffScreenIsDirty) {
	::BitBlt(fOffScreenDC, x,y,w,h, fGraphicsWindowDC[fCurrentGraphics], x,y, SRCCOPY);
	::BitBlt(fOffScreenDC, x,y,w,h, fTextMaskDC, x,y, SRCPAINT);
	::BitBlt(fOffScreenDC, x,y,w,h, fTextDC, x,y, SRCAND);

	fOffScreenIsDirty = FALSE;
	::SetRectEmpty(&fInvalidRect);
  }

  ::BitBlt(fWindowDC, x,y,w,h, fOffScreenDC, x,y, SRCCOPY);

  fCanUpdateWindow = TRUE;
}

LPCTSTR
CSTedScreenWin32::toUCS2(const char* in_str, int *out_len)
{
	return toUCS2(in_str, (int)::strlen(in_str), out_len);
}

LPCTSTR 
CSTedScreenWin32::toUCS2(const char* in_str, int in_len, int *out_len)
{
#ifdef _UNICODE
	int bytes;
	bytes = ::MultiByteToWideChar(CP_ACP, 0, in_str, in_len, fUCS2Buf, fUCS2BufLen);
	if (out_len) *out_len = bytes;
	return fUCS2Buf;
#else
	int bytes = in_len;
	if (fUCS2BufLen<=in_len) bytes = fUCS2BufLen-1;
	::memcpy((void *)fUCS2Buf, in_str, bytes);
	((char *)fUCS2Buf)[bytes] = '\0';
	if (out_len) *out_len = bytes;
	return fUCS2Buf;
#endif
}
  
const char*
CSTedScreenWin32::toSJIS(LPCTSTR in_str, int *out_len)
{
#ifdef _UNICODE
	int bytes;
	bytes = ::WideCharToMultiByte(CP_ACP, 0, in_str, -1, fSJISBuf, fSJISBufLen, NULL, NULL);
	if (out_len) *out_len = bytes;
	return fSJISBuf;
#else
	int bytes;
	bytes = (int)::strlen(in_str);
	if (bytes>=fSJISBufLen) bytes = fSJISBufLen-1;
	::memcpy((void *)fSJISBuf, in_str, bytes);
	fSJISBuf[bytes] = '\0';
	if (out_len) *out_len = bytes;
	return fSJISBuf;
#endif
}
