/*
  RCP player engine

  Copyright 1999 by Daisuke Nagano <breeze.nagano@nifty.ne.jp>
  Feb.05.1999
  Oct.16.2002


  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/

/* ------------------------------------------------------------------- */

#ifdef HAVE_CONFIG_H
#include "../../config.h"
#endif /* HAVE_CONFIG_H */

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>


#include "rcp.h"
#include "smf.h"
#include "rcp_functions.h"
#include "mididev.h"
#include "gettext_wrapper.h"
#include "version.h"

#ifndef timercmp
# define       timercmp(tvp, uvp, cmp)\
               ((tvp)->tv_sec cmp (uvp)->tv_sec ||\
               (tvp)->tv_sec == (uvp)->tv_sec &&\
               (tvp)->tv_usec cmp (uvp)->tv_usec)
#endif

/* ------------------------------------------------------------------- */

#include "rcddef.h"
extern struct RCD_HEAD *rcd;
static int is_sted2_support = FLAG_TRUE;
void (*sted2_md_put)( char );

/* ------------------------------------------------------------------- */
/* player's functions */

int reset_mode  = 2;

static int set_signals( void );
static int put_event( RCP_DATA *, int );
static int priority_init( void );

static int sted2_init( RCP_DATA * );
static int sted2_close( void );

extern int rcptomid_init_track_buffer( RCP_DATA * );
extern int rcptomid_read_rcp_header( RCP_DATA * );
extern int rcptomid_set_new_event( RCP_DATA *, int );

static void start_timer(int in_interval, RCP_DATA* in_rcp);
static void stop_timer(void);

static int is_player_alive;

inline long timerdiff( struct timeval *st, struct timeval *et ) {
  long reth, retl;
  retl = st->tv_usec - et->tv_usec;
  reth = st->tv_sec - et->tv_sec;
  if ( retl < 0 ) {
    reth--;
    retl+=(1000*1000);
  }
  return (reth*1000*1000)+retl;
}

static int
gettimeofday(struct timeval* st, void *dummy)
{
	DWORD time = timeGetTime();
	st->tv_usec = (time%1000)*1000;
	st->tv_sec = time/1000;

	return 0;
}

/* ------------------------------------------------------------------- */
/* accessor */

int
rcpplay_start(RCP_DATA* in_rcp)
{
	int trk;
	struct timeval st;

	if ( rcptomid_init_track_buffer( in_rcp ) != 0 ) return 1;
	if ( rcptomid_read_rcp_header( in_rcp )   != 0 ) return 1;

	/* prepare for STed2 support */
	sted2_init( in_rcp );

	/* start playing */

	sayleen_send_midi_reset();
	sayleen_myusleep( 50 * 1000 );

	rcd->stepcount = 0;
	rcd->act = 1;
	rcd->sts = 0;
	for ( trk=0 ; trk < TRK_NUM ; trk++ ) {
		rcd->step[trk]=-1;
		rcd->bar[trk]=-1;
	}

	if ( in_rcp->is_send_rtm == FLAG_TRUE ) {
		sayleen_send_rtm_start();
	}
	in_rcp->rtm_delta = in_rcp->timebase / 24;
	is_player_alive = FLAG_TRUE;

	in_rcp->rcd = (void *)rcd;

	gettimeofday(&st, NULL);
	in_rcp->timer_sec = st.tv_sec;
	in_rcp->timer_usec = st.tv_usec;

	start_timer(50*1000, in_rcp);

	return 0;
}

int
rcpplay_stop(void)
{
	stop_timer();
	sayleen_send_rtm_stop();
	sayleen_send_midi_reset();
	return 0;
}

int
rcpplay_close(void)
{
	sted2_close();
	return 0;
}


/* RCP player engine */
int
rcpplay( RCP_DATA *rcp )
{
	struct timeval st, et;
	long s;
	int trk;
	struct RCD_HEAD *rcd;
	int is_all_tracks_finished;

	if (!rcp) return -1;

	rcd = (struct RCD_HEAD *)rcp->rcd;

	st.tv_sec = rcp->timer_sec;
	st.tv_usec = rcp->timer_usec;

	while(1) {
		int track;

		is_all_tracks_finished = FLAG_TRUE;

		rcp->step++;

		rcd->stepcount++;
		for ( trk=0 ; trk < TRK_NUM ; trk++ ) {
			rcd->step[trk]=-1;
			rcd->bar[trk]=-1;
		}

		for ( track=0 ; track < rcp->tracks ; track++ ) {
			int n;

			/* Have the track finished ? */
			if ( rcp->track[track].finished == FLAG_TRUE &&
				rcp->track[track].all_notes_expired == FLAG_TRUE )
				continue;

			is_all_tracks_finished = FLAG_FALSE;

			/* checks each notes whether the gate time is expired */
			if ( rcp->track[track].all_notes_expired == FLAG_FALSE ) {
				for ( n = rcp->track[track].notes_min ;
					n <=rcp->track[track].notes_max ; n++ ) {
					int gate = rcp->track[track].notes[n];
					gate--;
					if ( gate == 0 ) {
						rcp_note_off( rcp, track, n );
						put_event( rcp, track );
					}
					if ( gate <  0 ) gate = 0;
					rcp->track[track].notes[n] = gate;
					}
			}

			/* checks whether the step time is expired */
			rcp->track[track].delta_step++;
			rcp->track[track].total_step++;
			rcp->track[track].step--;
			while ( rcp->track[track].step == 0 ) {
				if ( rcp->track[track].finished == FLAG_TRUE ) break;
				rcptomid_set_new_event( rcp, track );
				put_event( rcp, track );
			}
		}
		if ( is_all_tracks_finished == FLAG_TRUE ) break;


		/* wait a delta time */

		st.tv_usec += 1000 * 1000 * 60 / (rcp->realtempo * rcp->timebase);
		while ( st.tv_usec >= 1000*1000 ) {
			st.tv_usec-=1000*1000;
			st.tv_sec++;
		}

		rcp->rtm_delta--;
		if ( rcp->rtm_delta <= 0 ) {
			if ( rcp->is_send_rtm == FLAG_TRUE ) {
				sayleen_send_rtm_timingclock();
			}
			rcp->rtm_delta = rcp->timebase / 24;
		}

		sayleen_flush_midi();
		gettimeofday( &et, NULL );
		s = timerdiff( &st, &et );
		if ( s > 1*1000 ) {
			rcp->timer_sec = st.tv_sec;
			rcp->timer_usec = st.tv_usec;
			start_timer(s, rcp);
			return 0;
		}

		if ( et.tv_sec - st.tv_sec > 1 ) {
			st.tv_sec  = et.tv_sec;
			st.tv_usec = et.tv_usec;
		}
	}


	if ( rcp->is_send_rtm == FLAG_TRUE && is_all_tracks_finished == FLAG_TRUE) {
		sayleen_send_rtm_stop();
		rcd->act = 0;
		sted2_close();
	}

	return 0;
}

static void CALLBACK
timerproc(UINT in_id, UINT in_msg, DWORD in_dwUser, DWORD in_dw1, DWORD in_dw2)
{
	RCP_DATA* rcp = (RCP_DATA *)in_dwUser;
	rcpplay(rcp);
}

static UINT timer_id = 0;

static void
start_timer(int in_interval, RCP_DATA* in_rcp)
{
	timer_id = timeSetEvent(in_interval/1000, 0, timerproc, (DWORD)in_rcp, TIME_ONESHOT);
}

static void
stop_timer(void)
{
	if (timer_id>0) {
		timeKillEvent(timer_id);
		timer_id = 0;
	}
}

/* ------------------------------------------------------------------- */

#if 0
RETSIGTYPE sigexit( int num ) {
  sted2_close();

  sayleen_send_rtm_stop();
  send_midi_reset();
  close_midi_device();

  fprintf(stderr,"Signal caught : %d\n", num);
  exit(1);
}
#endif

/* ------------------------------------------------------------------- */
static int sted2_init ( RCP_DATA *rcp ) {

  sted2_md_put = NULL;

  is_sted2_support = FLAG_TRUE;
  /*sted2_md_put = rcd->md_put;*/

  return FLAG_TRUE;
}

static int sted2_close( void ) {

  return 0;
}

/* ------------------------------------------------------------------- */
static int put_event( RCP_DATA *rcp, int track ) {

  int i;
  static int current_port = -1;

  if ( rcp->result_smf[0] == SMF_TERM ) return 0;

  /* is the track disabled ? */
  if ( rcp->track[track].enabled == FLAG_FALSE && 
       rcp->result_smf[0] != MIDI_NOTEOFF ) {
    goto put_event_end;
  }
  
  if ( rcp->result_smf[0] == MIDI_META ) {
    switch( rcp->result_smf[1] ) {
    case META_PORT:
      break;

    case META_TEXT:
    case META_SEQNUM:
    case META_COPYRIGHT:
    case META_SEQNAME:
    case META_INSTNAME:
    case META_LYRIC:
      if ( rcp->enable_verbose == FLAG_TRUE ) {
	i=2;
	while( rcp->result_smf[i++] >= 0x80 ) {};
	while( rcp->result_smf[i] != SMF_TERM ) {
	  fprintf(stderr,"%c",rcp->result_smf[i++]);
	}
	fprintf(stderr,"\n");
      }
      break;

    default:
      break;
    }
    goto put_event_end;
  }

  /* flushing MIDI data */
  
  if ( rcp->track[track].delta_step < 0 ) {
    fprintf(stderr,"%d\n",(int)rcp->track[track].delta_step);
  }

  if ( current_port != rcp->track[track].port ) {
    sayleen_change_midi_port( rcp->track[track].port );
    current_port = rcp->track[track].port;
  }

  i=0;
  if ( rcp->result_smf[0] == 0xf0 || rcp->result_smf[0] == 0xf7 ) {
    sayleen_put_midi(rcp->result_smf[0]);
    i=2;
  }
  while ( rcp->result_smf[i] != SMF_TERM ) {
    sayleen_put_midi(rcp->result_smf[i]);
    i++;
  }

  rcp->track[track].delta_step = 0;
  
put_event_end:
  rcp->result_smf[0] = SMF_TERM;

  return 0;
}
