/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail.javamail;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import javax.activation.FileDataSource;
import javax.activation.URLDataSource;

import com.wankuma.commons.lang.Assertion;
import com.wankuma.mail.Header;
import com.wankuma.mail.MailBody;
import com.wankuma.mail.MailBodyFactory;
import com.wankuma.mail.helper.SimpleHeader;

/**
 * 日本語処理に対応された{@link MailBodyFactory}の実装です。
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */
public class JapaneseMailBodyFactory implements MailBodyFactory {

	/**
	 * @see com.wankuma.mail.MailBodyFactory#createBinaryMailBody(java.io.File)
	 */
	@Override
	public MailBody createBinaryMailBody(File file) {
		Assertion.notNull("file", file);

		final FileDataSource dataSource = new FileDataSource(file);
		return new MailBody() {
			@Override
			public String getContentType() {
				return dataSource.getContentType();
			}

			@Override
			public Header[] getHeaders() {
				return new Header[0];
			}

			@Override
			public InputStream getInputStream() throws IOException {
				return dataSource.getInputStream();
			}

			@Override
			public String getMessage() {
				return "";
			}
		};
	}

	/**
	 * @see com.wankuma.mail.MailBodyFactory#createBinaryMailBody(java.io.InputStream,
	 *      java.lang.String)
	 */
	@Override
	public MailBody createBinaryMailBody(final InputStream in,
			final String contentType) {
		Assertion.notNull("in", in);

		return new MailBody() {
			@Override
			public String getContentType() {
				return contentType;
			}

			@Override
			public Header[] getHeaders() {
				return new Header[0];
			}

			@Override
			public InputStream getInputStream() throws IOException {
				return in;
			}

			@Override
			public String getMessage() {
				return "";
			}
		};
	}

	/**
	 * @see com.wankuma.mail.MailBodyFactory#createBinaryMailBody(java.net.URL)
	 */
	@Override
	public MailBody createBinaryMailBody(URL url) {
		Assertion.notNull("url", url);

		final URLDataSource dataSource = new URLDataSource(url);
		return new MailBody() {
			@Override
			public String getContentType() {
				return dataSource.getContentType();
			}

			@Override
			public Header[] getHeaders() {
				return new Header[0];
			}

			@Override
			public InputStream getInputStream() throws IOException {
				return dataSource.getInputStream();
			}

			@Override
			public String getMessage() {
				return "";
			}
		};
	}

	/**
	 * @see com.wankuma.mail.MailBodyFactory#createHTMLMailBody(java.lang.String)
	 */
	@Override
	public MailBody createHTMLMailBody(String html) {
		return this.createHTMLMailBody(html, null);
	}

	/**
	 * @see com.wankuma.mail.MailBodyFactory#createHTMLMailBody(java.lang.String,
	 *      java.lang.String)
	 */
	@Override
	public MailBody createHTMLMailBody(final String html, final String charset) {
		Assertion.notNull("html", html);

		return new MailBody() {
			@Override
			public String getContentType() {
				return "text/html; charset=ISO-2022-JP";
			}

			@Override
			public Header[] getHeaders() {
				return new Header[] { new SimpleHeader(
						"Content-Transfer-Encoding", "7bit") };
			}

			@Override
			public InputStream getInputStream() throws IOException {
				return EncodeUtils.getContentStream(html);
			}

			@Override
			public String getMessage() {
				return html;
			}
		};
	}

	/**
	 * @see com.wankuma.mail.MailBodyFactory#createPlainTextMailBody(java.lang.String)
	 */
	@Override
	public MailBody createPlainTextMailBody(final String content) {
		Assertion.notNull("content", content);

		return new MailBody() {
			@Override
			public String getContentType() {
				return "text/plain; charset=ISO-2022-JP";
			}

			@Override
			public Header[] getHeaders() {
				return new Header[] { new SimpleHeader(
						"Content-Transfer-Encoding", "7bit") };
			}

			@Override
			public InputStream getInputStream() throws IOException {
				return EncodeUtils.getContentStream(content);
			}

			@Override
			public String getMessage() {
				return content;
			}
		};
	}
}
