/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail.javamail;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;

import com.wankuma.commons.io.StreamUtils;
import com.wankuma.commons.io.WritableInputStream;
import com.wankuma.commons.util.Base64;

/**
 * メールのエンコードをサポートするユーティリティです。<br>
 * <dl>
 * <dt>謝辞</dt>
 * <dd> このクラスは<a href="http://www.sk-jp.com/java/library/index.html">Shin
 * Kinoshita's Home</a>にて提供されている、 <a
 * href="http://www.sk-jp.com/java/library/skutility/index.html">Utilities</a>及び、<a
 * href="http://www.sk-jp.com/java/library/skmail/index.html">Message Access
 * Utilities</a>から、<br>
 * 一部のコードを利用させていただいております。 Shin Kinoshita氏には深く感謝いたします。</dd>
 * </dl>
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */
final class EncodeUtils {

	private static final byte[] SJIS_KANA_DATA = new byte[] { -127, 66, -127,
			117, -127, 118, -127, 65, -127, 69, -125, -110, -125, 64, -125, 66,
			-125, 68, -125, 70, -125, 72, -125, -125, -125, -123, -125, -121,
			-125, 98, -127, 91, -125, 65, -125, 67, -125, 69, -125, 71, -125,
			73, -125, 74, -125, 76, -125, 78, -125, 80, -125, 82, -125, 84,
			-125, 86, -125, 88, -125, 90, -125, 92, -125, 94, -125, 96, -125,
			99, -125, 101, -125, 103, -125, 105, -125, 106, -125, 107, -125,
			108, -125, 109, -125, 110, -125, 113, -125, 116, -125, 119, -125,
			122, -125, 125, -125, 126, -125, -128, -125, -127, -125, -126,
			-125, -124, -125, -122, -125, -120, -125, -119, -125, -118, -125,
			-117, -125, -116, -125, -115, -125, -113, -125, -109, -127, 74,
			-127, 75, };

	/**
	 * 指定の文字列をエンコードして返します。
	 * 
	 * @param str
	 *            文字列
	 * @return エンコードされた結果の文字列
	 * 
	 * @throws UnsupportedEncodingException
	 *             エンコード処理に失敗した場合
	 */
	static String encode(String str) throws UnsupportedEncodingException {
		if (str == null) {
			return null;
		}

		InputStream in = EncodeUtils.getContentStream(str);
		byte[] bytes;
		try {
			bytes = StreamUtils.getBytes(in);
		} catch (IOException e) {
			return null;
		}

		StringBuilder builder = new StringBuilder();
		builder.append("=?ISO-2022-JP?B?");
		builder.append(Base64.encode(bytes));
		builder.append("?=");

		return builder.toString();
	}

	/**
	 * 指定のメッセージ本文をエンコードした結果のストリームを返します。
	 * 
	 * @param content
	 *            文字列
	 * @return データソース
	 * 
	 * @throws UnsupportedEncodingException
	 *             エンコードに失敗した場合
	 */
	static InputStream getContentStream(final String content)
			throws UnsupportedEncodingException {
		return new WritableInputStream() {
			@Override
			protected void write(OutputStream out) throws IOException {
				Charset charset = Charset.forName("Windows-31J");
				CharBuffer charBuffer = CharBuffer.wrap(content);
				int limit = charBuffer.limit();
				for (int i = 0; i < limit; i++) {
					switch (charBuffer.get(i)) {
					case 0x301c:
						charBuffer.put(i, (char) 0xff5e);
						break;
					case 0x2016:
						charBuffer.put(i, (char) 0x2225);
						break;
					case 0x2212:
						charBuffer.put(i, (char) 0xff0d);
						break;
					default:
						break;
					}
				}

				ByteBuffer buffer = charset.newEncoder().encode(charBuffer);
				boolean nonAscii = false;
				limit = buffer.limit();

				for (int i = 0; i < limit; i++) {
					byte b = buffer.get();
					if (b >= 0) {
						if (nonAscii) {
							nonAscii = false;
							out.write(0x1b);
							out.write('(');
							out.write('B');
						}
						out.write(b);
					} else {
						if (!nonAscii) {
							nonAscii = true;
							out.write(0x1b);
							out.write('$');
							out.write('B');
						}
						int b1 = b & 0xff;
						if (b1 >= 0xa1 && b1 <= 0xdf) {
							int kanaIndex = (b1 - 0xA1) * 2;
							EncodeUtils.getContentStream0(out,
									EncodeUtils.SJIS_KANA_DATA[kanaIndex],
									EncodeUtils.SJIS_KANA_DATA[kanaIndex + 1]);
						} else {
							i++;
							if (i == limit) {
								break;
							}
							buffer.position(buffer.position());
							EncodeUtils.getContentStream0(out, b, buffer.get());
						}
					}
				}

				if (nonAscii) {
					out.write(0x1b);
					out.write('(');
					out.write('B');
				}
			}
		};
	}

	private static void getContentStream0(OutputStream out, byte bh, byte bl)
			throws IOException {
		int h = bh << 1 & 0xff;
		int l = bl & 0xff;

		if (l < 0x9f) {
			if (h < 0x3f) {
				h += 0x1f;
			} else {
				h -= 0x61;
			}
			if (l > 0x7e) {
				l -= 0x20;
			} else {
				l -= 0x1f;
			}
		} else {
			if (h < 0x3f) {
				h += 0x20;
			} else {
				h -= 0x60;
			}
			l -= 0x7e;
		}

		out.write(h);
		out.write(l);
	}

	private EncodeUtils() {
	}
}
