/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail.helper;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import com.wankuma.commons.lang.Assertion;
import com.wankuma.mail.AttachmentFile;

/**
 * シンプルな{@link AttachmentFile}の実装です。
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */
public class SimpleAttachmentFile implements AttachmentFile {

	private AttachmentFile file;

	/**
	 * 指定のファイルでインスタンスを構築します。
	 * 
	 * @param file
	 *            ファイル
	 * 
	 * @throws NullPointerException
	 *             fileが{@code null}の場合
	 */
	public SimpleAttachmentFile(final File file) {
		Assertion.notNull("file", file);

		this.file = new AttachmentFile() {

			@Override
			public String getFileName() {
				return file.getName();
			}

			@Override
			public InputStream getInputStream() throws IOException {
				return new FileInputStream(file);
			}
		};
	}

	/**
	 * 指定の入力ストリームとファイル名でインスタンスを構築します。
	 * 
	 * @param fileName
	 *            ファイル名
	 * @param in
	 *            入力ストリーム
	 * @throws NullPointerException
	 *             fileNameもしくはinが{@code null}の場合
	 */
	public SimpleAttachmentFile(final String fileName, final InputStream in) {
		Assertion.notNull("fileName", fileName);
		Assertion.notNull("in", in);

		this.file = new AttachmentFile() {

			@Override
			public String getFileName() {
				return fileName;
			}

			@Override
			public InputStream getInputStream() throws IOException {
				return in;
			}
		};
	}

	/**
	 * 指定のリソースを示すURLでインスタンスを構築します。
	 * 
	 * @param url
	 *            URL
	 * @throws NullPointerException
	 *             urlが{@code null}の場合
	 */
	public SimpleAttachmentFile(final URL url) {
		Assertion.notNull("url", url);

		this.file = new AttachmentFile() {

			@Override
			public String getFileName() {
				return url.getFile();
			}

			@Override
			public InputStream getInputStream() throws IOException {
				return url.openStream();
			}
		};
	}

	/**
	 * @see com.wankuma.mail.AttachmentFile#getFileName()
	 */
	@Override
	public String getFileName() {
		return this.file.getFileName();
	}

	/**
	 * @see com.wankuma.mail.AttachmentFile#getInputStream()
	 */
	@Override
	public InputStream getInputStream() throws IOException {
		return this.file.getInputStream();
	}
}
