/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail.javamail;

import java.util.Properties;

import com.wankuma.commons.i18n.Messages;
import com.wankuma.mail.Configuration;
import com.wankuma.mail.MailException;
import com.wankuma.mail.MailService;

/**
 * {@link MailService}の基底クラスです。
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */
public abstract class AbstractJavaMailService implements MailService {

	private static Messages messages = Messages
			.getMessages(AbstractJavaMailService.class);

	private boolean closed;

	private Properties properties;

	/**
	 * 指定の接続プロパティでインスタンスを構築します。
	 * 
	 * @param properties
	 *            プロパティ
	 */
	public AbstractJavaMailService(Properties properties) {
		this.properties = properties == null ? new Properties()
				: (Properties) properties.clone();
		this.closed = true;
	}

	/**
	 * 接続が既に行われていることを検証します。
	 * 
	 * @throws MailException
	 */
	protected void assertConnected() throws MailException {
		if (this.isClosed()) {
			throw new MailException(AbstractJavaMailService.messages
					.get("E001"));
		}
	}

	/**
	 * @see com.wankuma.mail.MailService#close()
	 */
	@Override
	public void close() throws MailException {
		if (this.isClosed()) {
			return;
		}

		this.disconnect();
		this.closed = true;
	}

	/**
	 * 指定のサーバ接続プロパティと接続設定でサーバとの接続を行います。
	 * 
	 * @param configuration
	 *            接続設定
	 * @param properties
	 *            接続プロパティ
	 * @throws MailException
	 *             サーバとの接続に失敗した場合
	 */
	protected abstract void connect(Configuration configuration,
			Properties properties) throws MailException;

	/**
	 * サーバとの接続を切断します。
	 * 
	 * @throws MailException
	 *             サーバとの接続の切断に失敗した場合
	 */
	protected abstract void disconnect() throws MailException;

	/**
	 * @see java.lang.Object#finalize()
	 */
	@Override
	protected void finalize() throws Throwable {
		try {
			super.finalize();
		} finally {
			this.close();
		}
	}

	/**
	 * @see com.wankuma.mail.MailService#isClosed()
	 */
	@Override
	public boolean isClosed() {
		return this.closed;
	}

	/**
	 * @see com.wankuma.mail.MailService#open()
	 */
	@Override
	public void open() throws MailException {
		this.open(null);
	}

	/**
	 * @see com.wankuma.mail.MailService#open(com.wankuma.mail.Configuration)
	 */
	@Override
	public void open(Configuration configuration) throws MailException {
		if (!this.isClosed()) {
			return;
		}

		Configuration configuration1 = configuration == null ? new Configuration()
				: configuration;
		Properties properties = new Properties(this.properties);

		this.connect(configuration1, properties);
		this.closed = false;
	}
}
