/*
 * Copyright 2006-2008 The Wankuma.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.wankuma.mail.helper;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.wankuma.commons.lang.Assertion;
import com.wankuma.mail.AttachmentFile;
import com.wankuma.mail.Header;
import com.wankuma.mail.Importance;
import com.wankuma.mail.Mail;
import com.wankuma.mail.MailAddress;
import com.wankuma.mail.MailBody;
import com.wankuma.mail.MessageIdGenerator;

/**
 * シンプルな{@link Mail}の実装です。
 * 
 * @author Katsunori Koyanagi
 * @version 1.0
 */
public class SimpleMail implements Mail {

	private List<AttachmentFile> attachmentFiles;

	private List<MailAddress> bcc;

	private MailBody body;

	private List<MailAddress> cc;

	private List<String> envelopeTo;

	private MailAddress from;

	private List<Header> headers;

	private Importance importance;

	private MessageIdGenerator messageIdGenerator;

	private List<MailAddress> replyTo;

	private String returnPath;

	private Date sentDate;

	private String subject;

	private List<MailAddress> to;

	/**
	 * インスタンスを構築します。
	 */
	public SimpleMail() {
		this.to = new ArrayList<MailAddress>();
		this.cc = new ArrayList<MailAddress>();
		this.bcc = new ArrayList<MailAddress>();
		this.replyTo = new ArrayList<MailAddress>();
		this.attachmentFiles = new ArrayList<AttachmentFile>();
		this.headers = new ArrayList<Header>();
		this.envelopeTo = new ArrayList<String>();
	}

	/**
	 * 添付ファイルを追加します。
	 * 
	 * @param attachmentFile
	 *            添付ファイル
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             attachmentFileが{@code null}の場合
	 */
	public SimpleMail addAttachmentFile(AttachmentFile attachmentFile) {
		Assertion.notNull("attachmentFile", attachmentFile);

		this.attachmentFiles.add(attachmentFile);
		return this;
	}

	/**
	 * BCCのメールアドレスを追加します。
	 * 
	 * @param address
	 *            メールアドレス
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             addressが{@code null}の場合
	 */
	public SimpleMail addBcc(MailAddress address) {
		Assertion.notNull("address", address);

		this.bcc.add(address);
		return this;
	}

	/**
	 * BCCのメールアドレスを追加します。
	 * 
	 * @param address
	 *            メールアドレス
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             addressが{@code null}の場合
	 */
	public SimpleMail addBcc(String address) {
		Assertion.notNull("address", address);

		return this.addBcc(address, null);
	}

	/**
	 * BCCのメールアドレスを追加します。
	 * 
	 * @param address
	 *            メールアドレス
	 * @param personal
	 *            名前
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             addressが{@code null}の場合
	 */
	public SimpleMail addBcc(String address, String personal) {
		Assertion.notNull("address", address);

		return this.addBcc(new SimpleMailAddress(address, personal));
	}

	/**
	 * CCのメールアドレスを追加します。
	 * 
	 * @param address
	 *            メールアドレス
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             addressが{@code null}の場合
	 */
	public SimpleMail addCc(MailAddress address) {
		Assertion.notNull("address", address);

		this.cc.add(address);
		return this;
	}

	/**
	 * CCのメールアドレスを追加します。
	 * 
	 * @param address
	 *            メールアドレス
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             addressが{@code null}の場合
	 */
	public SimpleMail addCc(String address) {
		Assertion.notNull("address", address);

		return this.addCc(address, null);
	}

	/**
	 * CCのメールアドレスを追加します。
	 * 
	 * @param address
	 *            メールアドレス
	 * @param personal
	 *            名前
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             addressが{@code null}の場合
	 */
	public SimpleMail addCc(String address, String personal) {
		Assertion.notNull("address", address);

		return this.addCc(new SimpleMailAddress(address, personal));
	}

	/**
	 * 実際の配信先のメールアドレスを追加します。
	 * 
	 * @param address
	 *            メールアドレス
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             addressが{@code null}の場合
	 */
	public SimpleMail addEnvelopeTo(String address) {
		Assertion.notNull("address", address);

		this.envelopeTo.add(address);
		return this;
	}

	/**
	 * メールヘッダを追加します。
	 * 
	 * @param header
	 *            メールヘッダ
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             headerが{@code null}の場合
	 */
	public SimpleMail addHeader(Header header) {
		Assertion.notNull("header", header);

		this.headers.add(header);
		return this;
	}

	/**
	 * メールヘッダを追加します。
	 * 
	 * @param name
	 *            名前
	 * @param value
	 *            値
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             nameが{@code null}の場合
	 */
	public SimpleMail addHeader(String name, String value) {
		Assertion.notNull("name", name);

		return this
				.addHeader(new SimpleHeader(name, value == null ? "" : value));
	}

	/**
	 * 返信先のメールアドレスを追加します。
	 * 
	 * @param address
	 *            メールアドレス
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             addressが{@code null}の場合
	 */
	public SimpleMail addReplyTo(MailAddress address) {
		Assertion.notNull("address", address);

		this.replyTo.add(address);
		return this;
	}

	/**
	 * 返信先のメールアドレスを追加します。
	 * 
	 * @param address
	 *            メールアドレス
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             addressが{@code null}の場合
	 */
	public SimpleMail addReplyTo(String address) {
		Assertion.notNull("address", address);

		return this.addReplyTo(address, null);
	}

	/**
	 * 返信先のメールアドレスを追加します。
	 * 
	 * @param address
	 *            メールアドレス
	 * @param personal
	 *            名前
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             addressが{@code null}の場合
	 */
	public SimpleMail addReplyTo(String address, String personal) {
		Assertion.notNull("address", address);

		return this.addReplyTo(new SimpleMailAddress(address, personal));
	}

	/**
	 * TOのメールアドレスを追加します。
	 * 
	 * @param address
	 *            メールアドレス
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             addressが{@code null}の場合
	 */
	public SimpleMail addTo(MailAddress address) {
		Assertion.notNull("address", address);

		this.to.add(address);
		return this;
	}

	/**
	 * TOのメールアドレスを追加します。
	 * 
	 * @param address
	 *            メールアドレス
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             addressが{@code null}の場合
	 */
	public SimpleMail addTo(String address) {
		Assertion.notNull("address", address);

		return this.addTo(address, null);
	}

	/**
	 * TOのメールアドレスを追加します。
	 * 
	 * @param address
	 *            メールアドレス
	 * @param personal
	 *            名前
	 * @return 自分自身のインスタンス
	 * 
	 * @throws NullPointerException
	 *             addressが{@code null}の場合
	 */
	public SimpleMail addTo(String address, String personal) {
		Assertion.notNull("address", address);

		return this.addTo(new SimpleMailAddress(address, personal));
	}

	/**
	 * 設定されている添付ファイルをクリアします。
	 * 
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail clearAttachmentFiles() {
		this.attachmentFiles.clear();
		return this;
	}

	/**
	 * 設定されているBCCのメールアドレスをクリアします。
	 * 
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail clearBcc() {
		this.bcc.clear();
		return this;
	}

	/**
	 * 設定されているCCのメールアドレスをクリアします。
	 * 
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail clearCc() {
		this.cc.clear();
		return this;
	}

	/**
	 * 設定されている実際の配信先のメールアドレスをクリアします。
	 * 
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail clearEnvelopeTo() {
		this.envelopeTo.clear();
		return this;
	}

	/**
	 * 設定されているヘッダをクリアします。
	 * 
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail clearHeaders() {
		this.headers.clear();
		return this;
	}

	/**
	 * 設定されている返信先のメールアドレスをクリアします。
	 * 
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail clearReplyTo() {
		this.replyTo.clear();
		return this;
	}

	/**
	 * 設定されているTOのメールアドレスをクリアします。
	 * 
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail clearTo() {
		this.to.clear();
		return this;
	}

	/**
	 * @see com.wankuma.mail.Mail#getAttachmentFiles()
	 */
	@Override
	public AttachmentFile[] getAttachmentFiles() {
		return this.attachmentFiles
				.toArray(new AttachmentFile[this.attachmentFiles.size()]);
	}

	/**
	 * @see com.wankuma.mail.Mail#getBcc()
	 */
	@Override
	public MailAddress[] getBcc() {
		return this.bcc.toArray(new MailAddress[this.bcc.size()]);
	}

	/**
	 * @see com.wankuma.mail.Mail#getBody()
	 */
	@Override
	public MailBody getBody() {
		return this.body;
	}

	/**
	 * @see com.wankuma.mail.Mail#getCc()
	 */
	@Override
	public MailAddress[] getCc() {
		return this.cc.toArray(new MailAddress[this.cc.size()]);
	}

	/**
	 * @see com.wankuma.mail.Mail#getEnvelopeTo()
	 */
	@Override
	public String[] getEnvelopeTo() {
		return this.envelopeTo.toArray(new String[this.envelopeTo.size()]);
	}

	/**
	 * @see com.wankuma.mail.Mail#getFrom()
	 */
	@Override
	public MailAddress getFrom() {
		return this.from;
	}

	/**
	 * @see com.wankuma.mail.Mail#getHeaders()
	 */
	@Override
	public Header[] getHeaders() {
		return this.headers.toArray(new Header[this.headers.size()]);
	}

	/**
	 * @see com.wankuma.mail.Mail#getImportance()
	 */
	@Override
	public Importance getImportance() {
		return this.importance;
	}

	/**
	 * @see com.wankuma.mail.Mail#getMessageIdGenerator()
	 */
	@Override
	public MessageIdGenerator getMessageIdGenerator() {
		return this.messageIdGenerator;
	}

	/**
	 * @see com.wankuma.mail.Mail#getReplayTo()
	 */
	@Override
	public MailAddress[] getReplayTo() {
		return this.replyTo.toArray(new MailAddress[this.replyTo.size()]);
	}

	/**
	 * @see com.wankuma.mail.Mail#getReturnPath()
	 */
	@Override
	public String getReturnPath() {
		return this.returnPath;
	}

	/**
	 * @see com.wankuma.mail.Mail#getSentDate()
	 */
	@Override
	public Date getSentDate() {
		return this.sentDate;
	}

	/**
	 * @see com.wankuma.mail.Mail#getSubject()
	 */
	@Override
	public String getSubject() {
		return this.subject;
	}

	/**
	 * @see com.wankuma.mail.Mail#getTo()
	 */
	@Override
	public MailAddress[] getTo() {
		return this.to.toArray(new MailAddress[this.to.size()]);
	}

	/**
	 * メールボディを設定します。
	 * 
	 * @param body
	 *            ボディ
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail setBody(MailBody body) {
		this.body = body;
		return this;
	}

	/**
	 * 送信先のメールアドレスを設定します。
	 * 
	 * @param from
	 *            メールアドレス
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail setFrom(MailAddress from) {
		this.from = from;
		return this;
	}

	/**
	 * 送信先のメールアドレスを設定します。
	 * 
	 * @param address
	 *            メールアドレス
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail setFrom(String address) {
		return this.setFrom(address == null ? null : new SimpleMailAddress(
				address));
	}

	/**
	 * 送信先のメールアドレスを設定します。
	 * 
	 * @param address
	 *            メールアドレス
	 * @param personal
	 *            名前
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail setFrom(String address, String personal) {
		return this.setFrom(address == null ? null : new SimpleMailAddress(
				address, personal));
	}

	/**
	 * 指定の重要度を設定します。
	 * 
	 * @param importance
	 *            重要度
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail setImportance(Importance importance) {
		this.importance = importance;
		return this;
	}

	/**
	 * 指定のメッセージIDを設定します。
	 * 
	 * @param messageId
	 *            メッセージID
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail setMessageId(final String messageId) {
		if (messageId == null) {
			this.messageIdGenerator = null;
			return this;
		}

		this.messageIdGenerator = new MessageIdGenerator() {
			@Override
			public String getMessageId() {
				return messageId;
			}
		};

		return this;
	}

	/**
	 * メッセージIDのジェネレータを設定します。
	 * 
	 * @param messageIdGenerator
	 *            メッセージIDのジェネレータ
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail setMessageIdGenerator(
			MessageIdGenerator messageIdGenerator) {
		this.messageIdGenerator = messageIdGenerator;
		return this;
	}

	/**
	 * 配信エラー時等に利用されるメールの返却先のメールアドレスを設定します。
	 * 
	 * @param returnPath
	 *            メールアドレス
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail setReturnPath(String returnPath) {
		this.returnPath = returnPath;
		return this;
	}

	/**
	 * 送信日時を設定します。
	 * 
	 * @param sentDate
	 *            送信日時
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail setSentDate(Date sentDate) {
		this.sentDate = sentDate;

		return this;
	}

	/**
	 * 件名を設定します。
	 * 
	 * @param subject
	 *            件名
	 * @return 自分自身のインスタンス
	 */
	public SimpleMail setSubject(String subject) {
		this.subject = subject;
		return this;
	}
}
