/*
 * @(#) $Id: CorrectedContentTypeDataSource.java,v 1.6 2004/07/12 04:48:14 Shin
 * Exp $ $Revision: 1.6 $ Copyright (c) 2000 Shin Kinoshita All Rights Reserved.
 */
package com.sk_jp.mail;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.activation.DataSource;
import javax.mail.MessageAware;
import javax.mail.MessageContext;
import javax.mail.MessagingException;
import javax.mail.Part;
import javax.mail.internet.ContentType;
import javax.mail.internet.ParseException;

/**
 * Content-Type:の不適合をISO-2022-JPに補正します。 使用方法は
 * 
 * <PRE>
 * Object o = new DataHandler(
 *               new CorrectedContentTypeDataSource(part, charset)
 *            ).getContent();
 * </PRE>
 * 
 * <P>
 * のようになります。
 * </P>
 * <P>
 * スレッドセーフではありませんので利用者側で排他制御を行ってください。
 * </P>
 * 
 * @author Shin
 * @version $Revision: 1.6 $ $Date: 2004/07/12 04:48:14 $
 */
class CorrectedContentTypeDataSource implements DataSource, MessageAware {
	protected String charset;

	protected DataSource source;

	public CorrectedContentTypeDataSource() {
	}

	public CorrectedContentTypeDataSource(DataSource dataSource,
			String defaultCharset) {
		this.setDataSource(dataSource);
		this.setDefaultCharset(defaultCharset);
	}

	public CorrectedContentTypeDataSource(Part part, String defaultCharset)
			throws MessagingException {
		this.setPart(part);
		this.setDefaultCharset(defaultCharset);
	}

	public String getContentType() {
		ContentType contentType = null;
		try {
			contentType = new ContentType(this.source.getContentType());
		} catch (ParseException e) {
			return "text/plain; charset=" + this.charset;
		}
		String specifiedCharset = contentType.getParameter("charset");
		if (specifiedCharset == null) {
			// Content-Type:が存在しない場合は"text/plain"になってしまう。
			// 本当にtext/plainだった場合は正しくない事になるが、
			// charset=ISO-2022-JPにする場合は一応表示上は問題ない。
			contentType.setParameter("charset", this.charset);
		}
		return contentType.toString();
	}

	public InputStream getInputStream() throws IOException {
		return this.source.getInputStream();
	}

	public synchronized MessageContext getMessageContext() {
		if (this.source instanceof MessageAware) {
			return ((MessageAware) this.source).getMessageContext();
		}
		throw new RuntimeException(this.source + " isn't MessageAware.");
	}

	public String getName() {
		return this.source.getName();
	}

	public OutputStream getOutputStream() throws IOException {
		return this.source.getOutputStream();
	}

	public void setDataSource(DataSource newSource) {
		this.source = newSource;
	}

	public void setDefaultCharset(String defaultCharset) {
		this.charset = defaultCharset;
	}

	public void setPart(Part part) throws MessagingException {
		// getDataHandler() method creates a implicit DataSource.
		this.setDataSource(part.getDataHandler().getDataSource());
	}
}