/*
 * @(#) $Id: ByteArrayDataSource.java,v 1.4 2001/03/11 03:14:14 shin Exp $
 * $Revision: 1.4 $ Copyright (c) 2000 Shin Kinoshita All Rights Reserved.
 */
package com.sk_jp.mail;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.UnsupportedEncodingException;

import javax.activation.DataSource;
import javax.mail.internet.ContentType;
import javax.mail.internet.ParseException;

/**
 * 任意のバイナリデータ及びテキストデータを表現するDataSourceです。 charsetパラメタに対応しています。<BR>
 * demoのByteArrayDataSourceを改良して日本語文字列に対応したものです。
 */
public class ByteArrayDataSource implements DataSource {
	private String contentType;

	private byte[] data;

	/**
	 * バイナリデータのデータソースを生成します。
	 */
	public ByteArrayDataSource(byte[] data, String contentType) {
		this.data = data;
		this.contentType = contentType;
	}

	/**
	 * バイナリデータのデータソースを生成します。
	 */
	public ByteArrayDataSource(InputStream in, String contentType)
			throws IOException {
		this.contentType = contentType;
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		byte[] buf = new byte[2048];
		int len;

		while ((len = in.read(buf)) != -1) {
			out.write(buf, 0, len);
		}
		this.data = out.toByteArray();
	}

	/**
	 * 文字ストリームからtext/*用のデータソースを生成します。
	 * contentTypeのcharsetパラメタに応じてエンコードされたデータを生成します。
	 * charsetパラメタは、Readerに施されたエンコーディングと 同じものが指定されなければなりません。
	 */
	public ByteArrayDataSource(Reader in, String contentType)
			throws IOException {
		this.contentType = contentType;
		String charset = null;
		try {
			ContentType ct = new ContentType(contentType);
			charset = ct.getParameter("charset");
		} catch (ParseException e) {
		}
		if (charset == null) {
			charset = "us-ascii";
		}
		BufferedReader reader = new BufferedReader(in);
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		String line;

		while ((line = reader.readLine()) != null) {
			out.write(line.getBytes(charset));
			out.write('\r');
			out.write('\n');
		}
		this.data = out.toByteArray();
	}

	/**
	 * 文字列からtext/*用のデータソースを生成します。 contentTypeのcharsetパラメタに応じてエンコードされたデータを生成します。
	 */
	public ByteArrayDataSource(String text, String contentType) {
		this.contentType = contentType;
		String charset = null;
		try {
			ContentType ct = new ContentType(contentType);
			charset = ct.getParameter("charset");
		} catch (ParseException e) {
		}
		if (charset == null) {
			charset = "us-ascii";
		}
		try {
			this.data = text.getBytes(charset);
		} catch (UnsupportedEncodingException e) {
			try {
				this.data = text.getBytes("us-ascii");
			} catch (UnsupportedEncodingException e2) {
			}
		}
	}

	// ////////////////////////////////////////////////////////////////////////
	// 以降はDataSource interfaceのメソッド群です。
	// 重要なのはgetInputStream()でそのデータのストリーム表現を返すことです。

	public String getContentType() {
		return this.contentType;
	}

	/**
	 * Return an InputStream for the data. Note - a new stream must be returned
	 * each time.
	 */
	public InputStream getInputStream() throws IOException {
		if (this.data == null) {
			throw new IOException("no data");
		}
		return new ByteArrayInputStream(this.data);
	}

	public String getName() {
		return "dummy";
	}

	public OutputStream getOutputStream() throws IOException {
		throw new IOException("cannot do this");
	}
}
