/*
 * @(#) $Id: UnicodeCorrectWriter.java,v 1.3 2001/03/11 03:11:40 shin Exp $
 * $Revision: 1.3 $ Copyright (c) 2000 Shin Kinoshita All Rights Reserved.
 */
package com.sk_jp.io;

import java.io.FilterWriter;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.io.Writer;

/**
 * UnicodeCorrectWriter.
 * <p>
 * WriterをラップしてWAVE DASHなどのtoUnicodeConverterの 変換揺れを補正します。
 * </p>
 * <p>
 * 既存のWriterオブジェクトに対してフィルタリングを 行いたい場合には、CorrectOutputStreamWriterではなくこちらを利用します。
 * </p>
 * <p>
 * JSPで文字化けを防ぐために用います。<BR>
 * JspWriterにラップしようと思ったが、結局JspWriter型にすることができなくて
 * いまいち。しかたないので別途wrappedOutみたいな変数にこれで ラップしたJspWriterオブジェクトを入れて、日本語を出力する部分に対しては
 * wrappedOutの方で出力するようにします。
 * </p>
 * 
 * <pre>例:
 * PrintWriter wrappedOut = new PrintWriter(new UnicodeCorrectWriter(out));
 * wrappedOut.print("～");
 * </pre>
 * 
 * <p>
 * &lt;%= arg %&gt;等の箇所は&lt;% wrappedOut.print(arg) %&gt;と記述する 必要があります。
 * </p>
 * 
 * @version $Revision: 1.3 $ $Date: 2001/03/11 03:11:40 $
 * @author Shin
 */
public class UnicodeCorrectWriter extends FilterWriter {
	private UnicodeCorrector corrector;

	/**
	 * Create an UnicodeCorrectWriter that uses the platform default character
	 * encoding.
	 * 
	 * @param out
	 *            An OutputStream
	 */
	public UnicodeCorrectWriter(Writer out) {
		super(out);
		try {
			this.corrector = UnicodeCorrector.getInstance(System
					.getProperty("file.encoding"));
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException(e.getMessage());
		}
	}

	/**
	 * Create an UnicodeCorrectWriter that uses the named character encoding.
	 * 
	 * @param out
	 *            An OutputStream
	 * @param enc
	 *            Name of the encoding to be used
	 * @exception UnsupportedEncodingException
	 *                If the named encoding is not supported
	 */
	public UnicodeCorrectWriter(Writer out, String enc)
			throws UnsupportedEncodingException {
		super(out);
		this.corrector = UnicodeCorrector.getInstance(enc);
	}

	/**
	 * Write a portion of an array of characters.
	 * <p>
	 * 特定の文字を特定エンコーディングで出力しようとした際の sun.ioコンバータでは正常に変換できない部分を補正します。
	 * </p>
	 * 
	 * @param cbuf
	 *            Buffer of characters
	 * @param off
	 *            Offset from which to start writing characters
	 * @param len
	 *            Number of characters to write
	 * 
	 * @exception IOException
	 *                If an I/O error occurs
	 */
	@Override
	public void write(char cbuf[], int off, int len) throws IOException {
		char[] buf = this.corrector.correct(cbuf, off, len);
		super.write(buf, 0, buf.length);
	}

	/**
	 * Write a single character.
	 * 
	 * @exception IOException
	 *                If an I/O error occurs
	 */
	@Override
	public void write(int c) throws IOException {
		char[] buf = new char[1];
		buf[0] = (char) c;
		this.write(buf, 0, 1);
	}

	/**
	 * Write a portion of a string.
	 * 
	 * @param str
	 *            String to be written
	 * @param off
	 *            Offset from which to start reading characters
	 * @param len
	 *            Number of characters to be written
	 * 
	 * @exception IOException
	 *                If an I/O error occurs
	 */
	@Override
	public void write(String str, int off, int len) throws IOException {
		this.write(str.toCharArray(), off, len);
	}
}
