/*!
******************************************************************************

	@file	x86.h

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#ifndef __X86_H__
#define __X86_H__

#pragma pack(1)

#define X86_STI()			__ASM__("sti")
#define X86_CLI()			__ASM__("cli")
#define X86_HLT()			__ASM__("hlt")
#define X86_LGDT(p)			__ASM__("lgdt %0" : : "m"(p))
#define X86_SGDT(p)			__ASM__("sgdt %0" : "=m"(p))
#define X86_LIDT(p)			__ASM__("lidt %0" : : "m"(p))
#define X86_SIDT(p)			__ASM__("sidt %0" : "=m"(p))
#define X86_LLDT(seg)		__ASM__("lldt %w0" : : "r"(seg))
#define X86_SLDT(seg)		__ASM__("sldt %0" : "=g"(seg))
#define X86_LTR(seg)		__ASM__("ltr  %w0" : : "r"(seg))
#define X86_STR(seg)		__ASM__("str  %0" : "=g"(seg))
#define X86_GET_CR(n, val)	__ASM__("movl %%cr"#n", %0" : "=r"(val))
#define X86_SET_CR(n, val)	__ASM__("movl %0, %%cr"#n"" : : "r"(val))
#define X86_GET_DR(n, val)	__ASM__("movl %%dr"#n", %0" : "=r"(val))
#define X86_SET_DR(n, val)	__ASM__("movl %0, %%dr"#n"" : : "r"(val))
#define X86_GET_FS(val)		__ASM__("movl %%fs, %0" : "=r"(val))
#define X86_SET_FS(val)		__ASM__("movl %0, %%fs" : : "r"(val))
#define X86_GET_GS(val)		__ASM__("movl %%gs, %0" : "=r"(val))
#define X86_SET_GS(val)		__ASM__("movl %0, %%gs" : : "r"(val))

#define V86_SEG_SS			0x1000
#define V86_SEG_ES			0x2000
#define V86_SEG_DS			0x3000
#define V86_SEG_CS			0x4000

typedef struct {
	u16		limit;
	u32		base;
} BASELIMIT;

typedef union {
	struct {
		u32		low32;
		u32		high32;
	};
	struct {
		u16			limit_b15_0;
		u16			base_b15_0;
		u8			base_b23_16;
		union {
			u16		flags;
			struct {
				u8	type		 : 5;
				u8	dpl			 : 2;
				u8	present		 : 1;
				u8	limit_b23_16 : 4;
				u8	available	 : 1;
				u8	long_mode	 : 1;
				u8	size32		 : 1;
				u8	granularity	 : 1;
			};
		};
		u8			base_b31_24;
	} seg;
	struct {
		u16		off_L;
		u16		seg;
		u8		count;
		u8		type;
		u16		off_H;
	} gate;
} DESCRIPTOR;

#define GDT_ENTRY_MAX		256
#define IDT_ENTRY_MAX		64

#define SEG_VMM_CODE		0x0008
#define SEG_VMM_DATA		0x0010
#define SEG_VMM_TSS			0x0018
#define SEG_V86_TSS			0x0020
//#define SEG_TSS(n)		(u16)(0x0028 + (n)*0x20 + 0x00)
//#define SEG_FS_GS(n)		(u16)(0x0028 + (n)*0x20 + 0x08)
//#define SEG_SS(n)			(u16)(0x0028 + (n)*0x20 + 0x10)
#define SEG_LDT(n)			(u16)(0x0028 + (n)*0x20 + 0x18)


#define DESC_SIZE32			0x4000
#define DESC_PRESENT		0x80
#define DESC_TSS_BUSY		0x02

#define DESC_TYPE_TSS16		0x01
#define DESC_TYPE_LDT		0x02
#define DESC_TYPE_TSS16_B	(DESC_TYPE_TSS16 | DESC_TSS_BUSY)
#define DESC_TYPE_CALL16	0x04
#define DESC_TYPE_TASK_G	0x05
#define DESC_TYPE_INT16		0x06
#define DESC_TYPE_TRAP16	0x07
#define DESC_TYPE_TSS32		0x09
#define DESC_TYPE_TSS32_B	(DESC_TYPE_TSS32 | DESC_TSS_BUSY)
#define DESC_TYPE_CALL32	0x0C
#define DESC_TYPE_INT32		0x0E
#define DESC_TYPE_TRAP32	0x0F
#define DESC_TYPE_DATA(opt)	(0x10 | (opt))
#define DS_EXPAND_DOWN		0x04
#define DS_WRITABLE			0x02
#define DS_ACCESSED			0x01
#define DESC_TYPE_CODE(opt)	(0x18 | (opt))
#define CS_CONFORMING		0x04
#define CS_READABLE			0x02
#define CS_ACCESSED			0x01
#define DESC_TYPE_CODE32	(DESC_SIZE32 | DESC_TYPE_CODE(CS_READABLE))
#define DESC_TYPE_DATA32	(DESC_SIZE32 | DESC_TYPE_DATA(DS_WRITABLE))
#define DESC_TYPE_STACK32	(DESC_SIZE32 | DESC_TYPE_DATA(DS_WRITABLE|DS_EXPAND_DOWN))
#define DESC_TYPE_CODE16	(DESC_TYPE_CODE(CS_READABLE))
#define DESC_TYPE_DATA16	(DESC_TYPE_DATA(DS_WRITABLE))
#define DESC_TYPE_STACK16	(DESC_TYPE_DATA(DS_WRITABLE|DS_EXPAND_DOWN))

#define F_CF			0x00000001
#define F_PF			0x00000004
#define F_AF			0x00000010
#define F_ZF			0x00000040
#define F_SF			0x00000080
#define F_TF			0x00000100
#define F_IF			0x00000200
#define F_DF			0x00000400
#define F_OF			0x00000800
#define F_IOPL0			0x00000000
#define F_IOPL1			0x00001000
#define F_IOPL2			0x00002000
#define F_IOPL3			0x00003000
#define F_NT			0x00004000
#define F_RF			0x00010000
#define F_VM			0x00020000
#define F_AC			0x00040000
#define F_VIF			0x00080000
#define F_VIP			0x00100000
#define F_ID			0x00200000

#define CR0_PE			0x00000001
#define CR0_MP			0x00000002
#define CR0_EM			0x00000004
#define CR0_TS			0x00000008
#define CR0_ET			0x00000010
#define CR0_NE			0x00000020
#define CR0_WP			0x00010000
#define CR0_AM			0x00040000
#define CR0_NW			0x20000000
#define CR0_CD			0x40000000
#define CR0_PG			0x80000000

#define CR3_PWT			0x00000008
#define CR3_PCD			0x00000010

#define CR4_VME			0x00000001
#define CR4_PVI			0x00000002
#define CR4_TSD			0x00000004
#define CR4_DE			0x00000008
#define CR4_PSE			0x00000010
#define CR4_PAE			0x00000020
#define CR4_MCE			0x00000040
#define CR4_PGE			0x00000080
#define CR4_OSFXSR		0x00000100
#define CR4_OSXMMEXCPT	0x00000200

#define DR6_B0			0x00000001
#define DR6_B1			0x00000002
#define DR6_B2			0x00000004
#define DR6_B3			0x00000008
#define DR6_BD			0x00002000
#define DR6_BS			0x00004000
#define DR6_BT			0x00008000

#define PDE_PRESENT		0x00000001
#define PDE_READWRITE	0x00000002
#define PDE_READONLY	0x00000000
#define PDE_USER		0x00000004
#define PDE_KERNEL		0x00000000
#define PDE_PWT			0x00000008
#define PDE_PCD			0x00000010
#define PDE_ACCESS		0x00000020
#define PDE_PS			0x00000080
#define PDE_GLOBAL		0x00000100

#define PTE_PRESENT		0x00000001
#define PTE_READWRITE	0x00000002
#define PTE_READONLY	0x00000000
#define PTE_USER		0x00000004
#define PTE_KERNEL		0x00000000
#define PTE_PWT			0x00000008
#define PTE_PCD			0x00000010
#define PTE_ACCESS		0x00000020
#define PTE_DIRTY		0x00000040
#define PTE_PAT			0x00000080
#define PTE_GLOBAL		0x00000100

#define EXCEPTION_DB	0x01

typedef struct {
	u32		edi, esi, ebp, esp, ebx, edx, ecx, eax;
	u32		es, ds;
	u32		n;
	u32		err, eip, cs, eflags;
	u32		r3esp, r3ss, r3es, r3ds, r3fs, r3gs;
} EXCEPTION_HANDLER_ARGS;

typedef void (CALLBACK *EXCEPTION_HANDLER)( EXCEPTION_HANDLER_ARGS * );

typedef struct {
	u32		eax, ecx, edx, ebx, ebp, esi, edi;
	u32		eflags;
	void *	es_base;
	void *	ds_base;
	u32		es_size, ds_size;
	u16		ret_es, ret_ds;
} V86INT_REGS;

typedef struct {
	u16		cw;
	u16		rsvd1;
	u16		sw;
	u16		rsvd2;
	u16		tw;
	u16		rsvd3;
	u32		ip_off;
	u16		ip_sel;
	u16		op;
	u32		op_off;
	u16		op_sel;
	u16		rsvd;
	u8		data_regs[80];
} FPU_REGS;

typedef struct {
	u32		ret_addr;
	u32		eip;
	u32		eflags;
	u32		eax, ecx, edx, ebx;
	u32		esp, ebp, esi, edi;
	u32		es, cs, ss, ds, fs, gs;
} V86_REGS;

typedef union _TSS {
	u8		b[512];
	struct {
		u32			prev_link;
		u32			esp0, ss0;
		u32			esp1, ss1;
		u32			esp2, ss2;
		u32			cr3;
		u32			eip;
		u32			eflags;
		u32			eax, ecx, edx, ebx;
		u32			esp, ebp, esi, edi;
		u32			es, cs, ss, ds, fs, gs;
		u32			ldtr;
		u16			trap_flag;
		u16			iomap_off;
		FPU_REGS	fpu;
		V86_REGS	v86;
		u32			intmap[8];
		u32			iomap[1];
	};
} TSS;

typedef struct {
	u32		off;
	u16		seg;
} TSS_PTR;

#pragma pack()

extern bool		x86_init( void );

extern TSS *	x86_alloc_tss( const void *, const void *, size_t );
extern void		x86_switch_task( register TSS *, register TSS * );

extern void		x86_set_gdt_desc( u16, u32, u32, u16 );
extern void		x86_set_idt_desc( u8,  u16, u32, u8  );
extern void		x86_v86int( u8 vector, V86INT_REGS *regs );

extern u32		x86_get_pte( void * );
extern void		x86_set_pte( void *, u32 );
extern void		x86_map_page( u32 pa );

extern void		x86_enable_single_step_exec( void );
extern void		x86_disable_single_step_exec( void );

#endif // !__X86_H__
