/*!
******************************************************************************

	@file	ohci_hub.cpp

	Copyright (C) 2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "pci.h"
#include "usb.h"
#include "ohci.h"
#include "usb/class/hub.h"
#include "timer.h"
#include "printf.h"

#define OHCI_HUB_STRING_ID_PRODUCT	1

static const char *ohci_hub_name = "Vsun86 OHCI Root Hub";
static const u32 ohci_hub_name_len = 20;

static const USB_STRING_DESC ohci_hub_str_desc = {
	sizeof(USB_STRING_DESC),	// bLength
	USB_DT_STRING >> 8,			// bDescriptorType
	{ { 0 } }					// wLANGID[0]
};

static const USB_DEVICE_DESC ohci_hub_dev_desc = {
	sizeof(USB_DEVICE_DESC),	// bLength
	USB_DT_DEVICE >> 8,			// bDescriptorType
	0x0200,						// bcdUSB
	USB_CLASS_HUB,				// bDeviceClass
	0,							// bDeviceSubClass
	0xFF,						// bDeviceProtocol
	64,							// bMaxPacketSize0
	0,							// idVendor
	0,							// idProduct
	0,							// bcdDevice
	0,							// iManufacturer
	OHCI_HUB_STRING_ID_PRODUCT,	// iProduct
	0,							// iSerialNumber
	1							// bNumConfigurations
};

static const struct {
	USB_CONFIG_DESC		cfg;
	USB_INTERFACE_DESC	iface;
} ohci_hub_cfg_desc = {
	{	// Configuration Descriptor
		sizeof(USB_CONFIG_DESC),		// bLength
		USB_DT_CONFIG >> 8,				// bDescriptorType
		sizeof(ohci_hub_cfg_desc),		// wTotalLength
		1,								// bNumInterfaces
		0,								// bConfigurationValue
		0,								// iConfiguration
		0,								// bmAttributes
		0								// bMaxPower
	},
	{	// Interface Descriptor
		sizeof(USB_INTERFACE_DESC),		// bLength
		USB_DT_INTERFACE >> 8,			// bDescriptorType
		0,								// bInterfaceNumber
		0,								// bAlternateSetting
		0,								// bNumEndpoints
		USB_CLASS_HUB,					// bInterfaceClass
		0,								// bInterfaceSubClass
		0,								// bInterfaceProtocol
		0,								// iInterface
	}
};

static bool ohci_hub_ctrl_xfer( USB_DEVICE *dev, u8 req_type, u8 request, u16 value, u16 index,
								void *buf, size_t buf_len, int timeout );
static bool ohci_hub_bulk_xfer( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout );
static bool ohci_hub_interrupt_xfer( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout );

static USB_HCD usb_hcd_hub_ohci = {
	ohci_alloc_queue,
	ohci_hub_ctrl_xfer,
	ohci_hub_bulk_xfer,
	ohci_hub_interrupt_xfer
};

static USB_HCD usb_hcd_ohci = {
	ohci_alloc_queue,
	ohci_ctrl_xfer,
	ohci_bulk_xfer,
	ohci_interrupt_xfer
};

static bool ohci_get_hub_descriptor( OHCI_HCD *hcd, u16 type, USB_HUB_DESC *desc, size_t buf_len );
static bool ohci_set_hub_feature( OHCI_HCD *hcd, u16 feature );
static bool ohci_clear_hub_feature( OHCI_HCD *hcd, u16 feature );
static bool ohci_set_port_feature( OHCI_HCD *hcd, u8 port, u16 feature );
static bool ohci_clear_port_feature( OHCI_HCD *hcd, u8 port, u16 feature );

bool ohci_init_hub( OHCI_HCD *hcd )
{
	hcd->dev = usb_alloc_device( &usb_hcd_hub_ohci, hcd, false );
	if ( hcd->dev == NULL )
		return false;

	OHCI_REGS *regs = hcd->regs;

	// ルートハブの電源を入れる
	ohci_write32( &regs->HcRhStatus, OHCI_REG_RHST_LPSC );	// SetGlobalPower
	timer_usleep( hcd->power_wait * 1000 );

	// リモートウェイクアップを有効にする
	u32 ctrl_reg = ohci_read32( &regs->HcControl );
	ctrl_reg |= OHCI_REG_CTRL_RWE;
	ohci_write32( &regs->HcControl,  ctrl_reg );			// RemoteWakeupEnable
	ohci_write32( &regs->HcRhStatus, OHCI_REG_RHST_DRWE );	// DeviceRemoteWakeupEnable

	if ( !usb_register( hcd->dev, &usb_hcd_ohci ) ) {
		vmm_printf( VMM_DEBUG, "usb_ohci(%d): [hub] usb_register() failed.\n", hcd->id );
		return false;
	}

	return true;
}

static bool ohci_hub_ctrl_xfer( USB_DEVICE *dev, u8 req_type, u8 request, u16 value, u16 index,
								void *buf, size_t buf_len, int timeout )
{
	(void)timeout;
	OHCI_HCD *hcd = (OHCI_HCD *)dev->hcd_priv;

//	vmm_printf( VMM_DEBUG, "usb_ohci(%d): [CTRL_XFER] req_type=%02x, request=%02x, value=%04x, index=%04x\n",
//				hcd->id, req_type, request, value, index );

	switch ( req_type & USB_REQ_TYPE_MASK )
	{
	case USB_REQ_TYPE_STD:
		{	// 標準リクエスト
			switch ( request )
			{
			case USB_GET_DESCRIPTOR:
				if ( USB_REQ_DIR_OUT == (req_type & USB_REQ_DIR_MASK) )
					break;
				if ( USB_REQ_DEVICE  != (req_type & USB_REQ_RECP_MASK) )
					break;
				if ( buf == NULL )
					break;
				switch ( value & 0xFF00 )
				{
				case USB_DT_DEVICE:
					{	// デバイスディスクリプタを返す
						if ( buf_len < sizeof(ohci_hub_dev_desc) )
							break;
						memcpy( buf, &ohci_hub_dev_desc, sizeof(ohci_hub_dev_desc) );
						return true;
					}
					break;

				case USB_DT_CONFIG:
					{	// コンフィギュレーションディスクリプタを返す
						if ( buf_len < sizeof(USB_CONFIG_DESC) )
							break;
						if ( buf_len < sizeof(ohci_hub_cfg_desc) ) {
							memcpy( buf, &ohci_hub_cfg_desc, sizeof(USB_CONFIG_DESC) );
							return true;
						}
						memcpy( buf, &ohci_hub_cfg_desc, sizeof(ohci_hub_cfg_desc) );
						return true;
					}
					break;

				case USB_DT_STRING:
					{	// ストリングディスクリプタを返す
						switch ( value & 0xFF )
						{
						case 0:
							{	// 言語コードを返す
								if ( buf_len < sizeof(ohci_hub_str_desc) )
									break;
								memcpy( buf, &ohci_hub_str_desc, sizeof(ohci_hub_str_desc) );
								return true;
							}
							break;

						case OHCI_HUB_STRING_ID_PRODUCT:
							{	// 製品名を返す
								if ( buf_len < sizeof(ohci_hub_str_desc) )
									break;
								USB_STRING_DESC *desc = (USB_STRING_DESC *)buf;
								desc->bLength = sizeof(ohci_hub_str_desc) + ohci_hub_name_len * 2;
								desc->bDescriptorType = USB_DT_STRING >> 8;
								if ( buf_len < desc->bLength )
									return true;
								for ( size_t i=0; i<ohci_hub_name_len; i++ )
									desc->wString[i] = (u16)ohci_hub_name[i];
								desc->wString[ohci_hub_name_len] = 0;
								return true;
							}
							break;
						}
					}
					break;
				}
				break;

			case USB_SET_ADDRESS:
			case USB_SET_CONFIGURATION:
			case USB_SET_INTERFACE:
				return true;
			}
		}
		break;

	case USB_REQ_TYPE_CLASS:
		{	// クラス固有リクエスト
			switch ( request )
			{
			case USB_HUB_GET_DESCRIPTOR:
				{	// GetHubDescriptor
					if ( USB_REQ_DIR_OUT == (req_type & USB_REQ_DIR_MASK) )
						break;
					if ( index != 0 )
						break;
					if ( !ohci_get_hub_descriptor( hcd, value, (USB_HUB_DESC *)buf, buf_len ) )
						break;
					return true;
				}
				break;

			case USB_HUB_GET_STATUS:
				{	// GetHubStatus / GetPortStatus
					if ( USB_REQ_DIR_OUT == (req_type & USB_REQ_DIR_MASK) )
						break;
					switch ( req_type & USB_REQ_RECP_MASK )
					{
					case USB_REQ_DEVICE:
						{	// GetHubStatus
							if ( (value != 0) || (index != 0) || (buf == NULL) || (buf_len != 4) )
								break;
							*(u32 *)buf = ohci_read32( &hcd->regs->HcRhStatus );
							return true;
						}
						break;
					case USB_REQ_OTHER:
						{	// GetPortStatus
							const u8 port = (u8)(index - 1);
							if ( (value != 0) || (port >= hcd->ports) || (buf == NULL) || (buf_len != 4) )
								break;
							*(u32 *)buf = ohci_read32( &hcd->regs->HcRhPortStatus[port] );
							return true;
						}
						break;
					}
				}
				break;

			case USB_HUB_SET_FEATURE:
				{	// SetHubFeature / SetPortFeature
					if ( USB_REQ_DIR_IN == (req_type & USB_REQ_DIR_MASK) )
						break;
					switch ( req_type & USB_REQ_RECP_MASK )
					{
					case USB_REQ_DEVICE:
						{	// SetHubFeature
							if ( (index != 0) || (buf_len != 0) )
								break;
							if ( !ohci_set_hub_feature( hcd, value ) )
								break;
							return true;
						}
						break;
					case USB_REQ_OTHER:
						{	// SetPortFeature
							const u8 port = (u8)(index - 1);
							if ( (port >= hcd->ports) || (buf_len != 0) )
								break;
							if ( !ohci_set_port_feature( hcd, port, value ) )
								break;
							return true;
						}
						break;
					}
				}
				break;

			case USB_HUB_CLEAR_FEATURE:
				{	// ClearHubFeature / ClearPortFeature
					if ( USB_REQ_DIR_IN == (req_type & USB_REQ_DIR_MASK) )
						break;
					switch ( req_type & USB_REQ_RECP_MASK )
					{
					case USB_REQ_DEVICE:
						{	// ClearHubFeature
							if ( (index != 0) || (buf_len != 0) )
								break;
							if ( !ohci_clear_hub_feature( hcd, value ) )
								break;
							return true;
						}
						break;
					case USB_REQ_OTHER:
						{	// ClearPortFeature
							const u8 port = (u8)(index - 1);
							if ( (port >= hcd->ports) || (buf_len != 0) )
								break;
							if ( !ohci_clear_port_feature( hcd, port, value ) )
								break;
							return true;
						}
						break;
					}
				}
				break;
			}
		}
		break;
	}

	vmm_printf( VMM_DEBUG, "ohci_hub_ctrl_msg() failed. (req_type=%02x, request=%02x, value=%04x, index=%04x)\n",
				req_type, request, value, index );

	return false;
}

static bool ohci_hub_bulk_xfer( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout )
{
	(void)dev;
	(void)ep;
	(void)buf;
	(void)buf_len;
	(void)timeout;

	return false;
}

static bool ohci_hub_interrupt_xfer( USB_DEVICE *dev, u8 ep, void *buf, size_t buf_len, int timeout )
{
	(void)dev;
	(void)ep;
	(void)buf;
	(void)buf_len;
	(void)timeout;

	return false;
}

static bool ohci_get_hub_descriptor( OHCI_HCD *hcd, u16 type, USB_HUB_DESC *desc, size_t buf_len )
{
	if ( type != USB_DT_HUB )
		return false;

	if ( buf_len < sizeof(USB_HUB_DESC) )
		return false;

	const u8 add_bytes = (hcd->ports <= 8)? 1:2;

	const u32 rhdesc_a = ohci_read32( &hcd->regs->HcRhDescriptorA );
	const u32 rhdesc_b = ohci_read32( &hcd->regs->HcRhDescriptorB );

	u16 hub_char = 0;
	if ( (rhdesc_a & OHCI_REG_RHDA_NPS) && (rhdesc_a & OHCI_REG_RHDA_PSM) )
		hub_char |= 0x01;
	if ( rhdesc_a & OHCI_REG_RHDA_NOCP ) {
		if ( rhdesc_a & OHCI_REG_RHDA_OCPM )
			hub_char |= 0x08;
	}
	else {
		hub_char |= 0x10;
	}

	desc->bLength			  = sizeof(USB_HUB_DESC) + (add_bytes << 1);
	desc->bDescriptorType	  = (u8)(USB_DT_HUB >> 8);
	desc->bNbrPorts			  = OHCI_RH_PORT_NUM(rhdesc_a);
	desc->wHubCharacteristics = hub_char;
	desc->bPwrOn2PwrGood	  = OHCI_RH_POTPGT(rhdesc_a);
	desc->bHubContrCurrent	  = 0;

	size_t buf_remain = buf_len - sizeof(USB_HUB_DESC);
	if ( buf_remain <= 0 )
		return true;

	u8 *p = (u8 *)desc + sizeof(USB_HUB_DESC);
	if ( buf_remain <= add_bytes ) {
		memcpy( p, &((u16 *)&rhdesc_b)[0], buf_remain );
		return true;
	}
	memcpy( p, &((u16 *)&rhdesc_b)[0], add_bytes );
	buf_remain -= add_bytes;

	p += add_bytes;
	if ( buf_remain <= add_bytes )
		memcpy( p, &((u16 *)&rhdesc_b)[1], buf_remain );
	else
		memcpy( p, &((u16 *)&rhdesc_b)[1], add_bytes );

	return true;
}

static bool ohci_set_hub_feature( OHCI_HCD *hcd, u16 feature )
{
	(void)hcd;
	(void)feature;

	return false;
}

static bool ohci_clear_hub_feature( OHCI_HCD *hcd, u16 feature )
{
	volatile u32 *reg = &hcd->regs->HcRhStatus;

	switch ( feature )
	{
	case USB_HUB_C_OVER_CURRENT:
		ohci_write32( reg, OHCI_REG_RHST_OCIC );	// OverCurrentIndicatorChange
		return true;
	}

	return false;
}

static bool ohci_set_port_feature( OHCI_HCD *hcd, u8 port, u16 feature )
{
	volatile u32 *reg = &hcd->regs->HcRhPortStatus[port];

	switch ( feature )
	{
	case USB_HUB_PORT_SUSPEND:
//		vmm_printf( VMM_DEBUG, "usb_ohci(%d): SetPortSuspend\n", hcd->id );
		ohci_write32( reg, OHCI_REG_PRTST_PSS );	// SetPortSuspend
		return true;

	case USB_HUB_PORT_RESET:
//		vmm_printf( VMM_DEBUG, "usb_ohci(%d): SetPortReset\n", hcd->id );
		ohci_write32( reg, OHCI_REG_PRTST_PRS );	// SetPortReset
		return true;

	case USB_HUB_PORT_POWER:
//		vmm_printf( VMM_DEBUG, "usb_ohci(%d): SetPortPower\n", hcd->id );
		ohci_write32( reg, OHCI_REG_PRTST_PPS );	// SetPortPower
		return true;
	}

	return false;
}

static bool ohci_clear_port_feature( OHCI_HCD *hcd, u8 port, u16 feature )
{
	volatile u32 *reg = &hcd->regs->HcRhPortStatus[port];

	switch ( feature )
	{
	case USB_HUB_PORT_ENABLE:
		ohci_write32( reg, OHCI_REG_PRTST_CCS );	// ClearPortEnable
		return true;

	case USB_HUB_PORT_SUSPEND:
		ohci_write32( reg, OHCI_REG_PRTST_POCI );	// ClearSuspendStatus
		return true;

	case USB_HUB_PORT_POWER:
		ohci_write32( reg, OHCI_REG_PRTST_LSDA );	// ClearPortPower
		return true;

	case USB_HUB_C_PORT_CONNECTION:
		ohci_write32( reg, OHCI_REG_PRTST_CSC );	// ConnectStatusChange
		return true;

	case USB_HUB_C_PORT_SUSPEND:
		ohci_write32( reg, OHCI_REG_PRTST_PSSC );	// PortSuspendStatusChange
		return true;

	case USB_HUB_C_PORT_OVER_CURRENT:
		ohci_write32( reg, OHCI_REG_PRTST_OCIC );	// PortOverCurrentIndicatorChange
		return true;

	case USB_HUB_C_PORT_RESET:
		ohci_write32( reg, OHCI_REG_PRTST_PRSC );	// PortResetStatusChange
		return true;
	}

	return false;
}
