/*!
******************************************************************************

	@file	ohci.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "pci.h"
#include "usb.h"
#include "ohci.h"
#include "usb/class/hub.h"
#include "timer.h"
#include "printf.h"

static u8 ohci_num;
static OHCI_HCD ohci_hcd[OHCI_HOST_MAX];

#define OHCI_HCCABUF		NON_CACHED_PTR( _ohci_hccabuf, void )
#define OHCI_HCCABUF_SIZE	0x00001000		// 4KB   (16*256B)
ALIGN(4096) static u8 _ohci_hccabuf[OHCI_HCCABUF_SIZE];

static bool ohci_irq_handler( u8 irq, void *args );

bool ohci_init( void )
{
	ohci_num = 0;
	memset( ohci_hcd, 0, sizeof(ohci_hcd) );

	return true;
}

bool ohci_probe( PCI_DEVICE *dev )
{
	const u8 index = ohci_num;

	if ( index >= OHCI_HOST_MAX )
		return false;

	u32 cmd_status = (u32)pci_read_config( dev, 0x04 );
	cmd_status |= 0x06;		// MA=1,BM=1
	pci_write_config( dev, 0x04, cmd_status );

	if ( !PCI_ADDR_IS_MEM( dev->base_addr[0] ) )
		return false;

	OHCI_REGS *regs = (OHCI_REGS *)(dev->base_addr[0] & 0xFFFFF000);
	if ( regs == NULL )
		return false;

	OHCI_HCD *hcd = &ohci_hcd[index];
	hcd->ctrl_ed_list = NULL;
	hcd->bulk_ed_list = NULL;

	OHCI_HCCA *hcca = &((OHCI_HCCA *)OHCI_HCCABUF)[index];
	for ( u32 i=0; i<32; i++ ) {
		hcd->int_ed[i] = NULL;
		ohci_write32( &hcca->HccaInterruptTable[i], OHCI_NULL_ED_PTR );
	}
	ohci_write16( &hcca->HccaFrameNumber, 0 );
	ohci_write16( &hcca->HccaPad1,		  0 );
	ohci_write32( &hcca->HccaDoneHead,	  0 );

	// SMMドライバ/BIOSドライバが動作しているか確認し、コントローラを初期化する
	u32 ctrl_reg = ohci_read32( &regs->HcControl );
	if ( ctrl_reg & OHCI_REG_CTRL_IR )
	{	// SMMドライバが動作中
//		vmm_printf( VMM_DEBUG, "usb_ohci(%d): SMM active.\n", index );
		ohci_write32( &regs->HcCommandStatus, OHCI_REG_CMDST_OCR );		// OwnerChangeRequest
		int timeout = 1000;
		while ( timeout ) {
			ctrl_reg = ohci_read32( &regs->HcControl );
			if ( !(ctrl_reg & OHCI_REG_CTRL_IR) )
				break;
			timer_usleep( 1000 );
			timeout--;
		}
		if ( timeout < 0 ) {
			vmm_printf( VMM_DEBUG, "usb_ohci(%d): OwnerChangeRequest failed.\n", index );
			return false;
		}
	}
	else if ( OHCI_STATE( regs ) != OHCI_STATE_RESET )
	{	// BIOSドライバが動作中
		vmm_printf( VMM_DEBUG, "usb_ohci(%d): BIOS active. (not supported)\n", index );
		return false;
	}
	else
	{	// SMMドライバ/BIOSドライバともに動作していない
//		vmm_printf( VMM_DEBUG, "usb_ohci(%d): not active.\n", index );
	}

	// HcFmIntervalレジスタの値を保存してHCをリセットする
	const u32 fm_interval = ohci_read32( &regs->HcFmInterval );
	u32 periodic_start = OHCI_FRAME_INTERVAL( fm_interval );
	periodic_start *= 9;
	periodic_start /= 10;
	ohci_write32( &regs->HcCommandStatus, OHCI_REG_CMDST_HCR );		// HostControllerReset
	timer_usleep( 10 );
	if ( OHCI_STATE(regs) != OHCI_STATE_SUSPEND ) {
		vmm_printf( VMM_ERROR, "usb_ohci(%d): OHCI_STATE != OHCI_STATE_SUSPEND\n", index );
		return false;
	}
	regs->HcFmInterval = fm_interval;

	// レジスタを設定する
	ctrl_reg |= OHCI_REG_CTRL_PLE | OHCI_REG_CTRL_IE | OHCI_REG_CTRL_CLE | OHCI_REG_CTRL_BLE;
	ohci_write32( &regs->HcControlHeadED,	OHCI_NULL_ED_PTR );
	ohci_write32( &regs->HcBulkHeadED,		OHCI_NULL_ED_PTR );
	ohci_write32( &regs->HcHCCA,			virt_to_phys( (void *)hcca ) );
	ohci_write32( &regs->HcInterruptEnable, (OHCI_REG_INT_ALL & ~OHCI_REG_INT_SF) | OHCI_REG_INT_MIE );
	ohci_write32( &regs->HcControl,			ctrl_reg );
	ohci_write32( &regs->HcPeriodicStart,	periodic_start );

	// 処理を開始する
	ctrl_reg |= OHCI_STATE_OPERATIONAL;
	ohci_write32( &regs->HcControl, ctrl_reg );

	const u32 root_hub_desc_a = ohci_read32( &regs->HcRhDescriptorA );
//	const u32 root_hub_desc_b = ohci_read32( &regs->HcRhDescriptorB );
	const u8 ports = OHCI_RH_PORT_NUM( root_hub_desc_a );
	const u8 power_wait = OHCI_RH_POTPGT( root_hub_desc_a ) * 2;

	hcd->id			= index;
	hcd->ports		= ports;
	hcd->power_wait	= power_wait;
	hcd->regs		= regs;
	hcd->hcca		= hcca;

//	if ( !pci_irq_register( dev, ohci_irq_handler, hcd ) ) {
//		vmm_printf( VMM_DEBUG, "usb_ohci(%d): pci_irq_register() failed. (irq %d)\n", index, dev->interrupt_line );
//		return false;
//	}
	(void)ohci_irq_handler;

	const u32 revision = ohci_read32( &regs->HcRevision );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): revision=%d.%d", index, (revision & 0xF0) >> 4, revision & 0x0F );
	if ( revision & 0x100 ) {
		hcd->legacy_support = true;
		vmm_printf( VMM_DEBUG, ", legacy supported\n" );
	}
	else {
		hcd->legacy_support = false;
		vmm_printf( VMM_DEBUG, ", legacy not supported\n" );
	}

	vmm_printf( VMM_DEBUG, "usb_ohci(%d): ports=%d, regs=%08x, hcca=%08x(%08x)\n",
				index, ports, regs, hcca, virt_to_phys( (void *)hcca ) );

	/*
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcControl ............ %08x\n",
				index, ohci_read32( &regs->HcControl ) );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcCommandStatus ...... %08x\n",
				index, ohci_read32( &regs->HcCommandStatus ) );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcInterruptStatus .... %08x\n",
				index, ohci_read32( &regs->HcInterruptStatus ) );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcRhStatus ........... %08x\n",
				index, ohci_read32( &regs->HcRhStatus ) );
	for ( int i=0; i<ports; i++ )
		vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcRhPortStatus[%d] .... %08x\n",
					index, i, ohci_read32( &regs->HcRhPortStatus[i] ) );

	vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcRhDescriptorA ...... %08x\n", index, root_hub_desc_a );
	vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcRhDescriptorB ...... %08x\n", index, root_hub_desc_b );
	*/

	ohci_num++;
	return true;
}

bool ohci_start( void )
{
	for ( u8 index=0; index<ohci_num; index++ ) {
		if ( !ohci_init_hub( &ohci_hcd[index] ) )
			continue;
	}

	return true;
}

u16 ohci_read16( volatile u16 *p )
{
	return *p;
}

u32 ohci_read32( volatile u32 *p )
{
	return *p;
}

void ohci_write16( volatile u16 *p, u16 data )
{
	*p = data;
}

void ohci_write32( volatile u32 *p, u32 data )
{
	*p = data;
}

static bool ohci_irq_handler( u8 irq, void *args )
{
	(void)irq;

	OHCI_HCD  *hcd  = (OHCI_HCD *)args;
	OHCI_REGS *regs = hcd->regs;

	vmm_printf( VMM_DEBUG, "usb_ohci(%d): interrupt !!!\n", hcd->id );

	// 割り込み要因をチェックする
	const u32 int_status = ohci_read32( &regs->HcInterruptStatus ) & regs->HcInterruptEnable;
	if ( int_status == 0 )
		return false;	// 割り込み要因がない

	// 処理が終わるまで割り込みを止める
	ohci_write32( &regs->HcInterruptDisable, OHCI_REG_INT_MIE );

	// 割り込みステータスをチェックしてクリアする
	if ( int_status & OHCI_REG_INT_SO )
	{	// SchedulingOverrun
		vmm_printf( VMM_DEBUG, "usb_ohci(%d): SchedulingOverrun\n", hcd->id );
		ohci_write32( &regs->HcInterruptStatus, OHCI_REG_INT_SO );
	}
	if ( int_status & OHCI_REG_INT_WDH )
	{	// WritebackDoneHead
//		vmm_printf( VMM_DEBUG, "usb_ohci(%d): WritebackDoneHead\n", hcd->id );
		ohci_write32( &regs->HcInterruptStatus, OHCI_REG_INT_WDH );
	}
	if ( int_status & OHCI_REG_INT_SF )
	{	// StartofFrame
//		vmm_printf( VMM_DEBUG, "usb_ohci(%d): StartofFrame\n", hcd->id );
		ohci_write32( &regs->HcInterruptStatus, OHCI_REG_INT_SF );
	}
	if ( int_status & OHCI_REG_INT_RD )
	{	// ResumeDetected
		vmm_printf( VMM_DEBUG, "usb_ohci(%d): ResumeDetected\n", hcd->id );
		ohci_write32( &regs->HcInterruptStatus, OHCI_REG_INT_RD );
	}
	if ( int_status & OHCI_REG_INT_UE )
	{	// UnrecoverableError
		vmm_printf( VMM_DEBUG, "usb_ohci(%d): UnrecoverableError\n", hcd->id );
		ohci_write32( &regs->HcInterruptStatus, OHCI_REG_INT_UE );
	}
	if ( int_status & OHCI_REG_INT_FNO )
	{	// FrameNumberOverflow
//		vmm_printf( VMM_DEBUG, "usb_ohci(%d): FrameNumberOverflow (HccaFrameNumber=%04x)\n",
//					hcd->id, hcd->hcca->HccaFrameNumber );
		ohci_write32( &regs->HcInterruptStatus, OHCI_REG_INT_FNO );
	}
	if ( int_status & OHCI_REG_INT_RHSC )
	{	// RootHubStatusChange
//		vmm_printf( VMM_DEBUG, "usb_ohci(%d): RootHubStatusChange (HcRhStatus=%08x)\n",
//					hcd->id, ohci_read32( &regs->HcRhStatus ) );
		ohci_write32( &regs->HcInterruptStatus, OHCI_REG_INT_RHSC );
	}
	if ( int_status & OHCI_REG_INT_OC )
	{	// OwnershipChange
		vmm_printf( VMM_DEBUG, "usb_ohci(%d): OwnershipChange\n", hcd->id );
		ohci_write32( &regs->HcInterruptStatus, OHCI_REG_INT_OC );
	}
//	vmm_printf( VMM_DEBUG, "usb_ohci(%d): HcInterruptStatus = %08x\n",
//				hcd->id, regs->HcInterruptStatus );

	// 割り込みを有効にする
	ohci_write32( &regs->HcInterruptEnable, OHCI_REG_INT_MIE );

	return true;
}
