/*!
******************************************************************************

	@file	shell.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "task.h"
#include "shell.h"
#include "keyboard.h"
#include "uart.h"
#include "syscall.h"
#include "mboot.h"
#include "timer.h"
#include "disk.h"
#include "usb.h"

#include <stdio.h>
#include <string.h>

#include "shell/apitest.h"
#include "shell/cpuinfo.h"
#include "shell/date.h"
#include "shell/vminit.h"
#include "shell/vmrun.h"
#include "shell/vmemdump.h"
#include "shell/biosdump.h"

#pragma pack(1)

typedef struct {
	u16		bfType;
	u32		bfSize;
	u16		bfReserved1;
	u16		bfReserved2;
	u32		bfOffBits;
} BITMAPFILEHEADER;

typedef struct {
	u32		biSize;
	long	biWidth;
	long	biHeight;
	u16		biPlanes;
	u16		biBitCount;
	u32		biCompression;
	u32		biSizeImage;
	long	biXPixPerMeter;
	long	biYPixPerMeter;
	u32		biClrUsed;
	u32		biClrImportant;
} BITMAPINFOHEADER;

#pragma pack()

typedef struct _console_line_info {
	struct {
		int		c;
		RGB32	color;
	} column[CONSOLE_COLUMN_MAX];
	struct _console_line_info	* next;
	struct _console_line_info	* prev;
} CONSOLE_LINE_INFO;

static struct {
	int		x;
	int		y;
	int		columns;
	int		lines;
	int		start_line;
	CONSOLE_LINE_INFO *	top_line;
	CONSOLE_LINE_INFO *	cur_line;
	CONSOLE_LINE_INFO	line[CONSOLE_LINE_MAX];
} console;

static bool shell_vm_running = false;
static VM_EVENT_CALLBACK shell_vm_eventcb = NULL;

static char shell_get_ascii( u8 mod, u8 key_code );
static void shell_exec_command( const char *cmd );

#define SHELL_CMD_BUF_MAX	512

bool shell_init( void )
{
	console.x			= 0;
	console.y			= 0;
	console.columns		= 80;
	console.lines		= 30;
	console.start_line	= 0;
	console.top_line	= &console.line[0];
	console.cur_line	= &console.line[0];
	memset( console.line, 0, sizeof(console.line) );
	console.line[0].prev = &console.line[console.lines-1];
	console.line[0].next = &console.line[1];
	for ( int i=1; i<console.lines-1; i++ ) {
		console.line[i].prev = &console.line[i-1];
		console.line[i].next = &console.line[i+1];
	}
	console.line[console.lines-1].prev = &console.line[console.lines-2];
	console.line[console.lines-1].next = &console.line[0];

	return true;
}

void shell_task( void *args )
{
	(void)args;

	char cmd_buf[SHELL_CMD_BUF_MAX + 1];
	u32  cmd_len;

	// Start Disk Manager
	if ( !disk_start() )
		abort();

#ifndef _VSUN86_PCSIM
	// Start USB Device Manager
	if ( !usb_start() )
		abort();

	const char *p = strchr( mboot_get_cmdline(), ' ' );
	if ( p != NULL ) {
		if ( strncmp( p+1, "nec98", 5 ) == 0 ) {
			shell_cmd_vminit( "nec98" );
			shell_vm_running = shell_cmd_vmrun( p+6 );
		}
	}
#endif	//!_VSUN86_PCSIM

	shell_put_string( "\n> " );
	memset( cmd_buf, 0, sizeof(cmd_buf) );
	cmd_len = 0;

	// 時計更新用にタイマハンドラを登録する
	u32 timer_id_update_clock = timer_add( 500, NULL, NULL, false, TASK_ID_SHELL );

	TASK_EVENT event;
	while ( task_wait_event( &event ) )
	{
		switch ( event.msg )
		{
		case MSG_SHELL_VM_EXIT:
			{	// VMが終了した
				shell_vm_running = false;
				shell_put_string( "\n> " );
			}
			break;

		case MSG_SHELL_EXEC_CMD:
			{	// コマンドを実行する
				if ( !shell_vm_running ) {
					shell_exec_command( (const char *)event.arg1 );
					memset( cmd_buf, 0, sizeof(cmd_buf) );
					cmd_len = 0;
					shell_put_string( "\n> " );
					task_send_event( event.from, MSG_SHELL_EXEC_FINISHED, 0, 0 );
				}
			}
			break;

		case MSG_KEY_UP:
			{	// キーが離された
				if ( shell_vm_running )
				{	// VM実行中→VMでイベントを処理する
					if ( shell_vm_eventcb != NULL )
						shell_vm_eventcb( VM_EVENT_KEY_UP, event.arg1, event.arg2 );
					break;
				}
			}
			break;

		case MSG_KEY_DOWN:
			{	// キーが押された
				const char c = shell_get_ascii( (u8)event.arg1, (u8)event.arg2 );
				if ( shell_vm_running )
				{	// VM実行中→VMでイベントを処理する
					if ( shell_vm_eventcb != NULL )
						shell_vm_eventcb( VM_EVENT_KEY_DOWN, event.arg1, event.arg2 );
					break;
				}
				if ( c != '\0' )
				{	// 英数字キー: 文字を出力
					if ( cmd_len < SHELL_CMD_BUF_MAX ) {
						cmd_buf[cmd_len++] = c;
						cmd_buf[cmd_len] = '\0';
						shell_put_char( c );
					}
				}
				else
				{	// 制御キー: キーコードで判定
					switch ( event.arg2 )
					{
					case KEY_BACKSPACE:
						{	// 1文字消す
							if ( cmd_len > 0 ) {
								cmd_len--;
								shell_put_char( '\b' );
							}
						}
						break;

					case KEY_RETURN:
						{	// コマンドを実行する
							if ( cmd_len > 0 ) {
								shell_exec_command( cmd_buf );
								memset( cmd_buf, 0, sizeof(cmd_buf) );
								cmd_len = 0;
							}
							shell_put_string( "\n> " );
						}
						break;

					case KEY_PRINTSCREEN:
						{	// VRAMの内容を転送する
#ifndef _VSUN86_PCSIM
							const u32 vram_size = disp_get_vram_size();
							BITMAPFILEHEADER bf = {
								0x4D42, sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER) + vram_size,
								0, 0, sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER)
							};
							BITMAPINFOHEADER bi = {
								sizeof(BITMAPINFOHEADER), 640, 480, 1, 32, 0, vram_size, 0, 0, 0, 0
							};
							shell_put_string( "Sending screenshot ... " );
							uart_send_binary( &bf, sizeof(bf) );
							uart_send_binary( &bi, sizeof(bi) );
							uart_send_binary( disp_lock(), vram_size );
							disp_unlock();
							shell_put_string( "finished.\n> " );
#endif	//!_VSUN86_PCSIM
						}
						break;
					}
				}
			}
			break;

		case MSG_TIMER_EXPIRED:
			{	// タイマ満了
				if ( event.arg1 == timer_id_update_clock )
				{	// 時計を更新する
					static const char *wday[] = { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };
					char clockbuf[32];
					VSUN86_SYSTEM_TIME t;
					timer_get_local_time( &t );
					sprintf( clockbuf, "%4d/%02d/%02d(%s) %02d:%02d:%02d",
							 t.year, t.mon, t.mday, wday[t.wday], t.hour, t.min, t.sec );
					disp_draw_text( 0, 464, clockbuf, (size_t)-1, COLOR_WHITE );
					timer_id_update_clock = timer_add( 500, NULL, NULL, false, TASK_ID_SHELL );
				}
			}
			break;
		}
	}
}

static char shell_get_ascii( u8 mod, u8 key_code )
{
	static const char ascii[2][64] =
	{	//	+0  +1  +2  +3  +4  +5  +6  +7  +8  +9  +A  +B  +C  +D  +E  +F
		{	 0 , 0 ,'1','2','3','4','5','6','7','8','9','0','-','^','\\',0 ,
			 0 ,'q','w','e','r','t','y','u','i','o','p','@','[', 0 , 0 ,'a',
			's','d','f','g','h','j','k','l',';',':',']', 0 , 0 , 0 ,'z','x',
			'c','v','b','n','m',',','.','/','\\',0 , 0 , 0 , 0 ,' ', 0 , 0
		},
		{	 0 , 0 ,'!','"','#','$','%','&','\'','(',')',0 ,'=','~','|', 0 ,
			 0 ,'Q','W','E','R','T','Y','U','I','O','P','`','{', 0 , 0 ,'A',
			'S','D','F','G','H','J','K','L','+','*','}', 0 , 0 , 0 ,'Z','X',
			'C','V','B','N','M','<','>','?','_', 0 , 0 , 0 , 0 ,' ', 0 , 0
		}
	};
	if ( key_code >= 0x40 )
		return '\0';

	if ( 0 != (mod & ~(KEY_MOD_LSHIFT|KEY_MOD_RSHIFT)) )
		return '\0';	// [CTRL]や[ALT]が押されている
	if ( 0 == (mod & (KEY_MOD_LSHIFT|KEY_MOD_RSHIFT)) )
		return ascii[0][key_code];
	else
		return ascii[1][key_code];
}

void shell_set_vm_eventcb( void *proc )
{
	shell_vm_eventcb = (VM_EVENT_CALLBACK)proc;
}

static void shell_exec_command( const char *cmd )
{
	if ( (cmd == NULL) || (*cmd == '\0') )
		return;

	shell_put_char( '\n' );
	if ( 0 == strcmp( cmd, "version" ) ) {
		shell_put_string( vsun86_ver_string );
		shell_put_string( "\n" );
	}
#ifndef _VSUN86_PCSIM
	else if ( 0 == strcmp( cmd, "apitest" ) )
		shell_cmd_apitest();							// "apitest"
	else if ( 0 == strcmp( cmd, "cpuinfo" ) )
		shell_cmd_cpuinfo();							// "cpuinfo"
	else if ( 0 == strcmp( cmd, "date" ) )
		shell_cmd_date();								// "date"
	else if ( 0 == strncmp( cmd, "vminit ", 7 ) )
		shell_cmd_vminit( &cmd[7] );					// "vminit"
	else if ( 0 == strncmp( cmd, "vmrun", 5 ) )
		shell_vm_running = shell_cmd_vmrun( &cmd[5] );	// "vmrun"
	else if ( 0 == strcmp( cmd, "vmemdump" ) )
		shell_cmd_vmemdump();							// "vmemdump"
	else if ( 0 == strcmp( cmd, "biosdump" ) )
		shell_cmd_biosdump();							// "biosdump"
#endif	//!_VSUN86_PCSIM
	else if ( 0 == strcmp( cmd, "taskdump" ) )
		task_dump();									// "taskdump"
	else
		shell_put_string( "Command not found.\n" );
}

void shell_put_char( int c, RGB32 color )
{
	int x = console.x;
	int y = console.y;

	switch ( c )
	{
	case '\b':
		x--;
		if ( x < 0 ) {
			x = console.columns - 1;
			y--;
			console.cur_line = console.cur_line->prev;
		}
		c = console.cur_line->column[x].c;
		console.cur_line->column[x].color = COLOR_NONE;
		disp_write_char( x<<3, y<<4, c, COLOR_NONE );
		break;

	case '\n':
		x = 0;
		y++;
		console.cur_line = console.cur_line->next;
		break;

	default:
		console.cur_line->column[x].c	  = c;
		console.cur_line->column[x].color = color;
		disp_write_char( x<<3, y<<4, c, color );
		x++;
		if ( x >= console.columns ) {
			x = 0;
			y++;
			console.cur_line = console.cur_line->next;
		}
		break;
	}

	if ( y != console.y ) {
		if ( y > console.lines - 2 )
		{	// 上方向にスクロール
			y = console.lines - 2;
			console.top_line = console.top_line->next;
			disp_vertical_scroll( 16, 463, -16, CONSOLE_BGCOLOR );
		}
		else if ( y < 0 )
		{	// 下方向にスクロール
			y = 0;
			console.top_line = console.top_line->prev;
			disp_vertical_scroll(  0, 447, +16, CONSOLE_BGCOLOR );
		}
	}

	console.x = x;
	console.y = y;
}

void shell_put_string( const char *str, RGB32 color )
{
	int x = console.x;
	int y = console.y;

	int start_x = x;
	int start_y = y;
	int off = 0;
	size_t cnt = 0;

	for ( int i=0; str[i] != '\0'; i++ ) {
		if ( (str[i] == '\n') || (x >= console.columns) ) {
			disp_draw_text( start_x<<3, start_y<<4, &str[off], cnt, color );
			console.cur_line = console.cur_line->next;
			x = 0;
			y++;
			if ( y != console.y ) {
				if ( y > console.lines - 2 )
				{	// 上方向にスクロール
					y = console.lines - 2;
					console.top_line = console.top_line->next;
					disp_vertical_scroll( 16, 463, -16, CONSOLE_BGCOLOR );
				}
			}
			start_x = x;
			start_y = y;
			off = i + 1;
			cnt = 0;
			continue;
		}
		x++;
		cnt++;
	}
	disp_draw_text( start_x<<3, start_y<<4, &str[off], cnt, color );

	console.x = x;
	console.y = y;
}
