/*!
******************************************************************************

	@file	pic.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "cpu.h"
#include "pic.h"
#include "irq.h"
#include "io.h"
#include "atomic.h"
#include "printf.h"

static volatile int pic_processing;
#ifdef	_VSUN86_PCSIM
static u8 pic_imr[2] = { 0xFF, 0xFF };
static u8 pic_irr[2] = { 0x00, 0x00 };
static u8 pic_isr[2] = { 0x00, 0x00 };
#endif	//_VSUN86_PCSIM

bool pic_init( void )
{
#ifndef _VSUN86_PCSIM
	const u8 vector = IRQ_START_VECTOR;

	outb( IO_PIC0_OCW1, 0xFF );			// OCW1: 全割り込み禁止
	outb( IO_PIC1_OCW1, 0xFF );			// OCW1: 全割り込み禁止

	outb( IO_PIC0_ICW1, 0x11 );			// ICW1: 初期化開始
	outb( IO_PIC0_ICW2, vector + 0 );	// ICW2: マスタPICの割り込み
	outb( IO_PIC0_ICW3, 0x04 );			// ICW3: IR2がスレーブに接続
	outb( IO_PIC0_ICW4, 0x01 );			// ICW4: 8086モード

	outb( IO_PIC1_ICW1, 0x11 );			// ICW1: 初期化開始
	outb( IO_PIC1_ICW2, vector + 8 );	// ICW2: スレーブPICの割り込み
	outb( IO_PIC1_ICW3, 0x02 );			// ICW3: スレーブID
	outb( IO_PIC1_ICW4, 0x01 );			// ICW4: 8086モード

	outb( IO_PIC0_OCW1, 0xFF );			// OCW1: 全割り込み禁止
	outb( IO_PIC1_OCW1, 0xFF );			// OCW1: 全割り込み禁止

	outb( IO_PIC0_OCW3, 0x4A );			// OCW3: 特殊マスクモードリセット
	outb( IO_PIC1_OCW3, 0x4A );			// OCW3: 特殊マスクモードリセット
#endif	//!_VSUN86_PCSIM

	pic_processing = 0;

	// スレーブPICからの割り込みを受け付ける
	pic_enable_irq( IRQ_CASCADE );

	// 割り込みハンドラテーブルを初期化する
	irq_init();

	// 割り込みを有効にする
	cpu_enable_interrupt();

	vmm_printf( VMM_DEBUG, "pic0: imr=%02x, irr=%02x, isr=%02x\n",
				pic_get_imr( PIC_MASTER ), pic_get_irr( PIC_MASTER ), pic_get_isr( PIC_MASTER ) );
	vmm_printf( VMM_DEBUG, "pic1: imr=%02x, irr=%02x, isr=%02x\n",
				pic_get_imr( PIC_SLAVE  ), pic_get_irr( PIC_SLAVE  ), pic_get_isr( PIC_SLAVE  ) );

	return true;
}

void pic_enable_irq( u8 irq )
{
	lock( &pic_processing );
	{
		irq &= 0x0F;
#ifndef _VSUN86_PCSIM
		if ( irq < 8 )
			outb( IO_PIC0_OCW1, inb( IO_PIC0_IMR ) & ~(1<<irq) );
		else
			outb( IO_PIC1_OCW1, inb( IO_PIC1_IMR ) & ~(1<<(irq-8)) );
#else	//_VSUN86_PCSIM
		pic_imr[irq >> 3] &= ~(1 << (irq & 0x07));
#endif	//_VSUN86_PCSIM
	}
	unlock( &pic_processing );
}

void pic_disable_irq( u8 irq )
{
	lock( &pic_processing );
	{
		irq &= 0x0F;
#ifndef _VSUN86_PCSIM
		if ( irq < 8 )
			outb( IO_PIC0_OCW1, inb( IO_PIC0_IMR ) | (1<<irq) );
		else
			outb( IO_PIC1_OCW1, inb( IO_PIC1_IMR ) | (1<<(irq-8)) );
#else	//_VSUN86_PCSIM
		pic_imr[irq >> 3] |= 1 << (irq & 0x07);
#endif	//_VSUN86_PCSIM
	}
	unlock( &pic_processing );
}

void pic_send_eoi( u8 irq )
{
//	lock( &pic_processing );
	{
#ifndef _VSUN86_PCSIM
		irq &= 0x0F;
		if ( irq < 8 )
			outb( IO_PIC0_OCW2, 0x60 | irq );
		else {
			outb( IO_PIC1_OCW2, 0x60 | (irq - 8) );
			outb( IO_PIC1_OCW3, 0x0B );			// OCW3 (b1-0=11:ISR)
			const u8 isr = inb( IO_PIC1_ISR );	// ISR
			if ( isr == 0x00 )
			{	// スレーブPICの割り込み要求がなくなったことをマスタPICに通知する
				outb( IO_PIC0_OCW2, 0x62 );
			}
		}
#else	//_VSUN86_PCSIM
		irq &= 0x0F;
		if ( irq < 8 )
			pic_isr[0] &= ~(1 << irq);
		else
			pic_isr[1] &= ~(1 << (irq-8));
#endif	//_VSUN86_PCSIM
	}
//	unlock( &pic_processing );
}

u8 pic_get_imr( u8 index )
{
	u8 imr;

	lock( &pic_processing );
	{
#ifndef _VSUN86_PCSIM
		if ( index == PIC_MASTER )
			imr = inb( IO_PIC0_IMR );
		else
			imr = inb( IO_PIC1_IMR );
#else	//_VSUN86_PCSIM
		imr = pic_imr[(index == PIC_MASTER)?0:1];
#endif	//_VSUN86_PCSIM
	}
	unlock( &pic_processing );

	return imr;
}

u8 pic_get_irr( u8 index )
{
	u8 irr;

	lock( &pic_processing );
	{
#ifndef _VSUN86_PCSIM
		if ( index == PIC_MASTER ) {
			outb( IO_PIC0_OCW3, 0x0A );
			irr = inb( IO_PIC0_IRR );
		}
		else {
			outb( IO_PIC1_OCW3, 0x0A );
			irr = inb( IO_PIC1_IRR );
		}
#else	//_VSUN86_PCSIM
		irr = pic_irr[(index == PIC_MASTER)?0:1];
#endif	//_VSUN86_PCSIM
	}
	unlock( &pic_processing );

	return irr;
}

u8 pic_get_isr( u8 index )
{
	u8 isr;

	lock( &pic_processing );
	{
#ifndef _VSUN86_PCSIM
		if ( index == PIC_MASTER ) {
			outb( IO_PIC0_OCW3, 0x0B );
			isr = inb( IO_PIC0_ISR );
		}
		else {
			outb( IO_PIC1_OCW3, 0x0B );
			isr = inb( IO_PIC1_ISR );
		}
#else	//_VSUN86_PCSIM
		isr = pic_isr[(index == PIC_MASTER)?0:1];
#endif	//_VSUN86_PCSIM
	}
	unlock( &pic_processing );

	return isr;
}

#ifdef	_VSUN86_PCSIM
int pic_set_irq( u8 irq )
{
	int result = -1;

	lock( &pic_processing );
	{
		irq &= 0x0F;
		u8 irq_mask;
		u8 reg_index;
		if ( irq < 8 ) {
			irq_mask  = 1 << irq;
			reg_index = 0;
		}
		else {
			irq_mask  = 1 << (irq - 8);
			reg_index = 1;
		}

		if ( !(pic_imr[reg_index] & irq_mask) ) {
			if ( !(pic_isr[reg_index] & irq_mask) ) {
				pic_isr[reg_index] |= irq_mask;
				result = 0;
			}
		}
	}
	unlock( &pic_processing );

	return result;
}
#endif	//_VSUN86_PCSIM
