/*!
******************************************************************************

	@file	svm.h

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#ifndef __X86_SVM_H__
#define __X86_SVM_H__

#pragma pack(1)

//! @brief	VMCB (Control Area)
typedef struct {
	u16		intercept_cr_read;		//!< +000h			CR0-15の読み出しのインターセプト指定
	u16		intercept_cr_write;		//!< +002h			CR0-15の書き込みのインターセプト指定
	u16		intercept_dr_read;		//!< +004h			DR0-15の読み出しのインターセプト指定
	u16		intercept_dr_write;		//!< +006h			DR0-15の書き込みのインターセプト指定
	u32		intercept_excp;			//!< +008h			例外のインターセプト指定
	u64		intercept_mask;			//!< +00Ch			その他のインターセプト指定
	u8		rsvd_014h_03Fh[44];		//!< +014h-03Fh		(予約:0)
	u64		iopm_base_pa;			//!< +040h			IOPMの物理ベースアドレス
	u64		msrpm_base_pa;			//!< +048h			MSRPMの物理ベースアドレス
	u64		tsc_offset;				//!< +050h			RDTSC/RDTSCPに加算される値
	u32		asid;					//!< +058h:b31-0	ゲストのASID
	u8		tlb_control;			//!<      :b39-32	=1:VMRUN実行時にTLBをフラッシュ
	u8		rsvd_058h_b63_40[3];	//!<      :b63-40	(予約:0)
	u64		vint_flags;				//!< +060h			ゲストの仮想割り込み関係の設定
	u8		interrupt_shadow;		//!< +068h:b0		割り込みシャドウ
	u8		rsvd_068h_b63_1[7];		//!<     :b63-1		(予約:0)
	u64		exit_code;				//!< +070h			EXITCODE
	u64		exit_info1;				//!< +078h			EXITINFO1
	u64		exit_info2;				//!< +080h			EXITINFO2
	u64		exit_int_info;			//!< +088h			EXITINTINFO
	u8		np_enable;				//!< +090h:b0		nested pagingの有効
	u8		rsvd_090h_b63_1[7];		//!<      :b63-1	(予約:0)
	u8		rsvd_098h_0A7h[16];		//!< +098h-0A7h		(予約:0)
	u64		event_inj;				//!< +0A8h			event injection
	u64		n_cr3;					//!< +0B0h			N_CR3 (nested pagingで利用)
	u8		lbr_virt_enable;		//!< +0B8h:b0		=1:LBRの仮想化を有効
	u8		rsvd_0B8h_b63_1[7];		//!<      :b63-1	(予約:0)
	u8		rsvd_0C0h_3FFh[832];	//!< +0C0h-3FFh		(予約:0)
} SVM_VMCB_CTRL_AREA;

#define SVM_INTERCEPT_INTR			0x0000000000000001ULL
#define SVM_INTERCEPT_NMI			0x0000000000000002ULL
#define SVM_INTERCEPT_SMI			0x0000000000000004ULL
#define SVM_INTERCEPT_INIT			0x0000000000000008ULL
#define SVM_INTERCEPT_VINTR			0x0000000000000010ULL
#define SVM_INTERCEPT_CR0_WRITE		0x0000000000000020ULL
#define SVM_INTERCEPT_IDTR_READ		0x0000000000000040ULL
#define SVM_INTERCEPT_GDTR_READ		0x0000000000000080ULL
#define SVM_INTERCEPT_LDTR_READ		0x0000000000000100ULL
#define SVM_INTERCEPT_TR_READ		0x0000000000000200ULL
#define SVM_INTERCEPT_IDTR_WRITE	0x0000000000000400ULL
#define SVM_INTERCEPT_GDTR_WRITE	0x0000000000000800ULL
#define SVM_INTERCEPT_LDTR_WRITE	0x0000000000001000ULL
#define SVM_INTERCEPT_TR_WRITE		0x0000000000002000ULL
#define SVM_INTERCEPT_RDTSC			0x0000000000004000ULL
#define SVM_INTERCEPT_RDPMC			0x0000000000008000ULL
#define SVM_INTERCEPT_PUSHF			0x0000000000010000ULL
#define SVM_INTERCEPT_POPF			0x0000000000020000ULL
#define SVM_INTERCEPT_CPUID			0x0000000000040000ULL
#define SVM_INTERCEPT_RSM			0x0000000000080000ULL
#define SVM_INTERCEPT_IRET			0x0000000000100000ULL
#define SVM_INTERCEPT_INT			0x0000000000200000ULL
#define SVM_INTERCEPT_INVD			0x0000000000400000ULL
#define SVM_INTERCEPT_PAUSE			0x0000000000800000ULL
#define SVM_INTERCEPT_HLT			0x0000000001000000ULL
#define SVM_INTERCEPT_INVLPG		0x0000000002000000ULL
#define SVM_INTERCEPT_INVLPGA		0x0000000004000000ULL
#define SVM_INTERCEPT_IOIO_PROT		0x0000000008000000ULL
#define SVM_INTERCEPT_MSR_PROT		0x0000000010000000ULL
#define SVM_INTERCEPT_TASK_SWITCH	0x0000000020000000ULL
#define SVM_INTERCEPT_FERR_FREEZE	0x0000000040000000ULL
#define SVM_INTERCEPT_SHUTDOWN		0x0000000080000000ULL
#define SVM_INTERCEPT_VMRUN			0x0000000100000000ULL
#define SVM_INTERCEPT_VMMCALL		0x0000000200000000ULL
#define SVM_INTERCEPT_VMLOAD		0x0000000400000000ULL
#define SVM_INTERCEPT_VMSAVE		0x0000000800000000ULL
#define SVM_INTERCEPT_STGI			0x0000001000000000ULL
#define SVM_INTERCEPT_CLGI			0x0000002000000000ULL
#define SVM_INTERCEPT_SKINIT		0x0000004000000000ULL
#define SVM_INTERCEPT_RDTSCP		0x0000008000000000ULL
#define SVM_INTERCEPT_ICEBP			0x0000010000000000ULL
#define SVM_INTERCEPT_WBINVD		0x0000020000000000ULL
#define SVM_INTERCEPT_MONITOR		0x0000040000000000ULL
#define SVM_INTERCEPT_MWAIT_UNCOND	0x0000080000000000ULL
#define SVM_INTERCEPT_MWAIT_ARMED	0x0000100000000000ULL
#define SVM_INTERCEPT_RESERVED		0xFFFFE00000000000ULL

#define SVM_V_IRQ					0x0000000000000100ULL
#define SVM_V_IGN_TPR				0x0000000000100000ULL
#define SVM_V_INTR_MASKING			0x0000000001000000ULL

#define SVM_VMEXIT_READ_CR(n)		(0x000+(n))
#define SVM_VMEXIT_WRITE_CR(n)		(0x010+(n))
#define SVM_VMEXIT_READ_DR(n)		(0x020+(n))
#define SVM_VMEXIT_WRITE_DR(n)		(0x030+(n))
#define SVM_VMEXIT_EXCP(n)			(0x040+(n))
#define SVM_VMEXIT_INTR				0x060
#define SVM_VMEXIT_NMI				0x061
#define SVM_VMEXIT_SMI				0x062
#define SVM_VMEXIT_INIT				0x063
#define SVM_VMEXIT_VINTR			0x064
#define SVM_VMEXIT_CR0_SEL_WRITE	0x065
#define SVM_VMEXIT_IDTR_READ		0x066
#define SVM_VMEXIT_GDTR_READ		0x067
#define SVM_VMEXIT_LDTR_READ		0x068
#define SVM_VMEXIT_TR_READ			0x069
#define SVM_VMEXIT_IDTR_WRITE		0x06A
#define SVM_VMEXIT_GDTR_WRITE		0x06B
#define SVM_VMEXIT_LDTR_WRITE		0x06C
#define SVM_VMEXIT_TR_WRITE			0x06D
#define SVM_VMEXIT_RDTSC			0x06E
#define SVM_VMEXIT_RDPMC			0x06F
#define SVM_VMEXIT_PUSHF			0x070
#define SVM_VMEXIT_POPF				0x071
#define SVM_VMEXIT_CPUID			0x072
#define SVM_VMEXIT_RSM				0x073
#define SVM_VMEXIT_IRET				0x074
#define SVM_VMEXIT_SWINT			0x075
#define SVM_VMEXIT_INVD				0x076
#define SVM_VMEXIT_PAUSE			0x077
#define SVM_VMEXIT_HLT				0x078
#define SVM_VMEXIT_INVLPG			0x079
#define SVM_VMEXIT_INVLPGA			0x07A
#define SVM_VMEXIT_IOIO				0x07B
#define SVM_VMEXIT_MSR				0x07C
#define SVM_VMEXIT_TASK_SWITCH		0x07D
#define SVM_VMEXIT_FERR_FREEZE		0x07E
#define SVM_VMEXIT_SHUTDOWN			0x07F
#define SVM_VMEXIT_VMRUN			0x080
#define SVM_VMEXIT_VMMCALL			0x081
#define SVM_VMEXIT_VMLOAD			0x082
#define SVM_VMEXIT_VMSAVE			0x083
#define SVM_VMEXIT_STGI				0x084
#define SVM_VMEXIT_CLGI				0x085
#define SVM_VMEXIT_SKINIT			0x086
#define SVM_VMEXIT_RDTSCP			0x087
#define SVM_VMEXIT_ICEBP			0x088
#define SVM_VMEXIT_WBINVD			0x089
#define SVM_VMEXIT_MONITOR			0x08A
#define SVM_VMEXIT_MWAIT			0x08B
#define SVM_VMEXIT_MWAIT_COND		0x08C
#define SVM_VMEXIT_NPF				0x400
#define SVM_VMEXIT_INVALID			(-1)

typedef struct {
	u16		selector;
	u16		attrib;
	u32		limit;
	u64		base;
} SVM_VMCB_SEG;

#define SVM_VMCB_SEG_ATTRIB_NULL	0x0000
#define SVM_VMCB_SEG_ATTRIB_CS(d,l,opt)		\
		(((d) << 10) | ((l) << 9) | DESC_PRESENT | DESC_TYPE_CODE(opt))
#define SVM_VMCB_SEG_ATTRIB_DS(b,dpl,opt)	\
		(((b) << 10) | DESC_PRESENT | ((dpl) << 5) | DESC_TYPE_DATA(opt))
#define SVM_VMCB_SEG_ATTRIB_DEFAULT_LDTR	DESC_TYPE_LDT
#define SVM_VMCB_SEG_ATTRIB_DEFAULT_TR		DESC_TYPE_TSS16_B
#define SVM_VMCB_SEG_SIZE_IS_32BIT(seg)		((seg).attrib & 0x0400)

typedef union {
	u64		q;
	u32		d;
	u16		w;
	struct {
		u8	bl;
		u8	bh;
	};
} SVM_REG;

//! @brief	VMCB (State Save Area)
typedef struct {
	SVM_VMCB_SEG	es;						//!< +000h		ES
	SVM_VMCB_SEG	cs;						//!< +010h		CS
	SVM_VMCB_SEG	ss;						//!< +020h		SS
	SVM_VMCB_SEG	ds;						//!< +030h		DS
	SVM_VMCB_SEG	fs;						//!< +040h		FS
	SVM_VMCB_SEG	gs;						//!< +050h		GS
	SVM_VMCB_SEG	gdtr;					//!< +060h		GDTR
	SVM_VMCB_SEG	ldtr;					//!< +070h		LDTR
	SVM_VMCB_SEG	idtr;					//!< +080h		IDTR
	SVM_VMCB_SEG	tr;						//!< +090h		TR
	u8				rsvd_0A0h_0CAh[43];		//!< +0A0h-0CAh	(予約)
	u8				cpl;					//!< +0CBh		CPL
	u8				rsvd_0CCh_0CFh[4];		//!< +0CCh-0CFh	(予約)
	u64				efer;					//!< +0D0h		EFER
	u8				rsvd_0D8h_147h[112];	//!< +0D8h-147h	(予約)
	SVM_REG			cr4;					//!< +148h		CR4
	SVM_REG			cr3;					//!< +150h		CR3
	SVM_REG			cr0;					//!< +158h		CR0
	SVM_REG			dr7;					//!< +160h		DR7
	SVM_REG			dr6;					//!< +168h		DR6
	SVM_REG			rflags;					//!< +170h		RFLAGS
	SVM_REG			rip;					//!< +178h		RIP
	u8				rsvd_180h_1D7h[88];		//!< +180h-1D7h	(予約)
	SVM_REG			rsp;					//!< +1D8h		RSP
	u8				rsvd_1E0h_1F7h[24];		//!< +1E0h-1F7h	(予約)
	SVM_REG			rax;					//!< +1F8h		RAX
	u64				star;					//!< +200h		STAR
	u64				lstar;					//!< +208h		LSTAR
	u64				cstar;					//!< +210h		CSTAR
	u64				sfmask;					//!< +218h		SFMASK
	u64				kernel_gs_base;			//!< +220h		KernelGsBase
	u64				sysenter_cs;			//!< +228h		SYSENTER_CS
	u64				sysenter_esp;			//!< +230h		SYSENTER_ESP
	u64				sysenter_eip;			//!< +238h		SYSENTER_EIP
	SVM_REG			cr2;					//!< +240h		CR2
	u8				rsvd_248h_267h[32];		//!< +248h-267h	(予約)
	u64				g_pat;					//!< +268h		(nested paging有効) ゲストのPAT
	u64				dbg_ctl;				//!< +270h		(LBRの仮想化が有効) DBGCTL MSR
	u64				br_from;				//!< +278h		(LBRの仮想化が有効) LastBranchFromIP MSR
	u64				br_to;					//!< +280h		(LBRの仮想化が有効) LastBranchToIP MSR
	u64				last_excep_from;		//!< +288h		(LBRの仮想化が有効) LastExceptionFromIP MSR
	u64				last_excep_to;			//!< +290h		(LBRの仮想化が有効) LastExceptionToIP MSR
} SVM_VMCB_STAT_AREA;

//! @brief	VMCB
typedef struct {
	SVM_VMCB_CTRL_AREA	ctrl;
	SVM_VMCB_STAT_AREA	stat;
} SVM_VMCB;

#define SVM_EXIT_CODE(vmcb)			((vmcb)->ctrl.exit_code)
#define SVM_EXIT_INFO1(vmcb)		((vmcb)->ctrl.exit_info1)
#define SVM_EXIT_INFO2(vmcb)		((vmcb)->ctrl.exit_info2)
#define SVM_EXIT_INT_INFO(vmcb)		((vmcb)->ctrl.exit_int_info)

#define SVM_GUEST_INT_VECTOR(int_info)	((int_info) & 0x000000FF)
#define SVM_GUEST_INT_TYPE(int_info)	((int_info) & 0x00000700)
#define SVM_GUEST_INT_TYPE_INTR			0x00000000
#define SVM_GUEST_INT_TYPE_NMI			0x00000200
#define SVM_GUEST_INT_TYPE_EXCEPTION	0x00000300
#define SVM_GUEST_INT_TYPE_SWINT		0x00000400

// EVENTINJタイプ
#define EVENTINJ_TYPE_INTR		0
#define EVENTINJ_TYPE_NMI		2
#define EVENTINJ_TYPE_EXCP		3
#define EVENTINJ_TYPE_SWINT		4

#pragma pack()

#endif // __X86_SVM_H__
