/*!
******************************************************************************

	@file	pfemu.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "cpu.h"
#include "pfemu.h"
#include "pfemu/vcpu.h"
#include "task.h"
#include "cpuid.h"
#include "msr.h"
#include "printf.h"

static bool			pfemu_enable = false;
static VSUN86_VM	pfemu_vm;
static VCPU			pfemu_vcpu;

ALIGN(4096) u8		vm_module[VM_MODULE_SIZE];

#define VM_MEM_SIZE		0x04000000	// 64MB
#define VM_IOPM_SIZE	0x3000		// 12KB
#define VM_MSRPM_SIZE	0x2000		// 8KB
#define VM_HSAVE_SIZE	0x1000		// 4KB
#define VM_VMCB_SIZE	0x1000		// 4KB

ALIGN(4096) static u32	_vm_page_dir[1024];
ALIGN(4096) static u32	_vm_page_table[1024*1024];
ALIGN(4096) static u8	_vm_mem[VM_MEM_SIZE];
ALIGN(4096) static u8	_vm_iopm [VM_IOPM_SIZE];
ALIGN(4096) static u8	_vm_msrpm[VM_MSRPM_SIZE];
ALIGN(4096) static u8	_vm_hsave[VM_HSAVE_SIZE];
ALIGN(4096) static u8	_vm_vmcb [VM_VMCB_SIZE];

#define VM_PAGE_DIR		(_vm_page_dir)
#define VM_PAGE_TABLE	(_vm_page_table)
#define VM_MEM			(_vm_mem)
#define VM_IOPM			(_vm_iopm)
#define VM_MSRPM		(_vm_msrpm)
#define VM_HSAVE		(_vm_hsave)
#define VM_VMCB			(_vm_vmcb)

static bool pfemu_enable_svm( void );

bool pfemu_init( void )
{
	VSUN86_VM *vm = &pfemu_vm;

	if ( !pfemu_enable_svm() )
		return false;	// SVM機能が有効にできない

	memset( vm, 0, sizeof(pfemu_vm) );
	vm->vmid	 = 1;	// ※複数のVMを扱えるようになったらここでVMを識別する
	vm->vcpu	 = &pfemu_vcpu;
	vm->vmcb	 = VM_VMCB;
	vm->page_dir = VM_PAGE_DIR;
	vm->page_tbl = VM_PAGE_TABLE;
	vm->mem		 = VM_MEM;
	vm->iopm	 = VM_IOPM;
	vm->msrpm	 = VM_MSRPM;
	memset( VM_IOPM,  0xFF, VM_IOPM_SIZE  );
	memset( VM_MSRPM, 0xFF, VM_MSRPM_SIZE );

	for ( u32 i=0; i<1024; i++ )
		VM_PAGE_TABLE[i] = ((u32)VM_MEM + (i<<12)) | PTE_READWRITE | PTE_PRESENT;
	VM_PAGE_DIR[0] = ((u32)VM_PAGE_TABLE) | PDE_READWRITE | PDE_PRESENT;
	for ( u32 i=1; i<1024; i++ )
		VM_PAGE_DIR[i] = 0;

	u8 *mem = VM_MEM;
	mem[0xFFFF0] = 0xF4;	// HLT

	mem = VM_MODULE;
	mem[0x00000] = 0xC3;	// RET

	if ( !vcpu_init( vm ) )
		return false;

	pfemu_enable = true;
	return true;
}

static bool pfemu_enable_svm( void )
{
	u32 eax, ebx, ecx, edx;
	u64 vm_cr, efer;

	eax = cpuid( 0x80000000, &ebx, &ecx, &edx );
	if ((eax < 0x8000000A) ||
		(ebx != CPUID_AMD_Fn8000_0000_EBX) ||
		(ecx != CPUID_AMD_Fn8000_0000_ECX) ||
		(edx != CPUID_AMD_Fn8000_0000_EDX))
	{	// AMD製のCPUじゃない
		vmm_printf( VMM_DEBUG, "cpuid(0x80000000) != \"AuthenticAMD\"\n" );
		return false;
	}

	eax = cpuid( 0x80000001, &ebx, &ecx, &edx );
	if ( (ecx & CPUID_AMD_Fn8000_0001_ECX_SVM) == 0 )
	{	// SVM機能に対応していない
		vmm_printf( VMM_DEBUG, "SVM is not available.\n" );
		return false;
	}

	vm_cr = rdmsr( MSR_VM_CR );
	if ( (vm_cr & VM_CR_SVMDIS) != 0 )
	{	// SVM機能が無効になっている
		vmm_printf( VMM_DEBUG, "SVM is disabled.\n" );
		return false;
	}

	// SVMを有効にする
	efer = rdmsr( MSR_EFER );
	wrmsr( MSR_EFER, efer | EFER_SVME );
	wrmsr( MSR_VM_HSAVE_PA, VM_HSAVE );

	eax = cpuid( 0x8000000A, &ecx, &ecx, &edx );
	if ( edx & CPUID_AMD_Fn8000_000A_EDX_NP )
	{	// Nested Paging が有効
		vmm_printf( VMM_DEBUG, "Nested Paging supported.\n" );
	}
	else
	{	// Nested Paging が無効
		vmm_printf( VMM_WARNING, "Nested Paging not supported.\n" );
	}

	return true;
}

VSUN86_VM * pfemu_get_vm( void )
{
	if ( !pfemu_enable )
		return NULL;

	return &pfemu_vm;
}

void * pfemu_get_vcpu( void )
{
	if ( !pfemu_enable )
		return NULL;

	return &pfemu_vcpu;
}

bool pfemu_run_vcpu( void *result )
{
	if ( !pfemu_enable )
		return false;

	return vcpu_run( &pfemu_vm, result );
}

bool pfemu_set_vcpu_event( void *e )
{
	if ( !pfemu_enable )
		return false;

	return vcpu_set_event( &pfemu_vm, e );
}

bool pfemu_set_vcpu_mmio( u32 start_page, u32 pages, void *procs )
{
	if ( !pfemu_enable )
		return false;

	return vcpu_set_mmio( &pfemu_vm, start_page, pages, procs );
}

bool pfemu_enable_a20m( void )
{
	if ( !pfemu_enable )
		return false;

	return vcpu_enable_a20m( &pfemu_vm );
}

bool pfemu_disable_a20m( void )
{
	if ( !pfemu_enable )
		return false;

	return vcpu_disable_a20m( &pfemu_vm );
}
