﻿/*
Copyright (c) 2009, Lawrence S. Maccherone, Jr.

Copyright (c) 2008, Adobe Systems Incorporated
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

* Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

* Neither the name of Adobe Systems Incorporated nor the names of its
contributors may be used to endorse or promote products derived from
this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

// http://maccherone.com/larry/projects/a-pretty-json-encoder-for-actionscript-3-as3/
// JSONEncoderをString出力からFileStream書き出しに改造
package
{
	import flash.utils.describeType;
	import flash.filesystem.FileStream;

	public class JSONEncodeWriter {
		private var doIt:Function;
		private var writer:FileStream;
		private function puts(str:String):void { writer.writeUTFBytes(str); }

		/** The current level */
		private var level:int;

		/** Above this length, an object or array will be split into multiple lines. Any value 2 or below will always split. */
		//private var maxLength:int;

		/** When true, the encoder will add spaces and split Arrays and Objects over maxLength into multiple lines */
		private var pretty:Boolean;

		/**
		 *
		 */
		public function JSONEncodeWriter( writerStream:FileStream, value:*, pretty:Boolean=true) { // , maxLength:int=60 ) {
			level = 0;
			this.pretty = pretty;
			// this.maxLength = pretty ? maxLength : int.MAX_VALUE;
			writer = writerStream;
			doIt = function ():void { putsJSON( value ); };
		}

		/**
		 * エンコードの実行
		 */
		public function write():void { doIt(); }

		private function putsJSON( value:* ):void {

			// determine what value is and convert it based on it's type
			if ( value is String ) {
				// escape the string so it's formatted correctly
				puts( escapeString( value as String ) );

			} else if ( value is Number ) {
				// only encode numbers that are finite
				puts( isFinite( value as Number) ? value.toString() : "null" );

			} else if ( value is Boolean ) {
				// convert boolean to string easily
				puts( value ? "true" : "false" );

			} else if ( value is Array ) {
				putsArray( value as Array );

			} else if ( value is Object && value != null ) {
				putsObject( value );
			} else puts( "null" );
		}

		/**
		 * Escapes a string accoding to the JSON specification.
		 *
		 * @param str The string to be escaped
		 * @return The string with escaped special characters
		 * 		according to the JSON specification
		 */
		private function escapeString( str:String ):String {
			// create a string to store the string's jsonstring value
			var s:String = "";
			// current character in the string we're processing
			var ch:String;
			// store the length in a local variable to reduce lookups
			var len:Number = str.length;

			// loop over all of the characters in the string
			for ( var i:int = 0; i < len; i++ ) {

				// examine the character to determine if we have to escape it
				ch = str.charAt( i );
				switch ( ch ) {

					case '"':	// quotation mark
						s += "\\\"";
					break;

					//case '/':	// solidus
					//	s += "\\/";
					//	break;

					case '\\':	// reverse solidus
						s += "\\\\";
					break;

					case '\b':	// bell
						s += "\\b";
					break;

					case '\f':	// form feed
						s += "\\f";
					break;

					case '\n':	// newline
						s += "\\n";
					break;

					case '\r':	// carriage return
						s += "\\r";
					break;

					case '\t':	// horizontal tab
						s += "\\t";
					break;

					default:	// everything else

						// check for a control character and escape as unicode
						if ( ch < ' ' ) {
							// get the hex digit(s) of the character (either 1 or 2 digits)
							var hexCode:String = ch.charCodeAt( 0 ).toString( 16 );

							// ensure that there are 4 digits by adjusting
							// the # of zeros accordingly.
							var zeroPad:String = hexCode.length == 2 ? "00" : "000";

							// create the unicode escape sequence with 4 hex digits
							s += "\\u" + zeroPad + hexCode;
					} else {

						// no need to do any special encoding, just pass-through
						s += ch;

					}
				}	// end switch

			}	// end for loop

			return "\"" + s + "\"";
		}

		private function putsArray( a:Array ):void {
			level++;

			var sepFlag:Boolean = false;
			puts( "[\n" );

			// loop over the elements in the array and add their converted
			// values to the string
			for ( var i:int = 0; i < a.length; i++ ) {
				// when the length is 0 we're adding the first element so
				// no comma is necessary
				if ( sepFlag ) puts( ",\n" );

				// convert the value to a string
				puts( getPadding(level) );
				putsJSON( a[i] );
				sepFlag = true;
			}

			// close the array
			level--;
			puts( "\n" + getPadding(level) + "]" );
		}

		private function putsObject( o:Object ):void
		{
			level++;

			var sepFlag:Boolean = false;
			puts( "{\n" );

			// determine if o is a class instance or a plain object
			var classInfo:XML = describeType( o );
			if ( classInfo.@name.toString() == "Object" )
			{
				// the value of o[key] in the loop below - store this
				// as a variable so we don't have to keep looking up o[key]
				// when testing for valid values to convert
				var value:Object;

				// loop over the keys in the object and add their converted
				// values to the string
				for ( var key:String in o )
				{
					// assign value to a variable for quick lookup
					value = o[key];

					// don't add function's to the JSON string
					if ( value is Function )
					{
						// skip this key and try another
						continue;
					}

					// when the length is 0 we're adding the first item so
					// no comma is necessary
					if ( sepFlag ) puts( ",\n" );

					puts( getPadding(level) + escapeString( key ) + ( pretty ? ": " : ":" ) );
					putsJSON( value );
					sepFlag = true;
				}
			}
			else // o is a class instance
			{
				// Loop over all of the variables and accessors in the class and
				// serialize them along with their values.
				for each ( var v:XML in classInfo..*.( name() == "variable" || name() == "accessor" ) )
				{
					// When the length is 0 we're adding the first item so
					// no comma is necessary
					if ( sepFlag ) puts( ",\n" );

					puts( getPadding(level) + escapeString( v.@name.toString() ) + ( pretty ? ": " : ":" ) );
					putsJSON( o[ v.@name ] );
					sepFlag = true;
				}

			}

			level--;
			puts( "\n" + getPadding(level) + "}" );
		}

		private static function getPadding(level:int):String {
			var s:String = "";
			for (var i:int=0; i<level; i++) s += "\t";
			return s;
		}

	}

}

