/*
 * Copyright (C) 2009 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 */

package jp.co.versus.provider;

import jp.co.versus.provider.VersusContract.Member;
import jp.co.versus.provider.VersusContract.Point;
import jp.co.versus.provider.VersusContract.Result;
import jp.co.versus.provider.VersusContract.Team;
import jp.co.versus.provider.VersusContract.TeamMember;
import jp.co.versus.provider.VersusDatabaseHelper.Tables;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.content.UriMatcher;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteQueryBuilder;
import android.net.Uri;
import android.provider.BaseColumns;
import android.util.Log;

public class VersusProvider extends AbstractVersusProvider {

    private static final String TAG = "VersusProvider";

    private static final ProfileAwareUriMatcher sUriMatcher =
            new ProfileAwareUriMatcher(UriMatcher.NO_MATCH);

    private static final int MEMBER = 1000;
    private static final int TEAM = 1001;
    private static final int TEAMMEMBER = 1002;
    private static final int RESULT = 1003;
    private static final int POINT = 1004;

    /** Contains just BaseColumns._COUNT */
    private static final ProjectionMap sCountProjectionMap = ProjectionMap.builder()
            .add(BaseColumns._COUNT, "COUNT(*)")
            .build();

    static {
        // Contacts URI matching table
        final UriMatcher matcher = sUriMatcher;
        matcher.addURI(VersusContract.AUTHORITY, "member", MEMBER);
        matcher.addURI(VersusContract.AUTHORITY, "team", TEAM);
        matcher.addURI(VersusContract.AUTHORITY, "teammember", TEAMMEMBER);
        matcher.addURI(VersusContract.AUTHORITY, "result", RESULT);
        matcher.addURI(VersusContract.AUTHORITY, "point", POINT);
    }

    // The database tag to use for representing the contacts DB in versus transactions.
    /* package */ static final String VERSUS_DB_TAG = "versus";

    /**
     * The thread-local holder of the active transaction.  Shared between this and the profile
     * provider, to keep transactions on both databases synchronized.
     */
    private final ThreadLocal<VersusTransaction> mTransactionHolder =
            new ThreadLocal<VersusTransaction>();

    private final ThreadLocal<VersusDatabaseHelper> mDbHelper =
            new ThreadLocal<VersusDatabaseHelper>();
    private VersusDatabaseHelper mVersusHelper;

    @Override
    public boolean onCreate() {
        super.onCreate();
        return initialize();
    }

    private boolean initialize() {
        mVersusHelper = getDatabaseHelper(getContext());
        mDbHelper.set(mVersusHelper);

        // Set up the DB helper for keeping transactions serialized.
        setDbHelperToSerializeOn(mVersusHelper, VERSUS_DB_TAG);

        return true;
    }

    @Override
    protected VersusDatabaseHelper getDatabaseHelper(final Context context) {
        return VersusDatabaseHelper.getInstance(context);
    }

    @Override
    protected ThreadLocal<VersusTransaction> getTransactionHolder() {
        return mTransactionHolder;
    }

    @Override
    public Uri insert(Uri uri, ContentValues values) {
        return super.insert(uri, values);
    }

    @Override
    public int update(Uri uri, ContentValues values, String selection, String[] selectionArgs) {
        return super.update(uri, values, selection, selectionArgs);
    }

    @Override
    public int delete(Uri uri, String selection, String[] selectionArgs) {
        return super.delete(uri, selection, selectionArgs);
    }

    @Override
    protected boolean yield(VersusTransaction transaction) {
        // Now proceed with the Versus DB yield.
        SQLiteDatabase versusDb = transaction.getDbForTag(VERSUS_DB_TAG);
        return versusDb != null && versusDb.yieldIfContendedSafely(SLEEP_AFTER_YIELD_DELAY);
    }

    @Override
    public void onBegin() {
        // Not used.
    }

    @Override
    public void onCommit() {
        // Not used.
    }

    @Override
    public void onRollback() {
        // Not used.
    }

    @Override
    protected void notifyChange() {
        // Not used.
    }

    @Override
    protected Uri insertInTransaction(Uri uri, ContentValues values) {
        final int match = sUriMatcher.match(uri);
        long id = 0;

        switch (match) {
            case MEMBER: {
                id = mDbHelper.get().getWritableDatabase().insert(Tables.MEMBER, null, values);
                break;
            }
            case TEAM:
                id = mDbHelper.get().getWritableDatabase().insert(Tables.TEAM, null, values);
                break;
            case TEAMMEMBER:
                id = mDbHelper.get().getWritableDatabase().insert(Tables.TEAMMEMBER, null, values);
                break;
            case RESULT:
                id = mDbHelper.get().getWritableDatabase().insert(Tables.RESULT, null, values);
                break;
            case POINT:
                id = mDbHelper.get().getWritableDatabase().insert(Tables.POINT, null, values);
                break;
            default:
                return null;
        }

        if (id < 0) {
            return null;
        }

        return ContentUris.withAppendedId(uri, id);
    }

    @Override
    protected int updateInTransaction(Uri uri, ContentValues values, String selection,
            String[] selectionArgs) {
        int count = 0;

        final int match = sUriMatcher.match(uri);
        switch(match) {
            case MEMBER:
                return mDbHelper.get().getWritableDatabase().update(Tables.MEMBER, values,
                        selection, selectionArgs);
            case TEAM:
                return mDbHelper.get().getWritableDatabase().update(Tables.TEAM, values,
                        selection, selectionArgs);
            case TEAMMEMBER:
                return mDbHelper.get().getWritableDatabase().update(Tables.TEAMMEMBER, values,
                        selection, selectionArgs);
            case RESULT:
                return mDbHelper.get().getWritableDatabase().update(Tables.RESULT, values,
                        selection, selectionArgs);
            case POINT:
                return mDbHelper.get().getWritableDatabase().update(Tables.POINT, values,
                        selection, selectionArgs);
            default: {
                break;
            }
        }

        return count;
    }

    @Override
    protected int deleteInTransaction(Uri uri, String selection, String[] selectionArgs) {
        final int match = sUriMatcher.match(uri);
        switch (match) {
            case MEMBER:
                return mDbHelper.get().getWritableDatabase().delete(Tables.MEMBER, selection,
                        selectionArgs);
            case TEAM:
                return mDbHelper.get().getWritableDatabase().delete(Tables.TEAM, selection,
                        selectionArgs);
            case TEAMMEMBER:
                return mDbHelper.get().getWritableDatabase().delete(Tables.TEAMMEMBER, selection,
                        selectionArgs);
            case RESULT:
                return mDbHelper.get().getWritableDatabase().delete(Tables.RESULT, selection,
                        selectionArgs);
            case POINT:
                return mDbHelper.get().getWritableDatabase().delete(Tables.POINT, selection,
                        selectionArgs);
            default: {
                return 0;
            }
        }
    }

    @Override
    public Cursor query(Uri uri, String[] projection, String selection, String[] selectionArgs,
            String sortOrder) {
        return queryLocal(uri, projection, selection, selectionArgs, sortOrder, -1);
    }

    protected Cursor queryLocal(Uri uri, String[] projection, String selection,
            String[] selectionArgs, String sortOrder, long directoryId) {
        SQLiteQueryBuilder qb = new SQLiteQueryBuilder();
        String groupBy = null;
        String limit = getLimit(uri);

        final int match = sUriMatcher.match(uri);
        switch (match) {
            case MEMBER: {
                qb.setTables(Tables.MEMBER);
                break;
            }
            case TEAM:
                qb.setTables(Tables.TEAM);
                break;
            case TEAMMEMBER:
                qb.setTables(Tables.TEAMMEMBER);
                break;
            case RESULT:
                qb.setTables(Tables.RESULT);
                break;
            case POINT:
                qb.setTables(Tables.POINT);
                break;
            default:
                return null;
        }

        mVersusHelper = getDatabaseHelper(getContext());
        mDbHelper.set(mVersusHelper);

        Cursor cursor =
                query(mDbHelper.get().getReadableDatabase(), qb, projection, selection, selectionArgs, sortOrder, groupBy,
                        limit);
        return cursor;
    }

    private Cursor query(final SQLiteDatabase db, SQLiteQueryBuilder qb, String[] projection,
            String selection, String[] selectionArgs, String sortOrder, String groupBy,
            String limit) {
        if (projection != null && projection.length == 1
                && BaseColumns._COUNT.equals(projection[0])) {
            qb.setProjectionMap(sCountProjectionMap);
        }
        final Cursor c = qb.query(db, projection, selection, selectionArgs, groupBy, null,
                sortOrder, limit);
        if (c != null) {
            c.setNotificationUri(getContext().getContentResolver(), VersusContract.AUTHORITY_URI);
        }
        return c;
    }

    /**
     * Gets the value of the "limit" URI query parameter.
     *
     * @return A string containing a non-negative integer, or <code>null</code> if
     *         the parameter is not set, or is set to an invalid value.
     */
    private String getLimit(Uri uri) {
        String limitParam = getQueryParameter(uri, VersusContract.LIMIT_PARAM_KEY);
        if (limitParam == null) {
            return null;
        }
        // make sure that the limit is a non-negative integer
        try {
            int l = Integer.parseInt(limitParam);
            if (l < 0) {
                Log.w(TAG, "Invalid limit parameter: " + limitParam);
                return null;
            }
            return String.valueOf(l);
        } catch (NumberFormatException ex) {
            Log.w(TAG, "Invalid limit parameter: " + limitParam);
            return null;
        }
    }

    /**
     * A fast re-implementation of {@link Uri#getQueryParameter}
     */
    /* package */ static String getQueryParameter(Uri uri, String parameter) {
        String query = uri.getEncodedQuery();
        if (query == null) {
            return null;
        }

        int queryLength = query.length();
        int parameterLength = parameter.length();

        String value;
        int index = 0;
        while (true) {
            index = query.indexOf(parameter, index);
            if (index == -1) {
                return null;
            }

            // Should match against the whole parameter instead of its suffix.
            // e.g. The parameter "param" must not be found in "some_param=val".
            if (index > 0) {
                char prevChar = query.charAt(index - 1);
                if (prevChar != '?' && prevChar != '&') {
                    // With "some_param=val1&param=val2", we should find second "param" occurrence.
                    index += parameterLength;
                    continue;
                }
            }

            index += parameterLength;

            if (queryLength == index) {
                return null;
            }

            if (query.charAt(index) == '=') {
                index++;
                break;
            }
        }

        int ampIndex = query.indexOf('&', index);
        if (ampIndex == -1) {
            value = query.substring(index);
        } else {
            value = query.substring(index, ampIndex);
        }

        return Uri.decode(value);
    }

    @Override
    public String getType(Uri uri) {

        final int match = sUriMatcher.match(uri);
        switch (match) {
            case MEMBER:
                return Member.CONTENT_TYPE;
            case TEAM:
                return Team.CONTENT_TYPE;
            case TEAMMEMBER:
                return TeamMember.CONTENT_TYPE;
            case RESULT:
                return Result.CONTENT_TYPE;
            case POINT:
                return Point.CONTENT_TYPE;
            default:
                return null;
        }
    }
}
