(****************************************************************************\
** Copyright 2019 Levashev Ivan Aleksandrovich                              **
**                                                                          **
** Licensed under the Apache License, Version 2.0 (the "License");          **
** you may not use this file except in compliance with the License.         **
** You may obtain a copy of the License at                                  **
**                                                                          **
**     http://www.apache.org/licenses/LICENSE-2.0                           **
**                                                                          **
** Unless required by applicable law or agreed to in writing, software      **
** distributed under the License is distributed on an "AS IS" BASIS,        **
** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. **
** See the License for the specific language governing permissions and      **
** limitations under the License.                                           **
\****************************************************************************)

unit vdsoleaut.Promises;

interface

uses
  Classes;

type
  IAsyncCallPromise = interface(IInterface)
    // No GUID
    function GetIndex: Integer;
    function GetIsCompleted: Boolean;
    function GetValue: OleVariant; // call from STA only !
    procedure Detach;
    function GetIsDetached: Boolean;
    procedure PostValue(Value: OleVariant); // call from MTA only !

    property Index: Integer read GetIndex;
    property IsCompleted: Boolean read GetIsCompleted;
    property Value: OleVariant read GetValue; // get from STA only !
    property IsDetached: Boolean read GetIsDetached;
  end;

function AllocatePromise: IAsyncCallPromise;
function AllocatedPromise(Index: Integer): IAsyncCallPromise;
procedure DeallocatePromise(Index: Integer);
procedure DeallocatePromises;

implementation

uses
  Windows, vdsoleaut.Impl, SysUtils, ActiveX, ComObj, Variants;

var
  AllocatedPromises: array of IAsyncCallPromise;

type
  TMarshalOleVariantMode = (movmNone, movmIUnknown, movmIDispatch);
  TAsyncCallPromiseImplementation = class(TInterfacedObject, IAsyncCallPromise)
  protected
    FCriticalSection: Windows.TRTLCriticalSection;
    FIndex: Integer;
    FIsCompleted: Boolean;
    FValue: OleVariant;
    FValueStream: ActiveX.IStream;
    FMarshaledMode: TMarshalOleVariantMode;
    function GetIndex: Integer;
    function GetIsCompleted: Boolean;
    function GetValue: OleVariant;
    function GetIsDetached: Boolean;
    procedure Marshal;
    procedure Unmarshal;
  public
    constructor Create(AIndex: Integer);
    destructor Destroy; override;
    procedure Detach;
    procedure PostValue(Value: OleVariant);

    property Index: Integer read GetIndex;
    property IsCompleted: Boolean read GetIsCompleted;
    property Value: OleVariant read GetValue;
    property IsDetached: Boolean read GetIsDetached;
  end;

function TAsyncCallPromiseImplementation.GetIndex: Integer;
begin
  EnterCriticalSection(FCriticalSection);
  try
    Result := FIndex;
  finally
    LeaveCriticalSection(FCriticalSection);
  end;
end;

function TAsyncCallPromiseImplementation.GetIsCompleted: Boolean;
begin
  EnterCriticalSection(FCriticalSection);
  try
    Result := FIsCompleted;
  finally
    LeaveCriticalSection(FCriticalSection);
  end;
end;

function TAsyncCallPromiseImplementation.GetValue: OleVariant;
begin
  EnterCriticalSection(FCriticalSection);
  try
    Unmarshal;
    Result := FValue;
  finally
    LeaveCriticalSection(FCriticalSection);
  end;
end;

function TAsyncCallPromiseImplementation.GetIsDetached: Boolean;
begin
  EnterCriticalSection(FCriticalSection);
  try
    Result := FIndex < 0;
  finally
    LeaveCriticalSection(FCriticalSection);
  end;
end;

constructor TAsyncCallPromiseImplementation.Create(AIndex: Integer);
begin
  inherited Create;
  InitializeCriticalSection(FCriticalSection);
  FIndex := AIndex;
end;

destructor TAsyncCallPromiseImplementation.Destroy;
begin
  Unmarshal;
  DeleteCriticalSection(FCriticalSection);

  inherited Destroy;
end;

procedure TAsyncCallPromiseImplementation.Detach;
begin
  EnterCriticalSection(FCriticalSection);
  try
    FIndex := -1;
  finally
    LeaveCriticalSection(FCriticalSection);
  end;
end;

procedure TAsyncCallPromiseImplementation.PostValue(Value: OleVariant);
var
  WHandle: THandle;
begin
  EnterCriticalSection(FCriticalSection);
  try
    if FIsCompleted then
    begin
      Exit;
    end;

    FIsCompleted := True;
    FValue := Value;
    Marshal;

    if (FIndex >= 0) and Assigned(vdsoleaut.Impl.WHandle) then
    begin
      WHandle := vdsoleaut.Impl.WHandle^;
      if (WHandle <> Windows.INVALID_HANDLE_VALUE) and (WHandle <> 0) then
      begin
        SysUtils.Win32Check(Windows.PostMessage(vdsoleaut.Impl.WHandle^,
          vdsoleaut.Impl.WM_VDSOLEASYNC, FIndex, 0));
      end;
    end;
  finally
    LeaveCriticalSection(FCriticalSection);
  end;
end;

procedure TAsyncCallPromiseImplementation.Marshal;
begin
  case TVarData(FValue).VType of
    varDispatch:
    begin
      FMarshaledMode := movmIDispatch;
      ComObj.OleCheck(ActiveX.CoMarshalInterThreadInterfaceInStream(IDispatch, IDispatch(FValue), FValueStream));
      FValue := Variants.Unassigned;
    end;
    varUnknown:
    begin
      FMarshaledMode := movmIUnknown;
      ComObj.OleCheck(ActiveX.CoMarshalInterThreadInterfaceInStream(IUnknown, IUnknown(FValue), FValueStream));
      FValue := Variants.Unassigned;
    end;
  else
    // do nothing
  end;
end;

procedure TAsyncCallPromiseImplementation.Unmarshal;
var
  TempD: IDispatch;
  TempU: IUnknown;
begin
  case FMarshaledMode of
    movmIDispatch:
    begin
      FMarshaledMode := movmNone;
      ComObj.OleCheck(ActiveX.CoGetInterfaceAndReleaseStream(FValueStream, IDispatch, TempD));
      FValueStream := nil;
      FValue := TempD;
    end;
    movmIUnknown:
    begin
      FMarshaledMode := movmNone;
      ComObj.OleCheck(ActiveX.CoGetInterfaceAndReleaseStream(FValueStream, IUnknown, TempU));
      FValueStream := nil;
      FValue := TempU;
    end;
    movmNone: // do nothing
  end;
end;

function AllocatePromise: IAsyncCallPromise;
var
  L: Integer;
begin
  L := Length(AllocatedPromises);
  SetLength(AllocatedPromises, L + 1);
  Result := TAsyncCallPromiseImplementation.Create(L + 1);
  AllocatedPromises[L] := Result;
end;

function AllocatedPromise(Index: Integer): IAsyncCallPromise;
begin
  if (Index < 1) or (Index > Length(AllocatedPromises)) then
  begin
    Result := nil;
    Exit;
  end;

  Result := AllocatedPromises[Index - 1];
end;

procedure DeallocatePromise(Index: Integer);
var
  L, I: Integer;
begin
  if (Index < 1) or (Index > Length(AllocatedPromises)) then
  begin
    Exit;
  end;

  if Assigned(AllocatedPromises[Index - 1]) then
  begin
    AllocatedPromises[Index - 1].Detach;
    AllocatedPromises[Index - 1] := nil;
  end;
  L := Length(AllocatedPromises);
  for I := L - 1 downto 0 do
  begin
    if Assigned(AllocatedPromises[I]) then
    begin
      if I <> L - 1 then
      begin
        SetLength(AllocatedPromises, I + 1);
      end;

      Exit;
    end;
  end;

  AllocatedPromises := nil;
end;

procedure DeallocatePromises;
var
  I: Integer;
begin
  for I := 0 to Length(AllocatedPromises) - 1 do
  begin
    if Assigned(AllocatedPromises[I]) then
    begin
      AllocatedPromises[I].Detach;
      AllocatedPromises[I] := nil;
    end;
  end;

  AllocatedPromises := nil;
end;

initialization
finalization
  DeallocatePromises;
end.
