' table/FieldProvider/FieldProvider.vb
'
' Copyright (c) 2008-2011, SystemBase Co.,Ltd.
' All rights reserved.
'
' Redistribution and use in source and binary forms, with or without
' modification, are permitted provided that the following conditions are met:
'
'    1. Redistributions of source code must retain the above copyright
'       notice, this list of conditions and the following disclaimer.
'    2. Redistributions in binary form must reproduce the above copyright
'       notice, this list of conditions and the following disclaimer in the
'       documentation and/or other materials provided with the distribution.
'
' THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
' IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
' ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
' LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
' CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
' SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
' INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
' CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
' ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
' POSSIBILITY OF SUCH DAMAGE.

Imports System.Drawing

Public Interface IFieldProvider
    Sub FieldInitialize(ByVal field As UTable.CField)
    Sub EditorInitialize(ByVal field As UTable.CField, ByVal editor As IEditor)
    Sub Render(ByVal g As Graphics, ByVal field As UTable.CField, ByVal rect As Rectangle, ByVal alter As Boolean)
    Sub SetBorder(ByVal field As UTable.CField, ByVal border As UTable.CBorder, ByVal merged As Boolean)
    Sub ToggleValue(ByVal field As UTable.CField, ByVal value As Object)
    Function Setting() As UTable.CSetting
    Function CreateField() As UTable.CField
    Function CreateEditor() As IEditor
    Function Focusable() As Boolean
    Function UndoEnabled() As Boolean
    Function ImeMode() As ImeMode
    Function GetAdjustSize(ByVal g As Graphics, ByVal field As UTable.CField) As Size
    Function ValueRegularize(ByVal value As Object, ByVal editor As IEditor) As Object
    Function CreateModifyField(ByVal field As UTable.CField) As UTable.CFieldModify
    Function Draggable() As Boolean
    Property Caption() As String
    Property TabOrder() As Integer
    Property Clipboard(ByVal field As UTable.CField) As String
End Interface

Public Class CFieldProvider
    Implements IFieldProvider

    Private _Caption As String
    Private _TabOrder As Integer = 0

    Public BorderLine As Integer = UTable.CBorder._T Or _
                                   UTable.CBorder._B Or _
                                   UTable.CBorder._L Or _
                                   UTable.CBorder._R

    Public Sub New()
        Me.New(Nothing)
    End Sub

    Public Sub New(ByVal caption As String)
        Me.Caption = caption
    End Sub

    Public Property Caption() As String Implements IFieldProvider.Caption
        Get
            Return Me._Caption
        End Get
        Set(ByVal value As String)
            Me._Caption = value
        End Set
    End Property

    Public Property TabOrder() As Integer Implements IFieldProvider.TabOrder
        Get
            Return Me._TabOrder
        End Get
        Set(ByVal value As Integer)
            Me._TabOrder = value
        End Set
    End Property

    Public Overridable Function CreateField() As UTable.CField Implements IFieldProvider.CreateField
        Return New UTable.CField()
    End Function

    Public Overridable Sub FieldInitialize(ByVal field As UTable.CField) Implements IFieldProvider.FieldInitialize
    End Sub

    Public Overridable Function CreateEditor() As IEditor Implements IFieldProvider.CreateEditor
        Return Nothing
    End Function

    Public Overridable Sub EditorInitialize(ByVal field As UTable.CField, ByVal editor As IEditor) Implements IFieldProvider.EditorInitialize
    End Sub

    Public Overridable Function Setting() As UTable.CSetting Implements IFieldProvider.Setting
        Return Nothing
    End Function

    Public Overridable Function Focusable() As Boolean Implements IFieldProvider.Focusable
        Return True
    End Function

    Public Overridable Function UndoEnabled() As Boolean Implements IFieldProvider.UndoEnabled
        Return True
    End Function

    Public Overridable Function ImeMode() As ImeMode Implements IFieldProvider.ImeMode
        Return Windows.Forms.ImeMode.Disable
    End Function

    Public Overridable Function Draggable() As Boolean Implements IFieldProvider.Draggable
        Return False
    End Function

    Public Overridable Function ValueRegularize(ByVal value As Object, ByVal editor As IEditor) As Object Implements IFieldProvider.ValueRegularize
        Return value
    End Function

    Public Overridable Sub Render(ByVal g As Graphics, ByVal field As UTable.CField, ByVal rect As Rectangle, ByVal alter As Boolean) Implements IFieldProvider.Render
        Dim s As UTable.CDynamicSetting = field.DynamicSetting
        Me.renderBackground(g, field, s, rect, alter)
        If field.Decorator IsNot Nothing Then
            field.Decorator.RenderBackground(g, field, rect)
        End If
        If field.Table.Focused AndAlso field.Focused AndAlso field.Table.FocusFieldDecorator IsNot Nothing Then
            field.Table.FocusFieldDecorator.RenderBackground(g, field, rect)
        End If
        Me.renderForeground(g, field, s, rect, alter)
        If field.Decorator IsNot Nothing Then
            field.Decorator.RenderForeground(g, field, rect)
        End If
        If field.Table.Focused AndAlso field.Focused AndAlso field.Table.FocusFieldDecorator IsNot Nothing Then
            field.Table.FocusFieldDecorator.RenderForeground(g, field, rect)
        End If
    End Sub

    Protected Overridable Sub renderBackground(ByVal g As Graphics, _
                                               ByVal field As UTable.CField, _
                                               ByVal s As UTable.CDynamicSetting, _
                                               ByVal rect As Rectangle, _
                                               ByVal alter As Boolean)
        RenderBackgroudRect(g, rect, BackColor(field, s, alter))
    End Sub

    Protected Overridable Sub renderForeground(ByVal g As Graphics, _
                                               ByVal field As UTable.CField, _
                                               ByVal s As UTable.CDynamicSetting, _
                                               ByVal rect As Rectangle, _
                                               ByVal alter As Boolean)
        If field.Editor Is Nothing Then
            RenderValue(g, CreateValueRect(rect, s, field.TopLevelContent.RenderingRect), Me.formatValue(field.Value), _
                        ForeColor(field, s, alter), s.Font, Me.GetBoundedStringFormat(g, field, s, rect))
        End If
    End Sub

    Public Overridable Function GetAdjustSize(ByVal g As Graphics, ByVal field As UTable.CField) As Size Implements IFieldProvider.GetAdjustSize
        With field.DynamicSetting
            Return g.MeasureString(Me.formatValue(field.Value), .Font, 100000, .GetStringFormat).ToSize
        End With
    End Function

    Protected Overridable Function formatValue(ByVal v As Object) As String
        If v IsNot Nothing Then
            Return v.ToString
        Else
            Return Nothing
        End If
    End Function

    Public Function GetBoundedStringFormat( _
      ByVal g As Graphics, _
      ByVal f As UTable.CField, _
      ByVal s As UTable.CDynamicSetting, _
      ByVal rect As Rectangle) As StringFormat
        Dim ret As StringFormat = s.GetStringFormat
        Dim ajs As Size
        If s.LeftBound = UTable.EBound.BOUND Or s.TopBound = UTable.EBound.BOUND Then
            ajs = g.MeasureString(Me.formatValue(f.Value), s.Font, rect.Width, s.GetStringFormat).ToSize
        End If
        If s.LeftBound = UTable.EBound.BOUND Then
            If ajs.Width > rect.Width Then
                ret.Alignment = StringAlignment.Near
            End If
        End If
        If s.TopBound = UTable.EBound.BOUND Then
            If ajs.Height > rect.Height Then
                ret.LineAlignment = StringAlignment.Near
            End If
        End If
        Return ret
    End Function

    Public Shared Function BackColor(ByVal field As UTable.CField, ByVal setting As UTable.CDynamicSetting, ByVal alter As Boolean) As Color
        If field Is field.Table.FocusField AndAlso _
           Not setting.FocusBackColor.Equals(Color.Transparent) AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
            field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusBackColor
        ElseIf field.Table.SelectRange IsNot Nothing AndAlso _
               field.Table.SelectRange.IsInclude(field) AndAlso _
               Not setting.RangedBackColor.Equals(Color.Transparent) AndAlso _
               (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
                field.Table.Setting.FocusColorAlways) Then
            Return setting.RangedBackColor
        ElseIf field.Record Is field.Table.FocusRecord AndAlso _
               Not setting.FocusRecordBackColor.Equals(Color.Transparent) AndAlso _
               (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
                field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusRecordBackColor
        ElseIf alter AndAlso Not setting.AlterBackColor.Equals(Color.Transparent) Then
            Return setting.AlterBackColor
        Else
            Return setting.BackColor
        End If
    End Function

    Public Shared Function ForeColor(ByVal field As UTable.CField, ByVal setting As UTable.CDynamicSetting, ByVal alter As Boolean) As Color
        If field Is field.Table.FocusField AndAlso _
           Not setting.FocusForeColor.Equals(Color.Transparent) AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
            field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusForeColor
        ElseIf field.Table.SelectRange IsNot Nothing AndAlso _
           field.Table.SelectRange.IsInclude(field) AndAlso _
           Not setting.RangedForeColor.Equals(Color.Transparent) AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
            field.Table.Setting.FocusColorAlways) Then
            Return setting.RangedForeColor
        ElseIf field.Record Is field.Table.FocusRecord AndAlso _
               Not setting.FocusRecordForeColor.Equals(Color.Transparent) AndAlso _
               (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
                field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusRecordForeColor
        Else
            Return setting.ForeColor
        End If
    End Function

    Public Shared Function CaptionBackColor(ByVal field As UTable.CField, ByVal setting As UTable.CDynamicSetting, ByVal drag As Boolean) As Color
        If drag AndAlso _
           Not setting.DraggingBackColor.Equals(Color.Transparent) Then
            Return setting.DraggingBackColor
        ElseIf field.Table.FocusField IsNot Nothing AndAlso _
           field.Table.FocusField.Desc.Layout IsNot Nothing AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or field.Table.Setting.FocusColorAlways) AndAlso _
           field.Content.TopLevelContent Is field.Table.HeaderContent AndAlso _
           Not setting.FocusCaptionBackColor.Equals(Color.Transparent) AndAlso _
           field.Content.Level.Equals(field.Table.FocusField.Content.Level) AndAlso _
           field Is field.Record.FindField(field.Table.FocusField.Desc.Layout.Point) Then
            Return setting.FocusCaptionBackColor
        ElseIf Not setting.FocusRecordCaptionBackColor.Equals(Color.Transparent) AndAlso _
           field.Record Is field.Table.FocusRecord AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
           field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusRecordCaptionBackColor
        ElseIf Not setting.IncludeFocusRecordCaptionBackColor.Equals(Color.Transparent) AndAlso _
           field.Table.FocusField IsNot Nothing AndAlso _
           _GetIncludeFocusRecord(field.Table.FocusRecord).IsChild(field) AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
           field.Table.Setting.FocusColorAlways) Then
            Return setting.IncludeFocusRecordCaptionBackColor
        Else
            Return setting.CaptionBackColor
        End If
    End Function

    Public Shared Function CaptionForeColor(ByVal field As UTable.CField, ByVal setting As UTable.CDynamicSetting, ByVal drag As Boolean) As Color
        If drag AndAlso _
           Not setting.DraggingForeColor.Equals(Color.Transparent) Then
            Return setting.DraggingForeColor
        ElseIf field.Table.FocusField IsNot Nothing AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or field.Table.Setting.FocusColorAlways) AndAlso _
           field.Content.TopLevelContent Is field.Table.HeaderContent AndAlso _
           Not setting.FocusCaptionForeColor.Equals(Color.Transparent) AndAlso _
           field.Content.Level.Equals(field.Table.FocusField.Content.Level) AndAlso _
           field Is field.Record.FindField(field.Table.FocusField.Desc.Layout.Point) Then
            Return setting.FocusCaptionForeColor
        ElseIf Not setting.FocusRecordCaptionForeColor.Equals(Color.Transparent) AndAlso _
           field.Record Is field.Table.FocusRecord AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
           field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusRecordCaptionForeColor
        ElseIf Not setting.IncludeFocusRecordCaptionForeColor.Equals(Color.Transparent) AndAlso _
           field.Table.FocusRecord IsNot Nothing AndAlso _
           _GetIncludeFocusRecord(field.Table.FocusRecord).IsChild(field) AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
           field.Table.Setting.FocusColorAlways) Then
            Return setting.IncludeFocusRecordCaptionForeColor
        Else
            Return setting.CaptionForeColor
        End If
    End Function

    Private Shared Function _GetIncludeFocusRecord(ByVal record As UTable.CRecord) As UTable.CRecord
        Dim ret As UTable.CRecord = record
        Do
            If ret.Content.LimitIncludeFocusColor Then
                Return ret
            End If
            If ret.Content.ParentRecord Is Nothing Then
                Return ret
            End If
            ret = ret.Content.ParentRecord
        Loop
    End Function

    Public Shared Function CreateValueRect(ByVal rect As Rectangle, ByVal s As UTable.CDynamicSetting, ByVal rrect As Rectangle) As RectangleF
        Dim l As Single = rect.Left
        Dim r As Single = rect.Right
        Dim t As Single = rect.Top
        Dim b As Single = rect.Bottom
        If s.StayHorizontal = UTable.EStay.STAY Then
            l = Math.Max(l, rrect.Left)
            r = Math.Min(r, rrect.Right)
        End If
        If s.StayVertical = UTable.EStay.STAY Then
            t = Math.Max(t, rrect.Top)
            b = Math.Min(b, rrect.Bottom)
        End If
        Return New RectangleF(l, t + 2, r - l, b - t - 2)
    End Function

    Public Shared Sub RenderBackgroudRect(ByVal g As Graphics, _
                                          ByVal rect As Rectangle, _
                                          ByVal color As Color)
        Using brush As New SolidBrush(color)
            g.FillRectangle(brush, rect)
        End Using
    End Sub

    Public Shared Sub RenderValue(ByVal g As Graphics, _
                                  ByVal rect As RectangleF, _
                                  ByVal formattedValue As String, _
                                  ByVal color As Color, _
                                  ByVal font As Font, _
                                  ByVal stringFormat As StringFormat)
        If Not String.IsNullOrEmpty(formattedValue) Then
            Using brush As New SolidBrush(color)
                g.DrawString(ValueSanitize(formattedValue), font, brush, rect, stringFormat)
            End Using
        End If
    End Sub

    Public Shared Sub RenderCaptionBackgroudRect(ByVal g As Graphics, _
                                                 ByVal rect As Rectangle, _
                                                 ByVal color As Color, _
                                                 ByVal style As UTable.ECaptionStyle)
        Select Case style
            Case UTable.ECaptionStyle.METALLIC
                Using b As New Drawing2D.LinearGradientBrush(rect, color, GradientColor(color, 32), Drawing2D.LinearGradientMode.Vertical)
                    g.FillRectangle(b, rect)
                End Using
                Using p As New Pen(GradientColor(color, 64))
                    g.DrawLine(p, New Point(rect.X + 1, rect.Y + 1), New Point(rect.Right - 1, rect.Y + 1))
                    g.DrawLine(p, New Point(rect.X + 1, rect.Y + 1), New Point(rect.X + 1, rect.Bottom - 1))
                End Using
            Case UTable.ECaptionStyle.GRADIENT
                Using b As New Drawing2D.LinearGradientBrush(rect, color, GradientColor(color, 32), Drawing2D.LinearGradientMode.Vertical)
                    g.FillRectangle(b, rect)
                End Using
            Case UTable.ECaptionStyle.FLAT
                Using b As New SolidBrush(color)
                    g.FillRectangle(b, rect)
                End Using
        End Select
    End Sub

    Public Shared Sub RenderCaptionValue(ByVal g As Graphics, _
                                         ByVal field As UTable.CField, _
                                         ByVal sortState As UTable.CSortState, _
                                         ByVal rect As RectangleF, _
                                         ByVal formattedValue As String, _
                                         ByVal color As Color, _
                                         ByVal font As Font, _
                                         ByVal stringFormat As StringFormat)
        Using brush As New SolidBrush(color)
            If rect.Width > 15 And field.Table.SortState.Field Is field Then
                g.DrawString(ValueSanitize(formattedValue), font, brush, _
                             New RectangleF(rect.X, rect.Y + 2, rect.Width - 15, rect.Height - 2), stringFormat)
                Dim points As New List(Of PointF)
                Select Case field.Table.SortState.Order
                    Case UTable.CSortState.EOrder.ASCEND
                        points.Add(New PointF(rect.Right - 14, rect.Bottom - 7))
                        points.Add(New PointF(rect.Right - 2, rect.Bottom - 7))
                        points.Add(New PointF(rect.Right - 8, rect.Bottom - 14))
                    Case UTable.CSortState.EOrder.DESCEND
                        points.Add(New PointF(rect.Right - 14, rect.Bottom - 12))
                        points.Add(New PointF(rect.Right - 2, rect.Bottom - 12))
                        points.Add(New PointF(rect.Right - 8, rect.Bottom - 6))
                End Select
                g.FillPolygon(brush, points.ToArray)
            Else
                g.DrawString(ValueSanitize(formattedValue), font, brush, rect, stringFormat)
            End If
        End Using
    End Sub

    Private Shared _checkedImage As Bitmap = Nothing
    Private Shared _uncheckedImage As Bitmap = Nothing
    Public Shared Sub RenderCheckBox(ByVal g As Graphics, ByVal rect As Rectangle, ByVal value As Boolean)
        Dim p As New Point(rect.Left + rect.Width / 2 - 6, rect.Top + rect.Height / 2 - 6)
        g.SetClip(rect)
        If _checkedImage Is Nothing Then
            _checkedImage = New Bitmap(13, 13)
            _uncheckedImage = New Bitmap(13, 13)
            CheckBoxRenderer.DrawCheckBox( _
              Graphics.FromImage(_checkedImage), _
              New Point(0, 0), _
              VisualStyles.CheckBoxState.CheckedNormal)
            CheckBoxRenderer.DrawCheckBox( _
              Graphics.FromImage(_uncheckedImage), _
              New Point(0, 0), _
              VisualStyles.CheckBoxState.UncheckedNormal)
        End If
        If value Then
            g.DrawImage(_checkedImage, p)
        Else
            g.DrawImage(_uncheckedImage, p)
        End If
        g.ResetClip()
    End Sub

    Public Shared Sub RenderNarrowChild(ByVal g As Graphics, ByVal rect As Rectangle, ByVal value As Boolean, ByVal enabled As Boolean)
        If enabled Then
            g.FillRectangle(Brushes.White, rect)
            g.DrawRectangle(Pens.Gray, rect)
            g.DrawLine(Pens.Black, New Point(rect.Left + 3, rect.Top + rect.Height / 2), New Point(rect.Right - 3, rect.Top + rect.Height / 2))
            If Not value Then
                g.DrawLine(Pens.Black, New Point(rect.Left + rect.Width / 2, rect.Top + 3), New Point(rect.Left + rect.Width / 2, rect.Top + rect.Height - 3))
            End If
        Else
            g.FillRectangle(Brushes.LightGray, rect)
            g.DrawRectangle(Pens.Gray, rect)
        End If
    End Sub

    Protected Overridable Function getToggleRect(ByVal field As UTable.CField, ByVal rect As Rectangle, ByVal w As Integer) As Rectangle
        Dim size As Integer = TOGGLERECT_SIZE + w
        Return New Rectangle(rect.Left + rect.Width / 2 - size / 2, rect.Top + rect.Height / 2 - size / 2, size, size)
    End Function

    Protected Shared Sub renderButton( _
      ByVal g As Graphics, _
      ByVal down As Boolean, _
      ByVal formattedValue As String, _
      ByVal image As Image, _
      ByVal disabledImage As Image, _
      ByVal s As UTable.CDynamicSetting, _
      ByVal rect As System.Drawing.Rectangle)
        If rect.Width <= 4 Then
            Exit Sub
        End If

        Dim font As Font = s.Font
        Dim format As StringFormat
        Dim tx As Integer = rect.X + 2
        Dim ix As Integer = 0
        Dim iy As Integer = 0
        Dim iw As Integer = 0
        Dim ih As Integer = 0

        If image Is Nothing Then
            format = s.GetStringFormat
        Else
            format = s.GetButtonStringFormat
            Dim w As Integer = rect.Width - 4
            Dim h As Integer = rect.Height - 4
            Dim tw As Integer = g.MeasureString(formattedValue, font, 1000000, format).Width
            If image IsNot Nothing Then
                iw = image.Width
                ih = image.Height
                If iw > w Or ih > h Then
                    Dim rw As Single = w / iw
                    Dim rh As Single = h / ih
                    If rw < rh Then
                        iw = iw * rw
                        ih = ih * rw
                    Else
                        iw = iw * rh
                        ih = ih * rh
                    End If
                End If
            End If
            Select Case s.HorizontalAlignment
                Case UTable.EHAlign.LEFT
                    ix = rect.X + 2
                    tx = rect.X + 2 + iw
                Case UTable.EHAlign.MIDDLE
                    ix = rect.X + Math.Max((rect.Width - iw - tw) / 2, 2)
                    tx = rect.X + Math.Max((rect.Width - iw - tw) / 2 + iw, iw + 2)
                Case UTable.EHAlign.RIGHT
                    ix = rect.X + Math.Max(rect.Width - tw - iw - 2, 2)
                    tx = rect.X + Math.Max(rect.Width - tw - 2, iw + 2)
            End Select
            Select Case s.VerticalAlignment
                Case UTable.EVAlign.TOP
                    iy = rect.Y + 2
                Case UTable.EVAlign.MIDDLE
                    iy = rect.Y + Math.Max((rect.Height - ih) / 2, 2)
                Case UTable.EVAlign.BOTTOM
                    iy = rect.Y + Math.Max(rect.Height - ih - 2, 2)
            End Select

        End If
        If s.Editable = UTable.EAllow.ALLOW Then
            Dim d As Integer = 0
            If down Then
                _renderButton_aux(g, rect, s, True)
                d = 2
            Else
                _renderButton_aux(g, rect, s, False)
            End If
            If image IsNot Nothing AndAlso iw > 0 And ih > 0 Then
                g.DrawImage(image, _
                            New Rectangle(ix, iy + d, iw, ih), _
                            New Rectangle(0, 0, image.Width, image.Height), GraphicsUnit.Pixel)
            End If
            If Not String.IsNullOrEmpty(formattedValue) Then
                Dim t As String = ValueSanitize(formattedValue)
                Using brush As New SolidBrush(s.ForeColor)
                    Dim ty As Integer = rect.Y + 2 + d
                    g.DrawString(t, font, brush, _
                      New RectangleF(tx, ty, rect.Right + d - tx, rect.Bottom + d - ty), format)
                End Using
            End If
        Else
            _renderButton_aux(g, rect, s, False)
            If disabledImage IsNot Nothing AndAlso iw > 0 And ih > 0 Then
                g.DrawImage(disabledImage, _
                            New Rectangle(ix, iy, iw, ih), _
                            New Rectangle(0, 0, image.Width, image.Height), GraphicsUnit.Pixel)
            End If
            If Not String.IsNullOrEmpty(formattedValue) Then
                Dim t As String = ValueSanitize(formattedValue)
                Dim ty As Integer = rect.Y + 2
                ControlPaint.DrawStringDisabled(g, t, s.Font, Color.White, _
                  New RectangleF(tx, ty, rect.Right - tx, rect.Bottom - ty), format)
            End If
        End If
    End Sub

    Private Shared Sub _renderButton_aux(ByVal g As Graphics, ByVal rect As Rectangle, ByVal s As UTable.CDynamicSetting, ByVal down As Boolean)
        Using b As New Drawing2D.LinearGradientBrush(New RectangleF(rect.X, rect.Y, rect.Width, rect.Height), s.ButtonBackColor, GradientColor(s.ButtonBackColor, 32), Drawing2D.LinearGradientMode.Vertical)
            g.FillRectangle(b, New Rectangle(rect.X + 2, rect.Y + 2, rect.Width - 3, rect.Height - 3))
        End Using
        Using p As New Pen(GradientColor(s.ButtonBackColor, -128))
            g.DrawRectangle(p, New Rectangle(rect.X + 2, rect.Y + 2, rect.Width - 4, rect.Height - 4))
        End Using
        If down Then
            Using p As New Pen(GradientColor(s.ButtonBackColor, 64))
                g.DrawLine(p, rect.Right - 3, rect.Y + 3, rect.Right - 3, rect.Bottom - 3)
                g.DrawLine(p, rect.X + 3, rect.Bottom - 3, rect.Right - 3, rect.Bottom - 3)
            End Using
            Using p As New Pen(GradientColor(s.ButtonBackColor, -64))
                g.DrawLine(p, rect.X + 3, rect.Y + 3, rect.Right - 3, rect.Y + 3)
                g.DrawLine(p, rect.X + 3, rect.Y + 3, rect.X + 3, rect.Bottom - 3)
            End Using
        Else
            Using p As New Pen(GradientColor(s.ButtonBackColor, -64))
                g.DrawLine(p, rect.Right - 3, rect.Y + 3, rect.Right - 3, rect.Bottom - 3)
                g.DrawLine(p, rect.X + 3, rect.Bottom - 3, rect.Right - 3, rect.Bottom - 3)
            End Using
            Using p As New Pen(GradientColor(s.ButtonBackColor, 64))
                g.DrawLine(p, rect.X + 3, rect.Y + 3, rect.Right - 3, rect.Y + 3)
                g.DrawLine(p, rect.X + 3, rect.Y + 3, rect.X + 3, rect.Bottom - 3)
            End Using
        End If
    End Sub

    Public Shared Function ValueSanitize(ByVal v As Object) As String
        If v Is Nothing Then
            Return ""
        Else
            Return System.Text.RegularExpressions.Regex.Replace(v.ToString, "]", "-")
        End If
    End Function

    Public Shared Function GradientColor(ByVal c As Color, ByVal d As Integer) As Color
        Dim r As Integer = c.R + d
        Dim g As Integer = c.G + d
        Dim b As Integer = c.B + d
        If r > 255 Then r = 255
        If r < 0 Then r = 0
        If g > 255 Then g = 255
        If g < 0 Then g = 0
        If b > 255 Then b = 255
        If b < 0 Then b = 0
        Return Color.FromArgb(r, g, b)
    End Function

    Public Overridable Sub SetBorder(ByVal field As UTable.CField, ByVal border As UTable.CBorder, ByVal merged As Boolean) Implements IFieldProvider.SetBorder
        SetFieldBorder(field, border, merged, Me.BorderLine)
    End Sub

    Public Shared Sub SetFieldBorder(ByVal field As UTable.CField, ByVal border As UTable.CBorder, ByVal merged As Boolean, ByVal borderLine As Integer)
        If field.Desc.Layout IsNot Nothing Then
            Dim b As Integer = borderLine
            If merged Then
                b = b And (UTable.CBorder._T Or UTable.CBorder._L Or UTable.CBorder._R)
            End If
            border.Fill(field.Desc.Layout, UTable.CBorder.EBorderType.FIELD, b)
        End If
    End Sub

    Public Shared Sub SetCaptionBorder(ByVal field As UTable.CField, ByVal border As UTable.CBorder, ByVal merged As Boolean, ByVal borderLine As Integer)
        If field.Desc.Layout IsNot Nothing Then
            Dim b As Integer = borderLine
            If merged Then
                b = b And (UTable.CBorder._T Or UTable.CBorder._L Or UTable.CBorder._R)
            End If
            border.Fill(field.Desc.Layout, UTable.CBorder.EBorderType.CAPTION, b)
        End If
    End Sub

    Public Shared Sub FlatBorder(ByVal descs As Dictionary(Of Object, UTable.CFieldDesc), ByVal key1 As Object, ByVal key2 As Object)
        Dim l As UTable.CGrid.CRegion = descs(key1).Layout.Union(descs(key2).Layout)
        For Each d As UTable.CFieldDesc In descs.Values
            If TypeOf d.Provider Is CFieldProvider AndAlso d.Layout IsNot Nothing Then
                If d.Layout.Row < (l.Row + l.Rows) And (d.Layout.Row + d.Layout.Rows) > l.Row And _
                   d.Layout.Col < (l.Col + l.Cols) And (d.Layout.Col + d.Layout.Cols) > l.Col Then
                    With CType(d.Provider, CFieldProvider)
                        .BorderLine = 0
                        If d.Layout.Row = l.Row Then
                            .BorderLine = .BorderLine Or UTable.CBorder._T
                        End If
                        If d.Layout.Row + d.Layout.Rows = l.Row + l.Rows Then
                            .BorderLine = .BorderLine Or UTable.CBorder._B
                        End If
                        If d.Layout.Col = l.Col Then
                            .BorderLine = .BorderLine Or UTable.CBorder._L
                        End If
                        If d.Layout.Col + d.Layout.Cols = l.Col + l.Cols Then
                            .BorderLine = .BorderLine Or UTable.CBorder._R
                        End If
                    End With
                End If
            End If
        Next
    End Sub

    Public Shared TOGGLERECT_SIZE As Integer = 10
    Public Toggle As Boolean = False

    Public Function SetToggle(ByVal field As UTable.CField) As UTable.CField
        AddHandler field.MouseDown, AddressOf CFieldProvider._toggle_mousedown
        AddHandler field.KeyPress, AddressOf CFieldProvider._toggle_keyPress
        Me.Toggle = True
        Return field
    End Function

    Private Shared Sub _toggle_mousedown(ByVal field As UTable.CField, ByVal location As Point, ByVal e As System.Windows.Forms.MouseEventArgs)
        If TypeOf field.Desc.Provider Is CFieldProvider Then
            Dim fp As CFieldProvider = field.Desc.Provider
            If e.Button = MouseButtons.Left AndAlso field.Table.FocusField Is field Then
                Dim f As UTable.CRenderCache.CField = field.Table.RenderCache.FindField(field)
                If f IsNot Nothing AndAlso f.Field.Editable = UTable.EAllow.ALLOW Then
                    If TOGGLERECT_SIZE = 0 OrElse _
                       fp.getToggleRect(f.Field, f.Rect, 2).Contains(e.Location) Then
                        field.Desc.Provider.ToggleValue(field, Not CType(field.Value, Boolean))
                    End If
                End If
            End If
        End If
    End Sub

    Private Shared Sub _toggle_keyPress(ByVal field As UTable.CField, ByVal e As System.Windows.Forms.KeyPressEventArgs)
        If TypeOf field.Desc.Provider Is CFieldProvider Then
            Using field.Table.RenderBlock
                If e.KeyChar.Equals(" "c) AndAlso field.Editable = UTable.EAllow.ALLOW Then
                    field.Desc.Provider.ToggleValue(field, Not CType(field.Value, Boolean))
                    If field.Table.SelectRange IsNot Nothing AndAlso field.Table.Setting.RangeToggleValue Then
                        For Each l As List(Of UTable.CField) In field.Table.SelectRange.FieldMatrix
                            For Each f As UTable.CField In l
                                If f IsNot Nothing AndAlso _
                                   TypeOf f.Desc.Provider Is CFieldProvider AndAlso _
                                   CType(f.Desc.Provider, CFieldProvider).Toggle AndAlso _
                                   f IsNot field AndAlso _
                                   f.Editable = UTable.EAllow.ALLOW Then
                                    f.Desc.Provider.ToggleValue(f, field.Value)
                                End If
                            Next
                        Next
                    End If
                End If
            End Using
        End If
    End Sub

    Public Overridable Sub ToggleValue(ByVal field As UTable.CField, ByVal value As Object) Implements IFieldProvider.ToggleValue
        field.SetValueIfValidated(value)
    End Sub

    Public Overridable Property Clipboard(ByVal field As UTable.CField) As String Implements IFieldProvider.Clipboard
        Get
            If field.Value IsNot Nothing Then
                Return field.Value.ToString
            Else
                Return ""
            End If
        End Get
        Set(ByVal value As String)
            field.SetValueIfValidated(value)
        End Set
    End Property

    Public Overridable Function CreateFieldModify(ByVal field As UTable.CField) As UTable.CFieldModify Implements IFieldProvider.CreateModifyField
        Return New UTable.CFieldModify(field)
    End Function

End Class
