package org.unitedfront2.dao;

import java.util.List;
import java.util.Locale;

import org.junit.Assert;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.unitedfront2.domain.accesscontrol.AccessControl;
import org.unitedfront2.domain.accesscontrol.FriendOnly;
import org.unitedfront2.domain.accesscontrol.OwnerOnly;
import org.unitedfront2.domain.accesscontrol.UserOnly;
import org.unitedfront2.domain.communication.Message;
import org.unitedfront2.domain.communication.MessageCodeUsedByOtherException;
import org.unitedfront2.domain.communication.Thread;
import org.unitedfront2.test.DaoTestCaseSupport;

public class ThreadDaoTest extends DaoTestCaseSupport<Thread> {

    @Autowired
    private ThreadDao threadDao;

    @Override
    protected Thread createTestDomain() {
        Thread t = new Thread();
        AccessControl rac = domainFactory.prototype(UserOnly.class);
        AccessControl wac = domainFactory.prototype(OwnerOnly.class);
        AccessControl pac = domainFactory.prototype(UserOnly.class);
        t.setOverview(createOverview(rac, wac));
        t.setOwnerId(simpleUser1.getId());
        t.setReadAccessControl(rac);
        t.setWriteAccessControl(wac);
        t.setPostAccessControl(pac);
        return t;
    }

    private Message createOverview(AccessControl readAccessControl, AccessControl writeAccessControl) {
        Message o = domainFactory.prototype(Message.class);
        o.setOwnerId(simpleUser1.getId());
        o.setAuthorId(simpleUser1.getId());
        o.setSubject("eXgXbh", Locale.JAPANESE);
        o.setBody("̓eXgp̃XbhłB", Locale.JAPANESE);
        o.setReadAccessControl(readAccessControl);
        o.setWriteAccessControl(writeAccessControl);
        return o;
    }

    @Override
    protected void editTestDomain(Thread thread) {
        thread.getOverview().setSubject("new_" + thread.getOverview().getSubject(Locale.JAPANESE), Locale.JAPANESE);
        thread.getOverview().setBody("new_" + thread.getOverview().getSubject(Locale.JAPANESE), Locale.JAPANESE);
        thread.getReadAccessControl().changeTo(FriendOnly.class);
    }

    @Override
    protected Object getDao() {
        return threadDao;
    }

    @Override
    protected void afterTestRegister(Thread thread) {
        Assert.assertSame(thread.getOwnerId(), thread.getOverview().getOwnerId());
        Assert.assertSame(thread.getOwnerId(), thread.getOverview().getAuthorId());
        Assert.assertEquals(thread.getReadAccessControl(), thread.getOverview().getReadAccessControl());
        Assert.assertEquals(thread.getWriteAccessControl(), thread.getOverview().getWriteAccessControl());
    }

    @Override
    protected void afterTestUpdate(Thread thread) {
        Assert.assertSame(thread.getOwnerId(), thread.getOverview().getOwnerId());
        Assert.assertSame(thread.getOwnerId(), thread.getOverview().getAuthorId());
        Assert.assertEquals(thread.getReadAccessControl(), thread.getOverview().getReadAccessControl());
        Assert.assertEquals(thread.getWriteAccessControl(), thread.getOverview().getWriteAccessControl());
    }

    @Test
    public void testRegisterEntry() throws MessageCodeUsedByOtherException {
        Thread t = createTestDomain();
        threadDao.register(t);
        Message e = createEntry(t.getReadAccessControl(), t.getWriteAccessControl());
        e.store();
        threadDao.registerEntry(t.getId(), e.getId());
        List<Message> entries = threadDao.findEntries(t.getId());
        Assert.assertSame(1, entries.size());
        Assert.assertEquals(e, entries.get(0));
    }

    private Message createEntry(AccessControl readAccessControl, AccessControl writeAccessControl) {
        Message e = domainFactory.prototype(Message.class);
        e.setOwnerId(simpleUser1.getId());
        e.setAuthorId(simpleUser1.getId());
        e.setSubject("eXgL", Locale.JAPANESE);
        e.setBody("̓eXgp̋LłB", Locale.JAPANESE);
        e.setReadAccessControl(readAccessControl);
        e.setWriteAccessControl(writeAccessControl);
        return e;
    }

    @Test
    public void testFindEntry() throws MessageCodeUsedByOtherException {
        Thread t = createTestDomain();
        threadDao.register(t);
        Message e = createEntry(t.getReadAccessControl(), t.getWriteAccessControl());
        e.store();
        threadDao.registerEntry(t.getId(), e.getId());
        Message found = threadDao.findEntry(t.getId(), e.getId());
        Assert.assertEquals(e, found);
    }

    @Test
    public void testFindEntryقȂXbhɏĂꍇ̓qbgȂ() throws MessageCodeUsedByOtherException {
        Thread t1 = createTestDomain();
        threadDao.register(t1);
        Message e = createEntry(t1.getReadAccessControl(), t1.getWriteAccessControl());
        e.store();
        threadDao.registerEntry(t1.getId(), e.getId());
        Thread t2 = createTestDomain();
        threadDao.register(t2);
        Message found = threadDao.findEntry(t2.getId(), e.getId());
        Assert.assertNull(found);
    }

    @Test
    public void testFindEntries() throws MessageCodeUsedByOtherException {
        Thread t1 = createTestDomain();
        threadDao.register(t1);
        Thread t2 = createTestDomain();
        threadDao.register(t2);
        Message e1 = createEntry(t1.getReadAccessControl(), t1.getWriteAccessControl());
        e1.store();
        threadDao.registerEntry(t1.getId(), e1.getId());
        Message e2 = createEntry(t1.getReadAccessControl(), t1.getWriteAccessControl());
        e2.store();
        threadDao.registerEntry(t1.getId(), e2.getId());
        Message e3 = createEntry(t1.getReadAccessControl(), t1.getWriteAccessControl());
        e3.store();
        threadDao.registerEntry(t2.getId(), e3.getId());
        List<Message> entries = threadDao.findEntries(t1.getId());
        Assert.assertSame(2, entries.size());
        Assert.assertEquals(e2, entries.get(0));
        Assert.assertEquals(e1, entries.get(1));
    }

    @Test
    public void testFindEntries3args() throws MessageCodeUsedByOtherException {
        Thread t = createTestDomain();
        threadDao.register(t);
        Message e1 = createEntry(t.getReadAccessControl(), t.getWriteAccessControl());
        e1.store();
        threadDao.registerEntry(t.getId(), e1.getId());
        Message e2 = createEntry(t.getReadAccessControl(), t.getWriteAccessControl());
        e2.store();
        threadDao.registerEntry(t.getId(), e2.getId());
        Message e3 = createEntry(t.getReadAccessControl(), t.getWriteAccessControl());
        e3.store();
        threadDao.registerEntry(t.getId(), e3.getId());
        List<Message> entries = threadDao.findEntries(t.getId(), 1, 2);
        Assert.assertSame(2, entries.size());
        Assert.assertEquals(e2, entries.get(0));
        Assert.assertEquals(e1, entries.get(1));
    }

    @Test
    public void testCountEntry() throws MessageCodeUsedByOtherException {
        Thread t = createTestDomain();
        threadDao.register(t);
        Message e1 = createEntry(t.getReadAccessControl(), t.getWriteAccessControl());
        e1.store();
        threadDao.registerEntry(t.getId(), e1.getId());
        Message e2 = createEntry(t.getReadAccessControl(), t.getWriteAccessControl());
        e2.store();
        threadDao.registerEntry(t.getId(), e2.getId());
        Assert.assertSame(2, threadDao.countEntry(t.getId()));
    }
}
